/*-
 * #%L
 * custom-connector-sdk
 * %%
 * Copyright (C) 2021 Amazon Web Services
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package com.amazonaws.appflow.custom.connector.marketplace;

import com.amazonaws.SdkBaseException;
import com.amazonaws.services.licensemanager.AWSLicenseManager;
import com.amazonaws.services.licensemanager.AWSLicenseManagerClientBuilder;
import com.amazonaws.services.licensemanager.model.CheckoutLicenseRequest;
import com.amazonaws.services.licensemanager.model.CheckoutLicenseResult;
import com.amazonaws.services.licensemanager.model.CheckoutType;
import com.amazonaws.services.licensemanager.model.EntitlementData;
import com.amazonaws.services.licensemanager.model.EntitlementDataUnit;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Utility class provides Marketplace Entitlement check for connectors.
 */
public final class EntitlementUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(EntitlementUtil.class);

    private static final String KEY_FINGERPRINT = "aws:294406891311:AWS/Marketplace:issuer-fingerprint";
    private static final String AWS_MARKETPLACE_USAGE_ENTITLEMENT_NAME = "AWS::Marketplace::Usage";

    private static AWSLicenseManager client;

    public EntitlementUtil() {
        client = AWSLicenseManagerClientBuilder.defaultClient();
    }

    /**
     * Checks if the Connector subscribed from Marketplace has entitlement to use or not for an AWS account.
     *
     * @param productSKU - ProductSKU generated by marketplace at the time of connector registration.
     * @return - True if the entitlement is successful otherwise false.
     */
    public boolean checkEntitlement(final String productSKU) {

        final CheckoutLicenseRequest request = new CheckoutLicenseRequest().withCheckoutType(CheckoutType.PROVISIONAL)
                .withProductSKU(productSKU)
                .withKeyFingerprint(KEY_FINGERPRINT)
                .withEntitlements(Collections.singleton(getMarketplaceUsageEntitlement()))
                .withClientToken(UUID.randomUUID().toString());

        try {
            final CheckoutLicenseResult result = client.checkoutLicense(request);
            final List<String> entitlements = result.getEntitlementsAllowed().stream().map(EntitlementData::getName)
                    .collect(Collectors.toList());
            if (entitlements.contains(AWS_MARKETPLACE_USAGE_ENTITLEMENT_NAME)) {
                return true;
            }
        } catch (SdkBaseException ex) {
            LOGGER.error("Entitlement check failed with exception" + ExceptionUtils.getStackTrace(ex));
        }
        return false;
    }

    private static EntitlementData getMarketplaceUsageEntitlement() {
        return new EntitlementData().withName(AWS_MARKETPLACE_USAGE_ENTITLEMENT_NAME).withUnit(EntitlementDataUnit.None);
    }
}
