/*
 *  Copyright (c) xlightweb.org, 2008 - 2010. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Please refer to the LGPL license at: http://www.gnu.org/copyleft/lesser.txt
 * The latest copy of this software may be found on http://www.xlightweb.org/
 */
package org.xlightweb.server;

import java.io.IOException;


import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import javax.net.ssl.SSLContext;



import org.xlightweb.BadMessageException;
import org.xlightweb.IHttpExchange;
import org.xlightweb.IHttpRequest;
import org.xlightweb.IHttpRequestHandler;
import org.xlightweb.IWebHandler;
import org.xlightweb.IWebSocketHandler;
import org.xlightweb.WebSocketConnection;

 


/**
 * A HttpServer. The http server accepts incoming connections and forwards the request to
 * the assigned {@link IHttpRequestHandler}. Example:
 * 
 * <pre>
 * 
 *  // defining a request handler 
 *  class MyRequestHandler implements IHttpRequestHandler  {
 *  
 *     public void onRequest(IHttpExchange exchange) throws IOException {
 *        IHttpRequest request = exchange.getRequest();
 *        //...
 *        
 *        exchange.send(new HttpResponse(200, "text/plain", "it works"));
 *     }
 *  } 
 *
 * 
 *  // creates a server
 *  IServer server = new XHttpServer(8080, new MyRequestHandler());
 * 
 *  // setting some properties 
 *  server.setMaxTransactions(400);
 *  server.setRequestTimeoutMillis(5 * 60 * 1000);  
 *  //...
 * 
 * 
 *  // executing the server 
 *  server.run();  // (blocks forever)
 * 
 *  // or server.start();  (blocks until the server has been started)
 *  //...
 * </pre> 
 *
 * 
 * @author grro@xlightweb.org
 */
public class XHttpServer extends HttpServer {

    
	/**
	 * constructor <br><br>
	 *
	 * @param webHandler the webHandler
	 * @throws IOException If some other I/O error occurs
	 * @throws UnknownHostException if the local host cannot determined
	 */
	public XHttpServer(IWebHandler webHandler) throws UnknownHostException, IOException {
	    this(webHandler, new HashMap<String, Object>());
	}




	/**
	 * constructor <br><br>
	 *
     * @param webHandler           the webHandler
	 * @param options              the socket options  
	 * @throws IOException If some other I/O error occurs
	 * @throws UnknownHostException if the local host cannot determined
	 */
	public XHttpServer(IWebHandler webHandler, Map<String, Object> options) throws UnknownHostException, IOException {
	    this(new InetSocketAddress(0), options, webHandler, null, false, MIN_SIZE_WORKER_POOL, SIZE_WORKER_POOL);
	}


	/**
	 * constructor  <br><br>
	 *
	 * @param port        the local port
     * @param webHandler  the webHandler
	 * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(int port, IWebHandler webHandler) throws UnknownHostException, IOException {
		this(port, webHandler, MIN_SIZE_WORKER_POOL, SIZE_WORKER_POOL);
	}

	
    /**
     * constructor  <br><br>
     *
     * @param port         the local port
     * @param webHandler   the webHandler
     * @param minPoolsize  the min workerpool size
     * @param maxPoolsize  the max workerpool size
     * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
     */
    public XHttpServer(int port, IWebHandler webHandler, int minPoolsize, int maxPoolsize) throws UnknownHostException, IOException {
        this(port, webHandler, null, false, minPoolsize, maxPoolsize);
    }	



	/**
	 * constructor <br><br>
	 *
	 *
	 * @param port                 the local port
     * @param webHandler           the webHandler
	 * @param options              the acceptor socket options 
	 * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(int port, IWebHandler webHandler, Map<String , Object> options) throws UnknownHostException, IOException {
		this(port, webHandler, options, null, false);
	}


	/**
	 * constructor <br><br>
	 *
	 *
	 * @param address     the local address
	 * @param port        the local port
     * @param webHandler  the webHandler
	 * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(InetAddress address, int port, IWebHandler webHandler) throws UnknownHostException, IOException {
		this(address, port, webHandler, null, false);
	}



	/**
	 * constructor  <br><br>
	 *
	 *
	 * @param ipAddress   the local ip address
	 * @param port        the local port
     * @param webHandler  the webHandler 
	 * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(String ipAddress, int port, IWebHandler webHandler) throws UnknownHostException, IOException {
		this(InetAddress.getByName(ipAddress), port, webHandler);
	}



	/**
	 * constructor <br><br>
	 *
	 *
	 * @param ipAddress            the local ip address
	 * @param port                 the local port
     * @param webHandler           the webHandler 
	 * @param options              the socket options 
	 * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(String ipAddress, int port, IWebHandler webHandler, Map<String, Object> options) throws UnknownHostException, IOException {
		this(InetAddress.getByName(ipAddress), port, webHandler, options, null, false);
	}




	/**
	 * constructor <br><br>
	 *
	 *
	 * @param port               local port
     * @param webHandler         the webHandler
	 * @param sslOn              true, is SSL should be activated
	 * @param sslContext         the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(int port, IWebHandler webHandler, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(port, webHandler, new HashMap<String, Object>(), sslContext, sslOn);
	}


	
	   /**
     * constructor <br><br>
     *
     *
     * @param port               local port
     * @param webHandler         the webHandler
     * @param sslOn              true, is SSL should be activated
     * @param sslContext         the ssl context to use
     * @param minPoolsize  the min workerpool size
     * @param maxPoolsize  the max workerpool size 
     * @throws UnknownHostException if the local host cannot determined
     * @throws IOException If some other I/O error occurs
     */
    public XHttpServer(int port, IWebHandler webHandler, SSLContext sslContext, boolean sslOn, int minPoolsize, int maxPoolsize) throws UnknownHostException, IOException {
        this(new InetSocketAddress(port), new HashMap<String, Object>(), webHandler, sslContext, sslOn, minPoolsize, maxPoolsize);
    }

	



	/**
	 * constructor <br><br>
	 *
	 * @param port                 local port
	 * @param options              the acceptor socket options
     * @param webHandler           the webHandler 
	 * @param sslOn                true, is SSL should be activated
	 * @param sslContext           the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(int port, IWebHandler webHandler, Map<String, Object> options, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(new InetSocketAddress(port), options, webHandler, sslContext, sslOn, MIN_SIZE_WORKER_POOL, SIZE_WORKER_POOL);
	}
 

	/**
	 * constructor <br><br>
	 *
	 * @param ipAddress          local ip address
	 * @param port               local port
     * @param webHandler         the webHandler 
	 * @param sslOn              true, is SSL should be activated
	 * @param sslContext         the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(String ipAddress, int port, IWebHandler webHandler, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(ipAddress, port, webHandler, new HashMap<String, Object>(), sslContext, sslOn);
	}



	/**
	 * constructor <br><br>
	 *
	 *
	 * @param ipAddress            local ip address
	 * @param port                 local port
	 * @param options              the acceptor socket options
     * @param webHandler           the webHandler 
	 * @param sslOn                true, is SSL should be activated
	 * @param sslContext           the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(String ipAddress, int port, IWebHandler webHandler, Map<String, Object> options, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(InetAddress.getByName(ipAddress), port, webHandler, options, sslContext, sslOn);
	}


	/**
	 * constructor <br><br>
	 *
	 *
	 * @param address            local address
	 * @param port               local port
     * @param webHandler         the webHandler
	 * @param sslOn              true, is SSL should be activated
	 * @param sslContext         the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(InetAddress address, int port, IWebHandler webHandler, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(address, port, webHandler, new HashMap<String, Object>(), sslContext, sslOn);
	}


	/**
	 * constructor <br><br>
	 *
	 *
	 * @param address              local address
	 * @param port                 local port
	 * @param options              the socket options
     * @param webHandler           the webHandler 
	 * @param sslOn                true, is SSL should be activated
	 * @param sslContext           the ssl context to use
	 * @throws UnknownHostException if the local host cannot determined
	 * @throws IOException If some other I/O error occurs
	 */
	public XHttpServer(InetAddress address, int port, IWebHandler webHandler, Map<String, Object> options, SSLContext sslContext, boolean sslOn) throws UnknownHostException, IOException {
		this(new InetSocketAddress(address, port), options, webHandler, sslContext, sslOn, MIN_SIZE_WORKER_POOL, SIZE_WORKER_POOL);
	}
	 
	
    XHttpServer(InetSocketAddress address, Map<String, Object> options, IWebHandler webHandler, SSLContext sslContext, boolean sslOn, int minPoolsize, int maxPoolsize) throws UnknownHostException, IOException {
        super(address, options, getRequestHandler(webHandler), new WebSocketUpgradeHandler(webHandler), sslContext, sslOn, minPoolsize, maxPoolsize);
    }


    private static IHttpRequestHandler getRequestHandler(IWebHandler webHandler) {
        if (IHttpRequestHandler.class.isAssignableFrom(webHandler.getClass())) {
            return ((IHttpRequestHandler) webHandler);
        } else {
            return null;
        }
    }
    
    
    
	private static final class WebSocketUpgradeHandler implements IUpgradeHandler {
	    
	    private final IWebSocketHandler webSocketHandler;
	    
	    public WebSocketUpgradeHandler(IWebHandler webHandler) {
	        
	        if (IWebSocketHandler.class.isAssignableFrom(webHandler.getClass())) {
	            webSocketHandler = (IWebSocketHandler) webHandler;
	        } else {
	            webSocketHandler = null;
	        }
        }
	    
	    public boolean onRequest(IHttpExchange exchange) throws IOException, BadMessageException {
	        IHttpRequest request = exchange.getRequest();
	        
            if ((webSocketHandler != null) && request.getHeader("Upgrade").equalsIgnoreCase("WebSocket") && request.getProtocolVersion().endsWith("1.1")) {
                new WebSocketConnection((HttpServerConnection) exchange.getConnection(), webSocketHandler, exchange);
                return true;
            } else {
                return false;
            }
	    }
	}	
}
