/*
 * Decompiled with CFR 0.152.
 */
package com.questdb.cutlass.http;

import com.questdb.cutlass.http.HttpException;
import com.questdb.cutlass.http.HttpHeaderParser;
import com.questdb.std.ObjectFactory;
import com.questdb.std.ObjectPool;
import com.questdb.std.Unsafe;
import com.questdb.std.str.DirectByteCharSequence;
import com.questdb.test.tools.TestUtils;
import org.junit.Assert;
import org.junit.Test;

public class HttpHeaderParserTest {
    private static final String request = "GET /status?x=1&a=%26b&c&d=x HTTP/1.1\r\nHost: localhost:9000\r\nConnection: keep-alive\r\nCache-Control: max-age=0\r\nAccept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8\r\nUser-Agent: Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/31.0.1650.48 Safari/537.36\r\nContent-Type: multipart/form-data; boundary=----WebKitFormBoundaryQ3pdBTBXxEFUWDML\r\nAccept-Encoding: gzip,deflate,sdch\r\nAccept-Language: en-US,en;q=0.8\r\nCookie: textwrapon=false; textautoformat=false; wysiwyg=textarea\r\n\r\n";
    private static ObjectPool<DirectByteCharSequence> pool = new ObjectPool((ObjectFactory)DirectByteCharSequence.FACTORY, 64);

    @Test
    public void testContentTypeAndCharset() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Type: text/html; charset=utf-8\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                TestUtils.assertEquals((CharSequence)"text/html", hp.getContentType());
                TestUtils.assertEquals((CharSequence)"utf-8", (CharSequence)hp.getCharset());
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDisposition() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=\"hello\"\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                TestUtils.assertEquals((CharSequence)"hello", hp.getContentDispositionName());
                Assert.assertNull((Object)hp.getContentDispositionFilename());
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDispositionAndFileName() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=\"hello\"; filename=\"xyz.dat\"\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                TestUtils.assertEquals((CharSequence)"hello", hp.getContentDispositionName());
                TestUtils.assertEquals((CharSequence)"xyz.dat", hp.getContentDispositionFilename());
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDispositionMissingKeyName() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=\"hello\";\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                Assert.fail();
            }
            catch (HttpException e) {
                TestUtils.assertContains(e.getMessage(), "missing value [key=name]");
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDispositionUnclosedQuote() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=\"hello\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                Assert.fail();
            }
            catch (HttpException e) {
                TestUtils.assertContains(e.getMessage(), "unclosed quote");
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDispositionUnknown() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=\"hello\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                Assert.fail();
            }
            catch (HttpException e) {
                TestUtils.assertContains(e.getMessage(), "unclosed quote");
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentDispositionUnquoted() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Disposition: form-data; name=hello\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                TestUtils.assertEquals((CharSequence)"hello", hp.getContentDispositionName());
                Assert.assertNull((Object)hp.getContentDispositionFilename());
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    @Test
    public void testContentTypeAndCharsetAndDanglingBoundary() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "Content-Type: text/html; charset=utf-8; \r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), false);
                Assert.fail();
            }
            catch (HttpException e) {
                TestUtils.assertContains(e.getMessage(), "Malformed content-type header");
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testContentTypeAndUnknown() {
        String v = "Content-Type: text/html; encoding=abc\r\n\r\n";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), false);
            TestUtils.assertEquals((CharSequence)"text/html", hp.getContentType());
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testContentTypeBoundaryAndUnknown() {
        String v = "Content-Type: text/html; boundary=----WebKitFormBoundaryQ3pdBTBXxEFUWDML; encoding=abc\r\n\r\n";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), false);
            TestUtils.assertEquals((CharSequence)"text/html", hp.getContentType());
            TestUtils.assertEquals((CharSequence)"----WebKitFormBoundaryQ3pdBTBXxEFUWDML", hp.getBoundary());
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testContentTypeNoCharset() {
        String v = "Content-Type: text/html\r\n\r\n";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), false);
            TestUtils.assertEquals((CharSequence)"text/html", hp.getContentType());
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testDanglingUrlParamWithoutValue() {
        String request = "GET /status?accept HTTP/1.1\r\nHost: localhost:9000\r\n\r\n";
        try (HttpHeaderParser hp = new HttpHeaderParser(4096, pool);){
            long p = TestUtils.toMemory(request);
            try {
                hp.parse(p, request.length(), true);
                Assert.assertNull((Object)hp.getUrlParam((CharSequence)"accept"));
            }
            finally {
                Unsafe.free((long)p, (long)request.length());
            }
        }
    }

    @Test
    public void testQueryInvalidEncoding() throws Exception {
        TestUtils.assertMemoryLeak(() -> {
            String v = "GET /status?x=1&a=%i6b&c&d=x HTTP/1.1\r\n\r\n";
            long p = TestUtils.toMemory(v);
            try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
                hp.parse(p, v.length(), true);
                Assert.fail();
            }
            catch (HttpException e) {
                TestUtils.assertContains(e.getMessage(), "invalid query encoding");
            }
            finally {
                Unsafe.free((long)p, (long)v.length());
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSplitWrite() {
        try (HttpHeaderParser hp = new HttpHeaderParser(4096, pool);){
            long p = TestUtils.toMemory(request);
            try {
                int n = request.length();
                for (int i = 0; i < n; ++i) {
                    hp.clear();
                    hp.parse(p, i, true);
                    Assert.assertTrue((boolean)hp.isIncomplete());
                    hp.parse(p + (long)i, n - i, true);
                    this.assertHeaders(hp);
                }
            }
            finally {
                Unsafe.free((long)p, (long)request.length());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlNoQuery() {
        String v = "GET /xyz HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"/xyz", hp.getUrl());
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamSingleQuote() {
        String v = "GET /ip?x=%27a%27&y==b HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"'a'", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b", hp.getUrlParam((CharSequence)"y"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsDecode() {
        String v = "GET /test?x=a&y=b+c%26&z=ab%20ba&w=2 HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b c&", hp.getUrlParam((CharSequence)"y"));
            TestUtils.assertEquals((CharSequence)"ab ba", hp.getUrlParam((CharSequence)"z"));
            TestUtils.assertEquals((CharSequence)"2", hp.getUrlParam((CharSequence)"w"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsDecodeSpace() {
        String v = "GET /ok?x=a&y=b+c&z=123 HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b c", hp.getUrlParam((CharSequence)"y"));
            TestUtils.assertEquals((CharSequence)"123", hp.getUrlParam((CharSequence)"z"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsDecodeTrailingSpace() {
        String v = "GET /xyz?x=a&y=b+c HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b c", hp.getUrlParam((CharSequence)"y"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsDuplicateAmp() {
        String v = "GET /query?x=a&&y==b HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b", hp.getUrlParam((CharSequence)"y"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsSimple() {
        String v = "GET /query?x=a&y=b HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b", hp.getUrlParam((CharSequence)"y"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsTrailingEmpty() {
        String v = "GET /ip?x=a&y=b&z= HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b", hp.getUrlParam((CharSequence)"y"));
            Assert.assertNull((Object)hp.getUrlParam((CharSequence)"z"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testUrlParamsTrailingNull() {
        String v = "GET /opi?x=a&y=b& HTTP/1.1";
        long p = TestUtils.toMemory(v);
        try (HttpHeaderParser hp = new HttpHeaderParser(1024, pool);){
            hp.parse(p, v.length(), true);
            TestUtils.assertEquals((CharSequence)"a", hp.getUrlParam((CharSequence)"x"));
            TestUtils.assertEquals((CharSequence)"b", hp.getUrlParam((CharSequence)"y"));
        }
        finally {
            Unsafe.free((long)p, (long)v.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testWrite() {
        try (HttpHeaderParser hp = new HttpHeaderParser(4096, pool);){
            long p = TestUtils.toMemory(request);
            try {
                hp.parse(p, request.length(), true);
                this.assertHeaders(hp);
            }
            finally {
                Unsafe.free((long)p, (long)request.length());
            }
        }
    }

    private void assertHeaders(HttpHeaderParser hp) {
        Assert.assertFalse((boolean)hp.isIncomplete());
        TestUtils.assertEquals((CharSequence)"GET", hp.getMethod());
        TestUtils.assertEquals((CharSequence)"/status", hp.getUrl());
        TestUtils.assertEquals((CharSequence)"GET /status?x=1&a=&b&c&d=x HTTP/1.1", hp.getMethodLine());
        Assert.assertEquals((long)9L, (long)hp.size());
        TestUtils.assertEquals((CharSequence)"localhost:9000", (CharSequence)hp.getHeader((CharSequence)"Host"));
        TestUtils.assertEquals((CharSequence)"keep-alive", (CharSequence)hp.getHeader((CharSequence)"Connection"));
        TestUtils.assertEquals((CharSequence)"max-age=0", (CharSequence)hp.getHeader((CharSequence)"Cache-Control"));
        TestUtils.assertEquals((CharSequence)"text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8", (CharSequence)hp.getHeader((CharSequence)"Accept"));
        TestUtils.assertEquals((CharSequence)"Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/31.0.1650.48 Safari/537.36", (CharSequence)hp.getHeader((CharSequence)"User-Agent"));
        TestUtils.assertEquals((CharSequence)"multipart/form-data; boundary=----WebKitFormBoundaryQ3pdBTBXxEFUWDML", (CharSequence)hp.getHeader((CharSequence)"Content-Type"));
        TestUtils.assertContains("----WebKitFormBoundaryQ3pdBTBXxEFUWDML", hp.getBoundary());
        TestUtils.assertEquals((CharSequence)"gzip,deflate,sdch", (CharSequence)hp.getHeader((CharSequence)"Accept-Encoding"));
        TestUtils.assertEquals((CharSequence)"en-US,en;q=0.8", (CharSequence)hp.getHeader((CharSequence)"Accept-Language"));
        TestUtils.assertEquals((CharSequence)"textwrapon=false; textautoformat=false; wysiwyg=textarea", (CharSequence)hp.getHeader((CharSequence)"Cookie"));
        TestUtils.assertEquals((CharSequence)"1", hp.getUrlParam((CharSequence)"x"));
        TestUtils.assertEquals((CharSequence)"&b", hp.getUrlParam((CharSequence)"a"));
        Assert.assertNull((Object)hp.getUrlParam((CharSequence)"c"));
        Assert.assertNull((Object)hp.getHeader((CharSequence)"xxx"));
    }
}

