/*
 * OfficeFloor - http://www.officefloor.net
 * Copyright (C) 2005-2018 Daniel Sagenschneider
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package net.officefloor.eclipse.editor;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.net.URL;
import java.util.Dictionary;
import java.util.Hashtable;

import org.eclipse.core.runtime.Plugin;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.url.URLStreamHandlerService;

import javafx.beans.property.ReadOnlyProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.scene.Scene;
import net.officefloor.eclipse.editor.internal.style.AbstractStyleRegistry;
import net.officefloor.eclipse.editor.internal.style.OsgiStyleRegistry;
import net.officefloor.eclipse.editor.internal.style.OsgiURLStreamHandlerService;
import net.officefloor.eclipse.editor.internal.style.StyleRegistry;
import net.officefloor.eclipse.editor.internal.style.SystemStyleRegistry;

/**
 * {@link Plugin} for the {@link AdaptedEditorModule}.
 * 
 * @author Daniel Sagenschneider
 */
public class AdaptedEditorPlugin extends AbstractUIPlugin {

	/**
	 * Creates a new {@link StyleRegistry}.
	 * 
	 * @return New {@link StyleRegistry}.
	 */
	public static StyleRegistry createStyleRegistry() {

		// Determine if running within OSGi environment
		if (INSTANCE == null) {

			// Running outside OSGi environment
			return new SystemStyleRegistry();
		}

		// Running within OSGi environment
		return new OsgiStyleRegistry();
	}

	/**
	 * Obtains the default style sheet.
	 * 
	 * @return Default style sheet.
	 */
	public static String getDefaultStyleSheet() {

		// Obtain the style sheet
		InputStream defaultStyleSheet = AdaptedEditorModule.class
				.getResourceAsStream(AdaptedEditorModule.class.getSimpleName() + ".css");
		if (defaultStyleSheet == null) {
			// Should not occur
			throw new IllegalStateException("Default style sheet not available on class path");
		}

		// Read in the deafult style sheet
		try (Reader reader = new InputStreamReader(defaultStyleSheet)) {

			// Read in the style sheet
			StringWriter stylesheet = new StringWriter();
			for (int character = reader.read(); character != -1; character = reader.read()) {
				stylesheet.write(character);
			}

			// Return the style sheet
			return stylesheet.toString();

		} catch (IOException ex) {
			// Should not occur
			throw new IllegalStateException("Failed to load default style sheet", ex);
		}
	}

	/**
	 * Loads the default style sheet.
	 * 
	 * @param scene
	 *            {@link Scene}.
	 */
	public static void loadDefaulStylesheet(Scene scene) {

		// Determine if running within OSGi environment
		if (INSTANCE == null) {

			// Running outside OSGi environment (so load from class path)
			scene.getStylesheets().add(AdaptedEditorModule.class.getName().replace('.', '/') + ".css");
			return;
		}

		// Ensure have the default style sheet loaded (does not change so URL constant)
		if (DEFAULT_STYLESHEET_URL == null) {

			// Obtain the default style sheet
			String styleSheet = getDefaultStyleSheet();

			// Provide style registry URL to default style sheet
			StyleRegistry registry = createStyleRegistry();
			ReadOnlyProperty<URL> defaultStyleSheetUrl = registry.registerStyle("_default_",
					new SimpleStringProperty(styleSheet));
			DEFAULT_STYLESHEET_URL = defaultStyleSheetUrl.getValue();
		}

		// Load the default style sheet
		scene.getStylesheets().add(DEFAULT_STYLESHEET_URL.toExternalForm());
	}

	/**
	 * Obtains the {@link AdaptedEditorPlugin} singleton.
	 * 
	 * @return {@link AdaptedEditorPlugin} singleton.
	 */
	public AdaptedEditorPlugin getDefault() {
		return INSTANCE;
	}

	/**
	 * Singleton.
	 */
	private static AdaptedEditorPlugin INSTANCE;

	/**
	 * Default stylesheet {@link URL}.
	 */
	private static URL DEFAULT_STYLESHEET_URL;

	/**
	 * {@link ServiceRegistration} for the {@link OsgiURLStreamHandlerService}.
	 */
	private ServiceRegistration<?> styleUrlHandler;

	/**
	 * Instantiate.
	 */
	public AdaptedEditorPlugin() {
		INSTANCE = this;
	}

	/*
	 * =============== AbstractUIPlugin =========================
	 */

	@Override
	public void start(BundleContext context) throws Exception {
		super.start(context);

		// Register the URL handler for styling
		Dictionary<String, String> properties = new Hashtable<>();
		properties.put("url.handler.protocol", AbstractStyleRegistry.PROTOCOL);
		this.styleUrlHandler = context.registerService(URLStreamHandlerService.class.getName(),
				new OsgiURLStreamHandlerService(), properties);
	}

	@Override
	public void stop(BundleContext context) throws Exception {
		super.stop(context);

		// Unregister style URL handler
		if (this.styleUrlHandler != null) {
			context.ungetService(this.styleUrlHandler.getReference());
			this.styleUrlHandler = null;
		}
	}

}