/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.core.http.StreamPriority;
import java.util.function.Function;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.reactivex.core.http.HttpClient} instance, via one of the methods corresponding to the
 * specific HTTP methods, or the generic request methods. On creation the request will not have been written to the
 * wire.
 * <p>
 * Once a request has been obtained, headers can be set on it, and data can be written to its body if required. Once
 * you are ready to send the request, one of the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} methods should be called.
 * <p>
 * Nothing is actually sent until the request has been internally assigned an HTTP connection.
 * <p>
 * The {@link io.vertx.reactivex.core.http.HttpClient} instance will return an instance of this class immediately, even if there are no HTTP
 * connections available in the pool. Any requests sent before a connection is assigned will be queued
 * internally and actually sent when an HTTP connection becomes available from the pool.
 * <p>
 * The headers of the request are queued for writing either when the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} method is called, or, when the first
 * part of the body is written, whichever occurs first.
 * <p>
 * This class supports both chunked and non-chunked HTTP.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pump} to pump data with flow control.
 * <p>
 * An example of using this class is as follows:
 * <p>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClientRequest original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rx.RxGen(io.vertx.core.http.HttpClientRequest.class)
public class HttpClientRequest extends io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> implements io.vertx.reactivex.core.streams.WriteStream<io.vertx.reactivex.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClientRequest that = (HttpClientRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.rx.TypeArg<HttpClientRequest> __TYPE_ARG = new io.vertx.lang.rx.TypeArg<>(    obj -> new HttpClientRequest((io.vertx.core.http.HttpClientRequest) obj),
    HttpClientRequest::getDelegate
  );

  private final io.vertx.core.http.HttpClientRequest delegate;
  
  public HttpClientRequest(io.vertx.core.http.HttpClientRequest delegate) {
    super(delegate);
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpClientRequest getDelegate() {
    return delegate;
  }

  private io.vertx.reactivex.WriteStreamObserver<io.vertx.reactivex.core.buffer.Buffer> observer;
  private io.vertx.reactivex.WriteStreamSubscriber<io.vertx.reactivex.core.buffer.Buffer> subscriber;

  public synchronized io.vertx.reactivex.WriteStreamObserver<io.vertx.reactivex.core.buffer.Buffer> toObserver() {
    if (observer == null) {
      java.util.function.Function<io.vertx.reactivex.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.reactivex.core.buffer.Buffer::getDelegate;
      observer = io.vertx.reactivex.RxHelper.toObserver(getDelegate(), conv);
    }
    return observer;
  }

  public synchronized io.vertx.reactivex.WriteStreamSubscriber<io.vertx.reactivex.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      java.util.function.Function<io.vertx.reactivex.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.reactivex.core.buffer.Buffer::getDelegate;
      subscriber = io.vertx.reactivex.RxHelper.toSubscriber(getDelegate(), conv);
    }
    return subscriber;
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @param handler 
   */
  public void write(io.vertx.reactivex.core.buffer.Buffer data, Handler<AsyncResult<Void>> handler) { 
    delegate.write(data.getDelegate(), handler);
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   */
  public void write(io.vertx.reactivex.core.buffer.Buffer data) {
    write(data, ar -> { });
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @return 
   */
  public Completable rxWrite(io.vertx.reactivex.core.buffer.Buffer data) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      write(data, handler);
    });
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpClientRequest#setWriteQueueMaxSize}
   * @return true if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  /**
   * @return the handler for the result
   */
  public Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> getHandler() { 
    Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> ret = new Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          delegate.getHandler().handle(io.vertx.core.Future.succeededFuture(ar.result().getDelegate()));
        } else {
          delegate.getHandler().handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    };
    return ret;
  }

  /**
   * The result of the operation. This will be null if the operation failed.
   * @return the result or null if the operation failed.
   */
  public io.vertx.reactivex.core.http.HttpClientResponse result() { 
    io.vertx.reactivex.core.http.HttpClientResponse ret = io.vertx.reactivex.core.http.HttpClientResponse.newInstance(delegate.result());
    return ret;
  }

  /**
   * Compose this future with a <code>mapper</code> function.<p>
   *
   * When this future (the one on which <code>compose</code> is called) succeeds, the <code>mapper</code> will be called with
   * the completed value and this mapper returns another future object. This returned future completion will complete
   * the future returned by this method call.<p>
   *
   * If the <code>mapper</code> throws an exception, the returned future will be failed with this exception.<p>
   *
   * When this future fails, the failure will be propagated to the returned future and the <code>mapper</code>
   * will not be called.
   * @param mapper the mapper function
   * @return the composed future
   */
  public <U> io.vertx.reactivex.core.Future<U> compose(Function<io.vertx.reactivex.core.http.HttpClientResponse, io.vertx.reactivex.core.Future<U>> mapper) { 
    io.vertx.reactivex.core.Future<U> ret = io.vertx.reactivex.core.Future.newInstance(delegate.compose(new java.util.function.Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<U>>() {
      public io.vertx.core.Future<U> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.vertx.reactivex.core.Future<U> ret = mapper.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(arg));
        return ret.getDelegate();
      }
    }), io.vertx.lang.rx.TypeArg.unknown());
    return ret;
  }

  /**
   * Apply a <code>mapper</code> function on this future.<p>
   *
   * When this future succeeds, the <code>mapper</code> will be called with the completed value and this mapper
   * returns a value. This value will complete the future returned by this method call.<p>
   *
   * If the <code>mapper</code> throws an exception, the returned future will be failed with this exception.<p>
   *
   * When this future fails, the failure will be propagated to the returned future and the <code>mapper</code>
   * will not be called.
   * @param mapper the mapper function
   * @return the mapped future
   */
  public <U> io.vertx.reactivex.core.Future<U> map(Function<io.vertx.reactivex.core.http.HttpClientResponse, U> mapper) { 
    io.vertx.reactivex.core.Future<U> ret = io.vertx.reactivex.core.Future.newInstance(delegate.map(new java.util.function.Function<io.vertx.core.http.HttpClientResponse,U>() {
      public U apply(io.vertx.core.http.HttpClientResponse arg) {
        U ret = mapper.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(arg));
        return ret;
      }
    }), io.vertx.lang.rx.TypeArg.unknown());
    return ret;
  }

  /**
   * Handles a failure of this Future by returning the result of another Future.
   * If the mapper fails, then the returned future will be failed with this failure.
   * @param mapper A function which takes the exception of a failure and returns a new future.
   * @return A recovered future
   */
  public io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> recover(Function<Throwable, io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse>> mapper) { 
    io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = io.vertx.reactivex.core.Future.newInstance(delegate.recover(new java.util.function.Function<java.lang.Throwable,io.vertx.core.Future<io.vertx.core.http.HttpClientResponse>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientResponse> apply(java.lang.Throwable arg) {
        io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = mapper.apply(arg);
        return ret.getDelegate();
      }
    }), (io.vertx.lang.rx.TypeArg)io.vertx.reactivex.core.http.HttpClientResponse.__TYPE_ARG);
    return ret;
  }

  /**
   * Apply a <code>mapper</code> function on this future.<p>
   *
   * When this future fails, the <code>mapper</code> will be called with the completed value and this mapper
   * returns a value. This value will complete the future returned by this method call.<p>
   *
   * If the <code>mapper</code> throws an exception, the returned future will be failed with this exception.<p>
   *
   * When this future succeeds, the result will be propagated to the returned future and the <code>mapper</code>
   * will not be called.
   * @param mapper the mapper function
   * @return the mapped future
   */
  public io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> otherwise(Function<Throwable, io.vertx.reactivex.core.http.HttpClientResponse> mapper) { 
    io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = io.vertx.reactivex.core.Future.newInstance(delegate.otherwise(new java.util.function.Function<java.lang.Throwable,io.vertx.core.http.HttpClientResponse>() {
      public io.vertx.core.http.HttpClientResponse apply(java.lang.Throwable arg) {
        io.vertx.reactivex.core.http.HttpClientResponse ret = mapper.apply(arg);
        return ret.getDelegate();
      }
    }), (io.vertx.lang.rx.TypeArg)io.vertx.reactivex.core.http.HttpClientResponse.__TYPE_ARG);
    return ret;
  }

  /**
   * Map the failure of a future to a specific <code>value</code>.<p>
   *
   * When this future fails, this <code>value</code> will complete the future returned by this method call.<p>
   *
   * When this future succeeds, the result will be propagated to the returned future.
   * @param value the value that eventually completes the mapped future
   * @return the mapped future
   */
  public io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> otherwise(io.vertx.reactivex.core.http.HttpClientResponse value) { 
    io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = io.vertx.reactivex.core.Future.newInstance(delegate.otherwise(value.getDelegate()), (io.vertx.lang.rx.TypeArg)io.vertx.reactivex.core.http.HttpClientResponse.__TYPE_ARG);
    return ret;
  }

  /**
   * Map the failure of a future to <code>null</code>.<p>
   *
   * This is a convenience for <code>future.otherwise((T) null)</code>.<p>
   *
   * When this future fails, the <code>null</code> value will complete the future returned by this method call.<p>
   *
   * When this future succeeds, the result will be propagated to the returned future.
   * @return the mapped future
   */
  public io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> otherwiseEmpty() { 
    io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = io.vertx.reactivex.core.Future.newInstance(delegate.otherwiseEmpty(), (io.vertx.lang.rx.TypeArg)io.vertx.reactivex.core.http.HttpClientResponse.__TYPE_ARG);
    return ret;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * Set the request to follow HTTP redirects up to {@link io.vertx.core.http.HttpClientOptions}.
   * @param followRedirects <code>true</code> to follow HTTP redirects
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setFollowRedirects(boolean followRedirects) { 
    delegate.setFollowRedirects(followRedirects);
    return this;
  }

  /**
   * Set the max number of HTTP redirects this request will follow. The default is <code>0</code> which means
   * no redirects.
   * @param maxRedirects the number of HTTP redirect to follow
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setMaxRedirects(int maxRedirects) { 
    delegate.setMaxRedirects(maxRedirects);
    return this;
  }

  /**
   * If chunked is true then the request will be set into HTTP chunked mode
   * @param chunked true if chunked encoding
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return Is the request chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * The HTTP method for the request.
   * @return 
   */
  public HttpMethod method() { 
    HttpMethod ret = delegate.method();
    return ret;
  }

  /**
   * @return the raw value of the method this request sends
   */
  public String getRawMethod() { 
    String ret = delegate.getRawMethod();
    return ret;
  }

  /**
   * Set the value the method to send when the method  is used.
   * @param method the raw method
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setRawMethod(String method) { 
    delegate.setRawMethod(method);
    return this;
  }

  /**
   * @return the absolute URI corresponding to the the HTTP request
   */
  public String absoluteURI() { 
    String ret = delegate.absoluteURI();
    return ret;
  }

  /**
   * @return The URI of the request.
   */
  public String uri() { 
    String ret = delegate.uri();
    return ret;
  }

  /**
   * @return The path part of the uri. For example /somepath/somemorepath/someresource.foo
   */
  public String path() { 
    String ret = delegate.path();
    return ret;
  }

  /**
   * @return the query part of the uri. For example someparam=32&amp;someotherparam=x
   */
  public String query() { 
    String ret = delegate.query();
    return ret;
  }

  /**
   * Set the request host.<p/>
   *
   * For HTTP/2 it sets the  pseudo header otherwise it sets the  header
   * @param host 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setHost(String host) { 
    delegate.setHost(host);
    return this;
  }

  /**
   * @return the request host. For HTTP/2 it returns the  pseudo header otherwise it returns the  header
   */
  public String getHost() { 
    String ret = delegate.getHost();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.reactivex.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.reactivex.core.MultiMap ret = io.vertx.reactivex.core.MultiMap.newInstance(delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name The header name
   * @param value The header value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void write(String chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.write(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void write(String chunk) {
    write(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public Completable rxWrite(String chunk) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      write(chunk, handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void write(String chunk, String enc, Handler<AsyncResult<Void>> handler) { 
    delegate.write(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void write(String chunk, String enc) {
    write(chunk, enc, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public Completable rxWrite(String chunk, String enc) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      write(chunk, enc, handler);
    });
  }

  /**
   * If you send an HTTP request with the header <code>Expect</code> set to the value <code>100-continue</code>
   * and the server responds with an interim HTTP response with a status code of <code>100</code> and a continue handler
   * has been set using this method, then the <code>handler</code> will be called.
   * <p>
   * You can then continue to write data to the request body and later end it. This is normally used in conjunction with
   * the {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} method to force the request header to be written before the request has ended.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest continueHandler(Handler<Void> handler) { 
    delegate.continueHandler(handler);
    return this;
  }

  /**
   * Forces the head of the request to be written before {@link io.vertx.reactivex.core.http.HttpClientRequest#end} is called on the request or any data is
   * written to it.
   * <p>
   * This is normally used to implement HTTP 100-continue handling, see  for
   * more information.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest sendHead() { 
    delegate.sendHead();
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * @param completionHandler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest sendHead(Handler<HttpVersion> completionHandler) { 
    delegate.sendHead(completionHandler);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(String chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(String chunk) {
    end(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public Completable rxEnd(String chunk) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      end(chunk, handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void end(String chunk, String enc, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void end(String chunk, String enc) {
    end(chunk, enc, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public Completable rxEnd(String chunk, String enc) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      end(chunk, enc, handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(io.vertx.reactivex.core.buffer.Buffer chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk.getDelegate(), handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(io.vertx.reactivex.core.buffer.Buffer chunk) {
    end(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public Completable rxEnd(io.vertx.reactivex.core.buffer.Buffer chunk) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      end(chunk, handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param handler 
   */
  public void end(Handler<AsyncResult<Void>> handler) { 
    delegate.end(handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   */
  public void end() {
    end(ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @return 
   */
  public Completable rxEnd() { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      end(handler);
    });
  }

  /**
   * Set's the amount of time after which if the request does not return any data within the timeout period an
   * {@link java.util.concurrent.TimeoutException} will be passed to the exception handler (if provided) and
   * the request will be closed.
   * <p>
   * Calling this method more than once has the effect of canceling any existing timeout and starting
   * the timeout from scratch.
   * @param timeoutMs The quantity of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setTimeout(long timeoutMs) { 
    delegate.setTimeout(timeoutMs);
    return this;
  }

  /**
   * Set a push handler for this request.<p/>
   *
   * The handler is called when the client receives a <i>push promise</i> from the server. The handler can be called
   * multiple times, for each push promise.<p/>
   *
   * The handler is called with a <i>read-only</i> {@link io.vertx.reactivex.core.http.HttpClientRequest}, the following methods can be called:<p/>
   *
   * <ul>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#method}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#uri}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#headers}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#getHost}</li>
   * </ul>
   *
   * In addition the handler should call the {@link io.vertx.reactivex.core.http.HttpClientRequest#setHandler} method to set an handler to
   * process the response.<p/>
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest pushHandler(Handler<io.vertx.reactivex.core.http.HttpClientRequest> handler) { 
    delegate.pushHandler(new Handler<io.vertx.core.http.HttpClientRequest>() {
      public void handle(io.vertx.core.http.HttpClientRequest event) {
        handler.handle(io.vertx.reactivex.core.http.HttpClientRequest.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Reset this stream with the error code <code>0</code>.
   * @return 
   */
  public boolean reset() { 
    boolean ret = delegate.reset();
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, this performs send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, this closes the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and false is returned as indicator.
   * <p/>
   * @param code the error code
   * @return true when reset has been performed
   */
  public boolean reset(long code) { 
    boolean ret = delegate.reset(code);
    return ret;
  }

  /**
   * @return the {@link io.vertx.reactivex.core.http.HttpConnection} associated with this request
   */
  public io.vertx.reactivex.core.http.HttpConnection connection() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.reactivex.core.http.HttpConnection ret = io.vertx.reactivex.core.http.HttpConnection.newInstance(delegate.connection());
    cached_1 = ret;
    return ret;
  }

  /**
   * Set a connection handler called when an HTTP connection has been established.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest connectionHandler(Handler<io.vertx.reactivex.core.http.HttpConnection> handler) { 
    delegate.connectionHandler(new Handler<io.vertx.core.http.HttpConnection>() {
      public void handle(io.vertx.core.http.HttpConnection event) {
        handler.handle(io.vertx.reactivex.core.http.HttpConnection.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Write an HTTP/2 frame to the request, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.<p>
   *
   * This method must be called after the request headers have been sent and only for the protocol HTTP/2.
   * The {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} should be used for this purpose.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest writeCustomFrame(int type, int flags, io.vertx.reactivex.core.buffer.Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * @return the id of the stream of this response,  when it is not yet determined, i.e the request has not been yet sent or it is not supported HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#writeCustomFrame} but with an {@link io.vertx.reactivex.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest writeCustomFrame(io.vertx.reactivex.core.http.HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  /**
   * Sets the priority of the associated stream.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority of this request's stream
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setStreamPriority(StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * @return the priority of the associated HTTP/2 stream for HTTP/2 otherwise <code>null</code>
   */
  public StreamPriority getStreamPriority() { 
    StreamPriority ret = delegate.getStreamPriority();
    return ret;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest setHandler(Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> handler) { 
    delegate.setHandler(new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest setHandler() {
    return 
setHandler(ar -> { });
  }

  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSetHandler() { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      setHandler(handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(CharSequence name, CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Put an HTTP header with multiple values
   * @param name The header name
   * @param values The header values
   * @return @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(String name, Iterable<String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(CharSequence name, Iterable<CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  private io.vertx.reactivex.core.MultiMap cached_0;
  private io.vertx.reactivex.core.http.HttpConnection cached_1;

  public static  HttpClientRequest newInstance(io.vertx.core.http.HttpClientRequest arg) {
    return arg != null ? new HttpClientRequest(arg) : null;
  }
}
