/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.core.http.RequestOptions;
import io.vertx.core.http.WebsocketVersion;
import java.util.function.Function;
import java.util.List;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.WebSocketConnectOptions;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * An asynchronous HTTP client.
 * <p>
 * It allows you to make requests to HTTP servers, and a single client can make requests to any server.
 * <p>
 * It also allows you to open WebSockets to servers.
 * <p>
 * The client can also pool HTTP connections.
 * <p>
 * For pooling to occur, keep-alive must be true on the {@link io.vertx.core.http.HttpClientOptions} (default is true).
 * In this case connections will be pooled and re-used if there are pending HTTP requests waiting to get a connection,
 * otherwise they will be closed.
 * <p>
 * This gives the benefits of keep alive when the client is loaded but means we don't keep connections hanging around
 * unnecessarily when there would be no benefits anyway.
 * <p>
 * The client also supports pipe-lining of requests. Pipe-lining means another request is sent on the same connection
 * before the response from the preceding one has returned. Pipe-lining is not appropriate for all requests.
 * <p>
 * To enable pipe-lining, it must be enabled on the {@link io.vertx.core.http.HttpClientOptions} (default is false).
 * <p>
 * When pipe-lining is enabled the connection will be automatically closed when all in-flight responses have returned
 * and there are no outstanding pending requests to write.
 * <p>
 * The client is designed to be reused between requests.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClient original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rx.RxGen(io.vertx.core.http.HttpClient.class)
public class HttpClient implements io.vertx.reactivex.core.metrics.Measured {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClient that = (HttpClient) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.rx.TypeArg<HttpClient> __TYPE_ARG = new io.vertx.lang.rx.TypeArg<>(    obj -> new HttpClient((io.vertx.core.http.HttpClient) obj),
    HttpClient::getDelegate
  );

  private final io.vertx.core.http.HttpClient delegate;
  
  public HttpClient(io.vertx.core.http.HttpClient delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpClient getDelegate() {
    return delegate;
  }

  /**
   * Whether the metrics are enabled for this measured object
   * @return <code>true</code> if metrics are enabled
   */
  public boolean isMetricsEnabled() { 
    boolean ret = delegate.isMetricsEnabled();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#request} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>options</code> parameter.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param options 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, serverAddress.getDelegate(), options));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server with the specified options.
   * @param method the HTTP method
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, options));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and port.
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, port, host, requestURI));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#request} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>host</code> and <code>port</code> parameters.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param port 
   * @param host 
   * @param requestURI 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, serverAddress.getDelegate(), port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and default port.
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server with the specified options, specifying a response handler to receive
   * @param method the HTTP method
   * @param options the request options
   * @param responseHandler 
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#request} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>options</code> parameter.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param options 
   * @param responseHandler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, serverAddress.getDelegate(), options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#request} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>host</code> and <code>port</code> parameters.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param port 
   * @param host 
   * @param requestURI 
   * @param responseHandler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, serverAddress.getDelegate(), port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the default host and port.
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest request(HttpMethod method, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.request(method, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server using an absolute URI
   * @param method the HTTP method
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest requestAbs(HttpMethod method, String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.requestAbs(method, absoluteURI));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#requestAbs} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>absoluteURI</code> parameter.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param absoluteURI 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest requestAbs(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.requestAbs(method, serverAddress.getDelegate(), absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest requestAbs(HttpMethod method, String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.requestAbs(method, absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClient#requestAbs} using the <code>serverAddress</code> parameter to connect to the
   * server instead of the <code>absoluteURI</code> parameter.
   * <p>
   * The request host header will still be created from the <code>absoluteURI</code> parameter.
   * <p>
   * Use  to connect to a unix domain socket server.
   * @param method 
   * @param serverAddress 
   * @param absoluteURI 
   * @param responseHandler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest requestAbs(HttpMethod method, io.vertx.reactivex.core.net.SocketAddress serverAddress, String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.requestAbs(method, serverAddress.getDelegate(), absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(options));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest get(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.get(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest getAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.getAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest getAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.getAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP GET request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.getNow(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(RequestOptions options) {
    return 
getNow(options, ar -> { });
  }

  /**
   * Sends an HTTP GET request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxGetNow(RequestOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      getNow(options, handler);
    });
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.getNow(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(int port, String host, String requestURI) {
    return 
getNow(port, host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxGetNow(int port, String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      getNow(port, host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.getNow(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(String host, String requestURI) {
    return 
getNow(host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxGetNow(String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      getNow(host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP GET request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.getNow(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient getNow(String requestURI) {
    return 
getNow(requestURI, ar -> { });
  }

  /**
   * Sends an HTTP GET request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxGetNow(String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      getNow(requestURI, handler);
    });
  }

  /**
   * Create an HTTP POST request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(options));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest post(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.post(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest postAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.postAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest postAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.postAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(options));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest head(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.head(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest headAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.headAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest headAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.headAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP HEAD request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.headNow(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(RequestOptions options) {
    return 
headNow(options, ar -> { });
  }

  /**
   * Sends an HTTP HEAD request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxHeadNow(RequestOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      headNow(options, handler);
    });
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.headNow(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(int port, String host, String requestURI) {
    return 
headNow(port, host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxHeadNow(int port, String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      headNow(port, host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.headNow(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(String host, String requestURI) {
    return 
headNow(host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxHeadNow(String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      headNow(host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP HEAD request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.headNow(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient headNow(String requestURI) {
    return 
headNow(requestURI, ar -> { });
  }

  /**
   * Sends an HTTP HEAD request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxHeadNow(String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      headNow(requestURI, handler);
    });
  }

  /**
   * Create an HTTP OPTIONS request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(options));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest options(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.options(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest optionsAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.optionsAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest optionsAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.optionsAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP OPTIONS request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.optionsNow(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(RequestOptions options) {
    return 
optionsNow(options, ar -> { });
  }

  /**
   * Sends an HTTP OPTIONS request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxOptionsNow(RequestOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      optionsNow(options, handler);
    });
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.optionsNow(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(int port, String host, String requestURI) {
    return 
optionsNow(port, host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxOptionsNow(int port, String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      optionsNow(port, host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.optionsNow(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(String host, String requestURI) {
    return 
optionsNow(host, requestURI, ar -> { });
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxOptionsNow(String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      optionsNow(host, requestURI, handler);
    });
  }

  /**
   * Sends an HTTP OPTIONS request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    delegate.optionsNow(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient optionsNow(String requestURI) {
    return 
optionsNow(requestURI, ar -> { });
  }

  /**
   * Sends an HTTP OPTIONS request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.http.HttpClientResponse> rxOptionsNow(String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      optionsNow(requestURI, handler);
    });
  }

  /**
   * Create an HTTP PUT request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(options));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest put(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.put(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.putAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.putAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(RequestOptions options) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(options));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(int port, String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(String host, String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(String requestURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest delete(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.delete(requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest deleteAbs(String absoluteURI) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.deleteAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public io.vertx.reactivex.core.http.HttpClientRequest deleteAbs(String absoluteURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientResponse>> responseHandler) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance(delegate.deleteAbs(absoluteURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientResponse> ar) {
        if (ar.succeeded()) {
          responseHandler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(ar.result())));
        } else {
          responseHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param handler handler that will be called with the websocket when connected
   */
  public void webSocket(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   */
  public void webSocket(int port, String host, String requestURI) {
    webSocket(port, host, requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(int port, String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      webSocket(port, host, requestURI, handler);
    });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @param handler handler that will be called with the websocket when connected
   */
  public void webSocket(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   */
  public void webSocket(String host, String requestURI) {
    webSocket(host, requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(String host, String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      webSocket(host, requestURI, handler);
    });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @param handler handler that will be called with the websocket when connected
   */
  public void webSocket(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   */
  public void webSocket(String requestURI) {
    webSocket(requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @return 
   */
  public Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(String requestURI) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      webSocket(requestURI, handler);
    });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   * @param handler 
   */
  public void webSocket(WebSocketConnectOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(options, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   */
  public void webSocket(WebSocketConnectOptions options) {
    webSocket(options, ar -> { });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   * @return 
   */
  public Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(WebSocketConnectOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      webSocket(options, handler);
    });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param handler handler that will be called if websocket connection fails
   */
  public void webSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, WebsocketVersion version, List<String> subProtocols, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocketAbs(url, headers.getDelegate(), version, subProtocols, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   */
  public void webSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, WebsocketVersion version, List<String> subProtocols) {
    webSocketAbs(url, headers, version, subProtocols, ar -> { });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @return 
   */
  public Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, WebsocketVersion version, List<String> subProtocols) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      webSocketAbs(url, headers, version, subProtocols, handler);
    });
  }

  /**
   * Set a connection handler for the client. This handler is called when a new connection is established.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient connectionHandler(Handler<io.vertx.reactivex.core.http.HttpConnection> handler) { 
    delegate.connectionHandler(new Handler<io.vertx.core.http.HttpConnection>() {
      public void handle(io.vertx.core.http.HttpConnection event) {
        handler.handle(io.vertx.reactivex.core.http.HttpConnection.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Set a redirect handler for the http client.
   * <p>
   * The redirect handler is called when a <code>3xx</code> response is received and the request is configured to
   * follow redirects with {@link io.vertx.reactivex.core.http.HttpClientRequest#setFollowRedirects}.
   * <p>
   * The redirect handler is passed the {@link io.vertx.reactivex.core.http.HttpClientResponse}, it can return an {@link io.vertx.reactivex.core.http.HttpClientRequest} or <code>null</code>.
   * <ul>
   *   <li>when null is returned, the original response is processed by the original request response handler</li>
   *   <li>when a new <code>Future<HttpClientRequest></code> is returned, the client will send this new request</li>
   * </ul>
   * The new request will get a copy of the previous request headers unless headers are set. In this case,
   * the client assumes that the redirect handler exclusively managers the headers of the new request.
   * <p>
   * The handler must return a <code>Future<HttpClientRequest></code> unsent so the client can further configure it and send it.
   * @param handler the new redirect handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient redirectHandler(Function<io.vertx.reactivex.core.http.HttpClientResponse, io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.redirectHandler(new java.util.function.Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.HttpClientRequest>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientRequest> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.vertx.reactivex.core.Future<io.vertx.reactivex.core.http.HttpClientRequest> ret = handler.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance(arg));
        return ret.getDelegate();
      }
    });
    return this;
  }

  /**
   * Close the client. Closing will close down any pooled connections.
   * Clients should always be closed after use.
   */
  public void close() { 
    delegate.close();
  }


  public static  HttpClient newInstance(io.vertx.core.http.HttpClient arg) {
    return arg != null ? new HttpClient(arg) : null;
  }
}
