/*
 * Copyright (c) 2011-2021 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */
package io.vertx.oracleclient.impl.commands;

import io.vertx.core.Future;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.impl.ContextInternal;
import io.vertx.oracleclient.OracleConnectOptions;
import io.vertx.oracleclient.OraclePrepareOptions;
import io.vertx.oracleclient.impl.Helper;
import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.Tuple;
import io.vertx.sqlclient.impl.command.ExtendedQueryCommand;
import oracle.jdbc.OracleConnection;
import oracle.jdbc.OraclePreparedStatement;

import java.sql.*;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.List;
import java.util.stream.Collector;

import static io.vertx.oracleclient.impl.Helper.closeQuietly;
import static io.vertx.oracleclient.impl.Helper.unwrapOraclePreparedStatement;

public class OraclePreparedBatch<C, R> extends QueryCommand<C, R> {

  private final ExtendedQueryCommand<R> query;
  private final List<Tuple> listParams;

  public OraclePreparedBatch(OracleConnectOptions options, ExtendedQueryCommand<R> query,
    Collector<Row, C, R> collector, List<Tuple> listParams) {
    super(options, collector);
    this.query = query;
    this.listParams = listParams;
  }

  @Override
  public Future<OracleResponse<R>> execute(OracleConnection conn, ContextInternal context) {
    boolean returnAutoGeneratedKeys = returnAutoGeneratedKeys(conn, query.options() instanceof OraclePrepareOptions ? (OraclePrepareOptions) query.options() : null);
    return prepare(query, conn, returnAutoGeneratedKeys, context)
      .flatMap(ps -> {
        try {
          for (Tuple params : listParams) {
            fillStatement(ps, conn, params);
            ps.addBatch();
          }
        } catch (SQLException e) {
          closeQuietly(ps);
          return context.failedFuture(e);
        }

        return executeBatch(ps, context)
          .map(res ->
            Helper.getOrHandleSQLException(
              () -> decode(ps, res, returnAutoGeneratedKeys))
          )
          .onComplete(ar ->
            closeQuietly(ps)
          );

      });
  }

  public Future<int[]> executeBatch(
    PreparedStatement batchUpdateStatement, ContextInternal context) {
    OraclePreparedStatement oraclePreparedStatement =
      unwrapOraclePreparedStatement(batchUpdateStatement);

    try {
      return Helper.collect(oraclePreparedStatement.executeBatchAsyncOracle(), context)
        .map(list -> {
          int[] res = new int[list.size()];
          for (int i = 0; i < list.size(); i++) {
            res[i] = list.get(i).intValue();
          }
          return res;
        });
    } catch (SQLException throwables) {
      return context.failedFuture(throwables);
    }
  }

  private void fillStatement(PreparedStatement ps, Connection conn, Tuple params) throws SQLException {

    for (int i = 0; i < params.size(); i++) {
      // we must convert types (to comply to JDBC)
      Object value = adaptType(conn, params.getValue(i));

      ps.setObject(i + 1, value);
    }
  }

  private Object adaptType(Connection conn, Object value) throws SQLException {
    // we must convert types (to comply to JDBC)

    if (value instanceof LocalTime) {
      // -> java.sql.Time
      LocalTime time = (LocalTime) value;
      return Time.valueOf(time);
    } else if (value instanceof LocalDate) {
      // -> java.sql.Date
      LocalDate date = (LocalDate) value;
      return Date.valueOf(date);
    } else if (value instanceof Instant) {
      // -> java.sql.Timestamp
      Instant timestamp = (Instant) value;
      return Timestamp.from(timestamp);
    } else if (value instanceof Buffer) {
      // -> java.sql.Blob
      Buffer blob = (Buffer) value;
      return conn.createBlob().setBytes(0, blob.getBytes());
    }

    return value;
  }
}
