/*
 * Decompiled with CFR 0.152.
 */
package io.sysr.springcontext.env;

import io.sysr.springcontext.env.exception.EnvContextLoaderException;
import java.io.File;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EnvContextLoader {
    private static final Logger logger = LoggerFactory.getLogger(EnvContextLoader.class);
    private final ConcurrentHashMap<String, String> propertiesMap = new ConcurrentHashMap();
    private static final Pattern ENV_FILE_NAME_PATTERN = Pattern.compile("^\\.env\\.?\\w*$");
    private static final Pattern VARIABLE_PATTERN_MATCHER = Pattern.compile("\\$\\{([^}]+)}");
    private static final Pattern VARIABLE_NAME_PATTERN = Pattern.compile("^[A-Za-z_][A-Za-z0-9_-]*$");
    private static final Pattern BAD_VARIABLE_PATTERN_MATCHER = Pattern.compile("\\$\\{\\s*\\}$|\\$\\{[^}]*$");

    public EnvContextLoader() {
        logger.trace("Spring context dotenv loader initiated");
    }

    public Properties getLoadedProperties() {
        Properties props = new Properties();
        this.propertiesMap.entrySet().forEach(entry -> props.put(entry.getKey(), entry.getValue()));
        return props;
    }

    public void load() {
        try {
            String dirPath;
            this.loadEnvFilesFromDirectory(System.getProperty("user.dir"));
            String userProvidedFilePath = this.findEnvPropertiesFile();
            if (Objects.nonNull(userProvidedFilePath)) {
                this.loadEnvFilesFromUserDefinedPath(userProvidedFilePath);
            }
            if (Objects.nonNull(dirPath = System.getenv("ENV_DIR_PATH"))) {
                dirPath = dirPath.replace("\\", "\\\\");
                this.loadEnvFilesFromDirectory(dirPath);
            }
        }
        catch (Exception e) {
            throw new EnvContextLoaderException(e.getLocalizedMessage(), e);
        }
    }

    private void loadEnvFilesFromUserDefinedPath(String envPropertiesFilePath) {
        try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(Path.of(envPropertiesFilePath, new String[0]), new OpenOption[0]), StandardCharsets.UTF_8);){
            Properties props = new Properties();
            props.load(reader);
            String directoryPath = Stream.of("BASE_DIR").map(key -> props.getProperty(key.toUpperCase())).filter(Objects::nonNull).findFirst().orElse(null);
            if (Objects.nonNull(directoryPath)) {
                List<String> fileNameKeys = props.stringPropertyNames().stream().filter(key -> key.toUpperCase().startsWith("FILE")).toList();
                for (String fileNameKey : fileNameKeys) {
                    String fileName = props.getProperty(fileNameKey);
                    Path path = Path.of(directoryPath, new String[0]).normalize().resolve(fileName).normalize();
                    File file = path.toFile();
                    if (!file.exists() || !file.isFile()) continue;
                    this.parse(path);
                    logger.info("Successfully loaded properties from {}", (Object)path.toAbsolutePath());
                }
            } else {
                logger.warn("BASE_DIR not found in the properties file - {}", (Object)envPropertiesFilePath);
            }
        }
        catch (Exception e) {
            throw new EnvContextLoaderException(e.getLocalizedMessage(), e);
        }
    }

    private void loadEnvFilesFromDirectory(String filePath) {
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(Path.of(filePath, new String[0]));){
            for (Path path : directoryStream) {
                File file = path.toFile();
                if (!file.isFile() || !file.getName().matches(ENV_FILE_NAME_PATTERN.pattern())) continue;
                this.parse(path.normalize());
                logger.info("Successfully loaded properties from {}", (Object)path.toAbsolutePath());
            }
        }
        catch (Exception e) {
            throw new EnvContextLoaderException(e.getLocalizedMessage(), e);
        }
    }

    private void parse(Path path) {
        try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(path, new OpenOption[0]), StandardCharsets.UTF_8);){
            Properties props = new Properties();
            props.load(reader);
            for (String key : props.stringPropertyNames()) {
                String value = this.getResolvedValue(props, key);
                if (!Objects.nonNull(value) || value.isBlank()) continue;
                this.propertiesMap.put(key.strip(), value.strip());
            }
        }
        catch (Exception e) {
            throw new EnvContextLoaderException(e.getLocalizedMessage(), e);
        }
    }

    private String getResolvedValue(Properties props, String key) {
        HashMap<String, String> resolved = new HashMap<String, String>();
        HashSet<String> resolving = new HashSet<String>();
        ArrayDeque<String> stack = new ArrayDeque<String>();
        if (!this.isValidVariableName(key)) {
            logger.warn("The variable name: {} is considered invalid. Please double check.", (Object)key);
            return null;
        }
        stack.push(key);
        while (!stack.isEmpty()) {
            StringBuilder sb;
            String name = (String)stack.peek();
            String value = props.getProperty(name);
            if ((Objects.isNull(value) || value.isBlank()) && (Objects.isNull(value = System.getenv(name)) || value.isBlank())) {
                value = System.getProperty(name);
            }
            if (Objects.isNull(value) || value.isBlank()) {
                logger.warn("The definition of the env variable {} is not found!", (Object)name);
                return null;
            }
            Matcher badVariable = BAD_VARIABLE_PATTERN_MATCHER.matcher(value);
            if (badVariable.matches()) {
                logger.warn("The variable definition {}={} is considered invalid. Please double check.", (Object)name, (Object)value);
                return null;
            }
            resolving.add(name);
            Matcher variableMatcher = VARIABLE_PATTERN_MATCHER.matcher(value);
            boolean isResolved = this.resolve(resolving, resolved, stack, variableMatcher, sb = new StringBuilder());
            if (!isResolved) continue;
            variableMatcher.appendTail(sb);
            resolved.put(name, sb.toString());
            resolving.remove(name);
            stack.pop();
        }
        return (String)resolved.get(key);
    }

    private boolean resolve(Set<String> resolving, Map<String, String> resolved, Deque<String> stack, Matcher variableMatcher, StringBuilder sb) {
        boolean isResolved = true;
        while (variableMatcher.find()) {
            String name = variableMatcher.group(1);
            if (resolving.contains(name)) {
                throw new EnvContextLoaderException("Circular dependency detected on variable %s.".formatted(name));
            }
            if (resolved.containsKey(name)) {
                variableMatcher.appendReplacement(sb, resolved.get(name));
                continue;
            }
            if (this.propertiesMap.containsKey(name)) {
                variableMatcher.appendReplacement(sb, this.propertiesMap.get(name.strip()));
                continue;
            }
            stack.push(name);
            isResolved = false;
        }
        return isResolved;
    }

    private String findEnvPropertiesFile() throws URISyntaxException {
        URL resourceUrl = this.getClass().getClassLoader().getResource("dotenv.properties");
        if (Objects.nonNull(resourceUrl)) {
            Path path = Path.of(resourceUrl.toURI());
            return path.toAbsolutePath().toString();
        }
        logger.warn("dotenv.properties not found in the classpath");
        return null;
    }

    private boolean isValidVariableName(String variableName) {
        Matcher matcher = VARIABLE_NAME_PATTERN.matcher(variableName);
        return matcher.matches();
    }
}

