/*
 * Copyright © 2022-2023 Apple Inc. and the ServiceTalk project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.servicetalk.http.api;

import io.servicetalk.buffer.api.BufferAllocator;
import io.servicetalk.concurrent.api.Executor;
import io.servicetalk.concurrent.api.Single;
import io.servicetalk.logging.api.LogLevel;
import io.servicetalk.transport.api.ConnectionAcceptorFactory;
import io.servicetalk.transport.api.EarlyConnectionAcceptor;
import io.servicetalk.transport.api.IoExecutor;
import io.servicetalk.transport.api.LateConnectionAcceptor;
import io.servicetalk.transport.api.ServerSslConfig;
import io.servicetalk.transport.api.TransportConfig;
import io.servicetalk.transport.api.TransportObserver;

import java.net.SocketOption;
import java.time.Duration;
import java.util.Map;
import java.util.function.BooleanSupplier;
import java.util.function.Predicate;

import static java.util.Objects.requireNonNull;

/**
 * A {@link HttpServerBuilder} that delegates all methods to another {@link HttpServerBuilder}.
 */
public class DelegatingHttpServerBuilder implements HttpServerBuilder {

    private HttpServerBuilder delegate;

    /**
     * Create a new instance.
     *
     * @param delegate {@link HttpServerBuilder} to which all methods are delegated.
     */
    public DelegatingHttpServerBuilder(final HttpServerBuilder delegate) {
        this.delegate = requireNonNull(delegate);
    }

    /**
     * Returns the {@link HttpServerBuilder} delegate.
     *
     * @return Delegate {@link HttpServerBuilder}.
     */
    protected final HttpServerBuilder delegate() {
        return delegate;
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "{delegate=" + delegate() + '}';
    }

    @Override
    public HttpServerBuilder protocols(final HttpProtocolConfig... protocols) {
        delegate = delegate.protocols(protocols);
        return this;
    }

    @Override
    public HttpServerBuilder sslConfig(final ServerSslConfig config) {
        delegate = delegate.sslConfig(config);
        return this;
    }

    @Override
    public HttpServerBuilder sslConfig(final ServerSslConfig defaultConfig, final Map<String, ServerSslConfig> sniMap) {
        delegate = delegate.sslConfig(defaultConfig, sniMap);
        return this;
    }

    @Override
    public HttpServerBuilder sslConfig(final ServerSslConfig defaultConfig, final Map<String, ServerSslConfig> sniMap,
                                       final int maxClientHelloLength, final Duration clientHelloTimeout) {
        delegate = delegate.sslConfig(defaultConfig, sniMap, maxClientHelloLength, clientHelloTimeout);
        return this;
    }

    @Override
    public HttpServerBuilder sslConfig(final ServerSslConfig config, final boolean acceptInsecureConnections) {
        delegate = delegate.sslConfig(config, acceptInsecureConnections);
        return this;
    }

    @Override
    public HttpServerBuilder transportConfig(final TransportConfig transportConfig) {
        delegate = delegate.transportConfig(transportConfig);
        return this;
    }

    @Override
    public <T> HttpServerBuilder socketOption(final SocketOption<T> option, final T value) {
        delegate = delegate.socketOption(option, value);
        return this;
    }

    @Override
    public <T> HttpServerBuilder listenSocketOption(final SocketOption<T> option, final T value) {
        delegate = delegate.listenSocketOption(option, value);
        return this;
    }

    @Override
    public HttpServerBuilder enableWireLogging(final String loggerName, final LogLevel logLevel,
                                               final BooleanSupplier logUserData) {
        delegate = delegate.enableWireLogging(loggerName, logLevel, logUserData);
        return this;
    }

    @Override
    public HttpServerBuilder transportObserver(final TransportObserver transportObserver) {
        delegate = delegate.transportObserver(transportObserver);
        return this;
    }

    @Override
    public HttpServerBuilder lifecycleObserver(final HttpLifecycleObserver lifecycleObserver) {
        delegate = delegate.lifecycleObserver(lifecycleObserver);
        return this;
    }

    @Override
    public HttpServerBuilder drainRequestPayloadBody(final boolean enable) {
        delegate = delegate.drainRequestPayloadBody(enable);
        return this;
    }

    @Override
    public HttpServerBuilder allowDropRequestTrailers(final boolean allowDrop) {
        delegate = delegate.allowDropRequestTrailers(allowDrop);
        return this;
    }

    @Override
    public HttpServerBuilder appendConnectionAcceptorFilter(final ConnectionAcceptorFactory factory) {
        delegate = delegate.appendConnectionAcceptorFilter(factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendEarlyConnectionAcceptor(final EarlyConnectionAcceptor factory) {
        delegate = delegate.appendEarlyConnectionAcceptor(factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendLateConnectionAcceptor(final LateConnectionAcceptor factory) {
        delegate = delegate.appendLateConnectionAcceptor(factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendNonOffloadingServiceFilter(final StreamingHttpServiceFilterFactory factory) {
        delegate = delegate.appendNonOffloadingServiceFilter(factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendNonOffloadingServiceFilter(final Predicate<StreamingHttpRequest> predicate,
                                                              final StreamingHttpServiceFilterFactory factory) {
        delegate = delegate.appendNonOffloadingServiceFilter(predicate, factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendServiceFilter(final StreamingHttpServiceFilterFactory factory) {
        delegate = delegate.appendServiceFilter(factory);
        return this;
    }

    @Override
    public HttpServerBuilder appendServiceFilter(final Predicate<StreamingHttpRequest> predicate,
                                                 final StreamingHttpServiceFilterFactory factory) {
        delegate = delegate.appendServiceFilter(predicate, factory);
        return this;
    }

    @Override
    public HttpServerBuilder ioExecutor(final IoExecutor ioExecutor) {
        delegate = delegate.ioExecutor(ioExecutor);
        return this;
    }

    @Override
    public HttpServerBuilder executor(final Executor executor) {
        delegate = delegate.executor(executor);
        return this;
    }

    @Override
    public HttpServerBuilder bufferAllocator(final BufferAllocator allocator) {
        delegate = delegate.bufferAllocator(allocator);
        return this;
    }

    @Override
    public HttpServerBuilder executionStrategy(final HttpExecutionStrategy strategy) {
        delegate = delegate.executionStrategy(strategy);
        return this;
    }

    @Override
    public Single<HttpServerContext> listenService(final HttpServiceBase service) {
        return delegate.listenService(service);
    }

    @Override
    public Single<HttpServerContext> listen(final HttpService service) {
        return delegate.listen(service);
    }

    @Override
    public Single<HttpServerContext> listenStreaming(final StreamingHttpService service) {
        return delegate.listenStreaming(service);
    }

    @Override
    public Single<HttpServerContext> listenBlocking(final BlockingHttpService service) {
        return delegate.listenBlocking(service);
    }

    @Override
    public Single<HttpServerContext> listenBlockingStreaming(final BlockingStreamingHttpService service) {
        return delegate.listenBlockingStreaming(service);
    }

    @Override
    public HttpServerContext listenServiceAndAwait(final HttpServiceBase service) throws Exception {
        return delegate.listenServiceAndAwait(service);
    }

    @Override
    public HttpServerContext listenAndAwait(final HttpService service) throws Exception {
        return delegate.listenAndAwait(service);
    }

    @Override
    public HttpServerContext listenStreamingAndAwait(final StreamingHttpService service) throws Exception {
        return delegate.listenStreamingAndAwait(service);
    }

    @Override
    public HttpServerContext listenBlockingAndAwait(final BlockingHttpService service) throws Exception {
        return delegate.listenBlockingAndAwait(service);
    }

    @Override
    public HttpServerContext listenBlockingStreamingAndAwait(final BlockingStreamingHttpService service)
            throws Exception {
        return delegate.listenBlockingStreamingAndAwait(service);
    }
}
