package io.sendon.sms;

import java.io.File;
import java.util.List;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.sms.response.CancelGroup;
import io.sendon.sms.response.CancelRepeat;
import io.sendon.sms.response.GenerateCsvPresignedUrl;
import io.sendon.sms.response.GetGroup;
import io.sendon.sms.response.GetImage;
import io.sendon.sms.response.GetRepeats;
import io.sendon.sms.response.SendGroup;
import io.sendon.sms.response.SendRepeat;
import io.sendon.sms.response.SendSms;
import io.sendon.sms.response.UploadImage;

public class SendonSms extends SendonClient {

  public enum MessageType {
    SMS("SMS"),
    LMS("LMS"),
    MMS("MMS");

    public final String messageType;

    MessageType(String messageType) {
      this.messageType = messageType;
    }
  }

  public enum RepeatUnit {
    MINUTE("MINUTE"),
    HOUR("HOUR"),
    DAY("DAY");

    public final String repeatUnit;

    RepeatUnit(String repeatUnit) {
      this.repeatUnit = repeatUnit;
    }
  }

  public static class Repeat {
    int messageCount; // limit
    RepeatUnit unit;
    int interval;
    boolean denyNightTime;
  }

  public static class Reservation {
    String datetime;
    Repeat repeat;
  }

  public static class ToWithName {
    String phone;
    String name;
  }

  public static class ToWithBucket {
    String bucket;
    String key;
  }

  public SendonSms(String userId, String apiKey) {
    super(userId, apiKey);
  }

  public SendonSms(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  public SendSms sendSms(String from, List<String> to, String message, boolean isAd, Reservation reservation) {
    JsonObject bodyJson = new JsonObject();
    bodyJson.addProperty("type", MessageType.SMS.messageType);
    bodyJson.addProperty("from", from);

    JsonArray toArray = new JsonArray();
    for (String recipient : to) toArray.add(recipient);

    bodyJson.add("to", toArray);
    bodyJson.addProperty("message", message);
    bodyJson.addProperty("isAd", isAd);

    if (reservation != null) {
      JsonObject reservationJson = new JsonObject();
      reservationJson.addProperty("datetime", reservation.datetime);

      if (reservation.repeat != null) {
        JsonObject repeatJson = new JsonObject();
        repeatJson.addProperty("limit", reservation.repeat.messageCount);
        repeatJson.addProperty("unit", reservation.repeat.unit.repeatUnit);
        repeatJson.addProperty("interval", reservation.repeat.interval);
        repeatJson.addProperty("denyNightTime", reservation.repeat.denyNightTime);
        reservationJson.add("repeat", repeatJson);
      }

      bodyJson.add("reservation", reservationJson);
    }

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms", bodyJson.toString()));
      return new SendSms(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public SendSms sendSmsWithName(String from, List<ToWithName> to, String message, boolean isAd, Reservation reservation) {
    JsonObject bodyJson = new JsonObject();
    bodyJson.addProperty("type", MessageType.SMS.messageType);
    bodyJson.addProperty("from", from);

    JsonArray toArray = new JsonArray();
    for (ToWithName recipient : to) {
      JsonObject toObject = new JsonObject();
      toObject.addProperty("phone", recipient.phone);
      toObject.addProperty("name", recipient.name);
      toArray.add(toObject);
    }

    bodyJson.add("to", toArray);
    bodyJson.addProperty("message", message);
    bodyJson.addProperty("isAd", isAd);

    if (reservation != null) {
      JsonObject reservationJson = new JsonObject();
      reservationJson.addProperty("datetime", reservation.datetime);

      if (reservation.repeat != null) {
        JsonObject repeatJson = new JsonObject();
        repeatJson.addProperty("limit", reservation.repeat.messageCount);
        repeatJson.addProperty("unit", reservation.repeat.unit.repeatUnit);
        repeatJson.addProperty("interval", reservation.repeat.interval);
        repeatJson.addProperty("denyNightTime", reservation.repeat.denyNightTime);
        reservationJson.add("repeat", repeatJson);
      }

      bodyJson.add("reservation", reservationJson);
    }

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms", bodyJson.toString()));
      return new SendSms(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public SendSms sendSmWithBucket(String from, List<ToWithBucket> to, String message, Reservation reservation) {
    JsonObject bodyJson = new JsonObject();
    bodyJson.addProperty("type", MessageType.SMS.messageType);
    bodyJson.addProperty("from", from);

    JsonArray toArray = new JsonArray();

    for (ToWithBucket recipient : to) {
      JsonObject toObject = new JsonObject();
      toObject.addProperty("bucket", recipient.bucket);
      toObject.addProperty("key", recipient.key);
      toArray.add(toObject);
    }

    bodyJson.add("to", toArray);
    bodyJson.addProperty("message", message);
    bodyJson.addProperty("isAd", false);

    if (reservation != null) {
      JsonObject reservationJson = new JsonObject();
      reservationJson.addProperty("datetime", reservation.datetime);

      if (reservation.repeat != null) {
        JsonObject repeatJson = new JsonObject();
        repeatJson.addProperty("limit", reservation.repeat.messageCount);
        repeatJson.addProperty("unit", reservation.repeat.unit.repeatUnit);
        repeatJson.addProperty("interval", reservation.repeat.interval);
        repeatJson.addProperty("denyNightTime", reservation.repeat.denyNightTime);
        reservationJson.add("repeat", repeatJson);
      }

      bodyJson.add("reservation", reservationJson);
    }

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms", bodyJson.toString()));
      return new SendSms(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GetGroup getGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/groups/" + groupId));
      return new GetGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public SendGroup sendGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/send", null));
      return new SendGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;

  }

  public CancelGroup cancelGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/cancel", null));
      return new CancelGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GetRepeats getRepeats(String groupId, int limit, String cursor) {
    if (cursor == null) cursor = "";

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/groups/" + groupId + "/repeats?limit=" + limit + "&cursor=" + cursor));
      return new GetRepeats(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public SendRepeat sendRepeat(String groupId, String repeatId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/repeats/" + repeatId + "/send", null));
      return new SendRepeat(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public CancelRepeat cancelRepeat(String groupId, String repeatId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/repeats/" + repeatId + "/cancel", null));
      return new CancelRepeat(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GenerateCsvPresignedUrl generateCsvPresignedUrl() {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/csv/presigned", null));
      return new GenerateCsvPresignedUrl(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public UploadImage uploadImages(List<File> images) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(postImagesWithMultipartFormData("/v2/messages/sms/image/upload", images));
      return new UploadImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GetImage getImages(List<String> images) {
    try {
      int index = 1;
      String query = "";
      for(String image : images) {
        query += "imageId" + index + "=" + image + "&";
        index++;
      }
      query+="&expireIn=60";

      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/image/url?" + query));
      return new GetImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }


}