package io.sendon.base;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.nio.file.Files;
import java.util.List;
import java.util.UUID;

public class ApacheHttpClient extends HttpClient {
  public ApacheHttpClient(String baseUrl, String id, String pw) {
    super(baseUrl, id, pw);
  }

  @Override
  public String post(String endpoint, String body) {
    return sendRequest("POST", endpoint, body);
  }

  @Override
  public String put(String endpoint, String body) {
    return sendRequest("PUT", endpoint, body);
  }

  @Override
  public String delete(String endpoint) {
    return sendRequest("DELETE", endpoint, null);
  }

  @Override
  public String get(String endpoint) {
    return sendRequest("GET", endpoint, null);
  }

  @Override
  public String sendRequest(String method, String endpoint, String body) {
    try {
      URL url = new URL(baseUrl + endpoint);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setConnectTimeout(30*1000);
      conn.setReadTimeout(30*1000);
      conn.setRequestMethod(method);
      conn.setRequestProperty(AUTH_HEADER, authBasic);
      conn.setRequestProperty(CONTENT_TYPE_HEADER, APPLICATION_JSON);
      conn.setDoOutput(body != null);

      if (body != null) {
        try (OutputStream os = conn.getOutputStream()) {
          os.write(body.getBytes());
          os.flush();
        }
      }

      int responseCode = conn.getResponseCode();
      InputStream responseStream = responseCode >= 400 ? conn.getErrorStream() : conn.getInputStream();

      if (responseStream != null) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(responseStream))) {
          return reader.lines().reduce("", (a, b) -> a + b);
        }
      } else {
        return "";
      }

    } catch (MalformedURLException e) {
      e.printStackTrace();
    } catch (ProtocolException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return null;
  }

  @Override
  public String postImagesWithMultipartFormData(String path, List<File> images) {
    Log.d("postImagesWithMultipartFormData: " + path);

    String boundary = UUID.randomUUID().toString();
    String LINE_FEED = "\r\n";

    try {
      URL url = new URL(baseUrl + path);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setConnectTimeout(30 * 1000);
      conn.setReadTimeout(30 * 1000);
      conn.setRequestMethod("POST");
      conn.setRequestProperty(AUTH_HEADER, authBasic);
      conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
      conn.setDoOutput(true);

      try (OutputStream os = conn.getOutputStream()) {
        for (File image : images) {
          String contentType = "image/png";
          StringBuilder sb = new StringBuilder();
          sb.append("--").append(boundary).append(LINE_FEED);
          sb.append("Content-Disposition: form-data; name=\"images\"; filename=\"")
            .append(image.getName()).append("\"").append(LINE_FEED);
          sb.append("Content-Type: ").append(contentType).append(LINE_FEED);
          sb.append(LINE_FEED);
          os.write(sb.toString().getBytes());
          Files.copy(image.toPath(), os);
          os.write(LINE_FEED.getBytes());
        }
        os.write(("--" + boundary + "--").getBytes());
        os.write(LINE_FEED.getBytes());
        os.flush();
      }

      int responseCode = conn.getResponseCode();
      InputStream responseStream = responseCode >= 400 ? conn.getErrorStream() : conn.getInputStream();

      if (responseStream != null) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(responseStream))) {
          return reader.lines().reduce("", (a, b) -> a + b);
        }
      } else {
        return "";
      }

    } catch (MalformedURLException e) {
      e.printStackTrace();
    } catch (ProtocolException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return null;
  }

  @Override
  public String postImageWithMulStringFormData(String path, File image) {
    Log.d("postImageWithMulStringFormData: " + path);

    String boundary = UUID.randomUUID().toString();
    String LINE_FEED = "\r\n";

    try {
      URL url = new URL(baseUrl + path);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setConnectTimeout(30 * 1000);
      conn.setReadTimeout(30 * 1000);
      conn.setRequestMethod("POST");
      conn.setRequestProperty(AUTH_HEADER, authBasic);
      conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
      conn.setDoOutput(true);

      try (OutputStream os = conn.getOutputStream()) {
        StringBuilder sb = new StringBuilder();
        sb.append("--").append(boundary).append(LINE_FEED);
        sb.append("Content-Disposition: form-data; name=\"image\"; filename=\"")
          .append(image.getName()).append("\"").append(LINE_FEED);
        sb.append("Content-Type: ").append("image/png").append(LINE_FEED);
        sb.append(LINE_FEED);
        os.write(sb.toString().getBytes());
        Files.copy(image.toPath(), os);
        os.write(LINE_FEED.getBytes());
        os.write(("--" + boundary + "--").getBytes());
        os.write(LINE_FEED.getBytes());
        os.flush();
      }

      int responseCode = conn.getResponseCode();
      InputStream responseStream = responseCode >= 400 ? conn.getErrorStream() : conn.getInputStream();

      if (responseStream != null) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(responseStream))) {
          return reader.lines().reduce("", (a, b) -> a + b);
        }
      } else {
        return "";
      }

    } catch (MalformedURLException e) {
      e.printStackTrace();
    } catch (ProtocolException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return null;
  }
}