/*
 * Decompiled with CFR 0.152.
 */
package io.scalecube.transport;

import io.scalecube.transport.Address;
import io.scalecube.transport.Message;
import io.scalecube.transport.NetworkEmulatorException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public final class NetworkEmulator {
    private static final Logger LOGGER = LoggerFactory.getLogger(NetworkEmulator.class);
    private volatile OutboundSettings defaultOutboundSettings = new OutboundSettings(0, 0);
    private volatile InboundSettings defaultInboundSettings = new InboundSettings(true);
    private final Map<Address, OutboundSettings> outboundSettings = new ConcurrentHashMap<Address, OutboundSettings>();
    private final Map<Address, InboundSettings> inboundSettings = new ConcurrentHashMap<Address, InboundSettings>();
    private final AtomicLong totalMessageSentCount = new AtomicLong();
    private final AtomicLong totalOutboundMessageLostCount = new AtomicLong();
    private final AtomicLong totalInboundMessageLostCount = new AtomicLong();
    private final boolean enabled;
    private final Address address;

    NetworkEmulator(Address address, boolean enabled) {
        this.address = address;
        this.enabled = enabled;
    }

    public OutboundSettings outboundSettings(Address destination) {
        return this.outboundSettings.getOrDefault(destination, this.defaultOutboundSettings);
    }

    public void outboundSettings(Address destination, int lossPercent, int meanDelay) {
        if (!this.enabled) {
            return;
        }
        OutboundSettings settings = new OutboundSettings(lossPercent, meanDelay);
        this.outboundSettings.put(destination, settings);
        LOGGER.debug("Set outbound settings {} from {} to {}", new Object[]{settings, this.address, destination});
    }

    public void setDefaultOutboundSettings(int lossPercent, int meanDelay) {
        if (!this.enabled) {
            return;
        }
        this.defaultOutboundSettings = new OutboundSettings(lossPercent, meanDelay);
        LOGGER.debug("Set default outbound settings {} for {}", (Object)this.defaultOutboundSettings, (Object)this.address);
    }

    public void blockAllOutbound() {
        this.setDefaultOutboundSettings(100, 0);
    }

    public void blockOutbound(Address ... destinations) {
        this.blockOutbound(Arrays.asList(destinations));
    }

    public void blockOutbound(Collection<Address> destinations) {
        if (!this.enabled) {
            return;
        }
        for (Address destination : destinations) {
            this.outboundSettings.put(destination, new OutboundSettings(100, 0));
        }
        LOGGER.debug("Blocked outbound from {} to {}", (Object)this.address, destinations);
    }

    public void unblockOutbound(Address ... destinations) {
        this.unblockOutbound(Arrays.asList(destinations));
    }

    public void unblockOutbound(Collection<Address> destinations) {
        if (!this.enabled) {
            return;
        }
        destinations.forEach(this.outboundSettings::remove);
        LOGGER.debug("Unblocked outbound from {} to {}", (Object)this.address, destinations);
    }

    public void unblockAllOutbound() {
        if (!this.enabled) {
            return;
        }
        this.outboundSettings.clear();
        this.setDefaultOutboundSettings(0, 0);
        LOGGER.debug("Unblocked outbound from {} to all destinations", (Object)this.address);
    }

    public long totalMessageSentCount() {
        if (!this.enabled) {
            return 0L;
        }
        return this.totalMessageSentCount.get();
    }

    public long totalOutboundMessageLostCount() {
        if (!this.enabled) {
            return 0L;
        }
        return this.totalOutboundMessageLostCount.get();
    }

    public Mono<Message> tryFailOutbound(Message msg, Address address) {
        return Mono.defer(() -> {
            if (!this.enabled) {
                return Mono.just((Object)msg);
            }
            this.totalMessageSentCount.incrementAndGet();
            boolean isLost = this.outboundSettings(address).evaluateLoss();
            if (isLost) {
                this.totalOutboundMessageLostCount.incrementAndGet();
                return Mono.error((Throwable)new NetworkEmulatorException("NETWORK_BREAK detected, didn't send " + msg));
            }
            return Mono.just((Object)msg);
        });
    }

    public Mono<Message> tryDelayOutbound(Message msg, Address address) {
        return Mono.defer(() -> {
            if (!this.enabled) {
                return Mono.just((Object)msg);
            }
            this.totalMessageSentCount.incrementAndGet();
            int delay = (int)this.outboundSettings(address).evaluateDelay();
            if (delay > 0) {
                return Mono.just((Object)msg).delayElement(Duration.ofMillis(delay));
            }
            return Mono.just((Object)msg);
        });
    }

    public InboundSettings inboundSettings(Address destination) {
        return this.inboundSettings.getOrDefault(destination, this.defaultInboundSettings);
    }

    public void inboundSettings(Address destination, boolean shallPass) {
        if (!this.enabled) {
            return;
        }
        InboundSettings settings = new InboundSettings(shallPass);
        this.inboundSettings.put(destination, settings);
        LOGGER.debug("Set inbound settings {} from {} to {}", new Object[]{settings, this.address, destination});
    }

    public void setDefaultInboundSettings(boolean shallPass) {
        if (!this.enabled) {
            return;
        }
        this.defaultInboundSettings = new InboundSettings(shallPass);
        LOGGER.debug("Set default inbound settings {} for {}", (Object)this.defaultInboundSettings, (Object)this.address);
    }

    public void blockAllInbound() {
        this.setDefaultInboundSettings(false);
    }

    public void blockInbound(Address ... destinations) {
        this.blockInbound(Arrays.asList(destinations));
    }

    public void blockInbound(Collection<Address> destinations) {
        if (!this.enabled) {
            return;
        }
        for (Address destination : destinations) {
            this.inboundSettings.put(destination, new InboundSettings(false));
        }
        LOGGER.debug("Blocked inbound from {} to {}", (Object)this.address, destinations);
    }

    public void unblockInbound(Address ... destinations) {
        this.unblockInbound(Arrays.asList(destinations));
    }

    public void unblockInbound(Collection<Address> destinations) {
        if (!this.enabled) {
            return;
        }
        destinations.forEach(this.inboundSettings::remove);
        LOGGER.debug("Unblocked inbound from {} to {}", (Object)this.address, destinations);
    }

    public void unblockAllInbound() {
        if (!this.enabled) {
            return;
        }
        this.inboundSettings.clear();
        this.setDefaultInboundSettings(true);
        LOGGER.debug("Unblocked inbound from {} to all destinations", (Object)this.address);
    }

    public long totalInboundMessageLostCount() {
        if (!this.enabled) {
            return 0L;
        }
        return this.totalInboundMessageLostCount.get();
    }

    public static class InboundSettings {
        private final boolean shallPass;

        public InboundSettings(boolean shallPass) {
            this.shallPass = shallPass;
        }

        public boolean shallPass() {
            return this.shallPass;
        }

        public String toString() {
            return "InboundSettings{shallPass=" + this.shallPass + '}';
        }
    }

    public static final class OutboundSettings {
        private final int lossPercent;
        private final int meanDelay;

        public OutboundSettings(int lossPercent, int meanDelay) {
            this.lossPercent = lossPercent;
            this.meanDelay = meanDelay;
        }

        public int lossPercent() {
            return this.lossPercent;
        }

        public int meanDelay() {
            return this.meanDelay;
        }

        public boolean evaluateLoss() {
            return this.lossPercent > 0 && (this.lossPercent >= 100 || ThreadLocalRandom.current().nextInt(100) < this.lossPercent);
        }

        public long evaluateDelay() {
            if (this.meanDelay > 0) {
                double x0 = ThreadLocalRandom.current().nextDouble();
                double y0 = -Math.log(1.0 - x0) * (double)this.meanDelay;
                return (long)y0;
            }
            return 0L;
        }

        public String toString() {
            return "OutboundSettings{loss=" + this.lossPercent + ", delay=" + this.meanDelay + '}';
        }
    }
}

