package io.scalecube.docker.utils;

import static com.google.common.base.Preconditions.checkArgument;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.command.CreateNetworkResponse;
import com.github.dockerjava.api.command.InspectExecResponse;
import com.github.dockerjava.api.exception.DockerClientException;
import com.github.dockerjava.api.exception.NotModifiedException;
import com.github.dockerjava.api.model.ContainerNetwork;
import com.github.dockerjava.api.model.Frame;
import com.github.dockerjava.api.model.Network;
import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.async.ResultCallbackTemplate;
import com.github.dockerjava.core.command.ExecStartResultCallback;
import com.github.dockerjava.core.command.LogContainerResultCallback;
import com.github.dockerjava.core.command.PullImageResultCallback;
import com.github.dockerjava.core.command.WaitContainerResultCallback;
import com.github.dockerjava.netty.NettyDockerCmdExecFactory;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.io.ByteStreams;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public final class Containers {
  public static final DateTimeFormatter UTC_ISO_DATE_TIME_FORMATTER =
      ISODateTimeFormat.dateTime().withZoneUTC();
  public static final DateTimeFormatter UTC_LOG_CFG_DATE_TIME_FORMATTER =
      DateTimeFormat.forPattern("MMdd-HHmm:ss,SSS").withZoneUTC();
  private static final Logger LOGGER = LoggerFactory.getLogger(Containers.class);
  private static final long SHORT_TIMEOUT = 5;
  private static final long LONG_TIMEOUT = 30;

  private static final int IMAGE_ID_LENGTH = 12;

  private static DockerClient dockerClient;

  static {
    initClient();
  }

  private Containers() {}

  /**
   * Call this in case when it's needed to reinitialize docker-machine client For example, if you previously closed the
   * client calling {@code closeClient()}
   */
  public static void initClient() {
    dockerClient =
        DockerClientBuilder.getInstance()
            .withDockerCmdExecFactory(new NettyDockerCmdExecFactory())
            .build();
  }

  /**
   * Shuts down the docker-machine client
   *
   * @throws IOException
   */
  public static void closeClient() throws IOException {
    dockerClient.close();
  }

  /** Shuts down the docker-machine client quietly (logging exception if any) */
  public static void closeClientQuietly() {
    try {
      closeClient();
    } catch (Exception e) {
      LOGGER.error("Got exception at closing docker client, cause: {}", e);
    }
  }

  public static String createNetwork(String subnet) {
    CreateNetworkResponse createNetworkResponse =
        dockerClient
            .createNetworkCmd()
            .withName(subnet)
            .withCheckDuplicate(true)
            .withIpam(new Network.Ipam().withConfig(new Network.Ipam.Config().withSubnet(subnet)))
            .exec();
    return createNetworkResponse.getId();
  }

  public static String getNetworkAsString(String networkId) {
    Network network = dockerClient.inspectNetworkCmd().withNetworkId(networkId).exec();
    return network.toString();
  }

  public static void removeNetwork(String network) {
    if (network != null) {
      dockerClient.removeNetworkCmd(network).exec();
    }
  }

  public static void removeNetworkQuietly(String network) {
    try {
      removeNetwork(network);
    } catch (Exception e) {
      LOGGER.error("Got exception at removing network: {}, cause: {}", network, e);
    }
  }

  public static String execCommand(Container container, String... cmd) {
    return execCommand(container, SHORT_TIMEOUT, TimeUnit.SECONDS, cmd);
  }

  public static String execCommand(
      Container container, long timeout, TimeUnit timeUnit, String... cmd) {
    checkArgument(cmd.length > 0, "at least 1 cmd token is required");
    try {
      String execId =
          dockerClient
              .execCreateCmd(container.getContainerName())
              .withCmd(cmd)
              .withAttachStdout(true)
              .withAttachStderr(true)
              .exec()
              .getId();

      ExecResultOutputMatchCallback callback = new ExecResultOutputMatchCallback();
      boolean completion =
          dockerClient.execStartCmd(execId).exec(callback).awaitCompletion(timeout, timeUnit);

      InspectExecResponse inspectExec = dockerClient.inspectExecCmd(execId).exec();
      LOGGER.debug("Issued cmd: {}, execStatus: {}", Arrays.toString(cmd), inspectExec);

      if (inspectExec.isRunning() != null
          && !inspectExec.isRunning()
          && inspectExec.getExitCode() != 0) {
        throw new DockerClientException(
            "Cmd: " + Arrays.toString(cmd) + " returned exit code " + inspectExec.getExitCode());
      }
      if (!completion) {
        throw new DockerClientException(
            "Timeout, cmd: " + Arrays.toString(cmd) + " didn't complete in " + timeout + "sec");
      }
      if (!callback.serrResult.isEmpty()) {
        throw new DockerClientException(
            "Cmd: " + Arrays.toString(cmd) + " produced stderr: " + callback.serrAsString());
      }

      LOGGER.debug("Issued cmd: {}, stdout: {}", Arrays.toString(cmd), callback.soutAsString());
      return callback.soutAsString();
    } catch (Exception e) {
      throw Throwables.propagate(e);
    }
  }

  public static void runContainer(Container container, String... env) {
    String imageId;
    String imageName = container.getImageName();
    String registryHost = container.getRegistryHost();

    if (!Strings.isNullOrEmpty(registryHost)) {
      imageId = pullImage(registryHost, imageName, container.getTag());
    } else {
      imageId = getImage(imageName);
    }

    dockerClient
        .createContainerCmd(imageId)
        .withNetworkMode(container.getNetwork())
        .withIpv4Address(container.getIpAddr())
        .withName(container.getContainerName())
        .withEnv(env)
        .withPrivileged(true)
        .exec();

    dockerClient.startContainerCmd(container.getContainerName()).exec();
  }

  public static void removeContainer(Container container) {
    int stopTimeoutSec = (int) SHORT_TIMEOUT;
    removeContainer(container, stopTimeoutSec, stopTimeoutSec, TimeUnit.SECONDS);
  }

  public static void removeContainer(
      Container container, int stopTimeoutSec, long waitTimeout, TimeUnit timeUnit) {
    if (container != null) {
      try {
        dockerClient
            .stopContainerCmd(container.getContainerName())
            .withTimeout(stopTimeoutSec)
            .exec();
      } catch (NotModifiedException ignore) {
        // don't care if container already stopped
      }

      try {
        WaitContainerResultCallback resultCallback = new WaitContainerResultCallback();
        dockerClient
            .waitContainerCmd(container.getContainerName())
            .exec(resultCallback)
            .awaitStatusCode(waitTimeout, timeUnit);
      } catch (Exception e) {
        LOGGER.error("Failed when wait container for exit, cause: {}", e);
      }

      String containerLogPath = container.getContainerLogPath();
      String containerLogHostPath = container.getContainerLogHostPath();
      if (!Strings.isNullOrEmpty(containerLogPath)
          && !Strings.isNullOrEmpty(containerLogHostPath)) {
        try (InputStream inputStream =
            dockerClient
                .copyArchiveFromContainerCmd(container.getContainerName(), containerLogPath)
                .withHostPath(containerLogHostPath)
                .exec()) {
          // process stream and copy log file from container
          copyContainerLog(inputStream, containerLogHostPath);
        } catch (Exception e) {
          LOGGER.error(
              "Can't copy container logs. Container: {}, container path: {}, host path: {}, cause: {}",
              container.getContainerName(),
              containerLogPath,
              containerLogHostPath,
              e);
        }
      }

      // remove container and its volume(s)
      dockerClient
          .removeContainerCmd(container.getContainerName())
          .withForce(true)
          .withRemoveVolumes(true)
          .exec();
    }
  }

  public static void restartContainer(Container container) {
    restartContainer(container, (int) SHORT_TIMEOUT);
  }

  public static void restartContainer(Container container, int timeoutSec) {
    dockerClient.restartContainerCmd(container.getContainerName()).withtTimeout(timeoutSec).exec();
  }

  /**
   * @deprecated use {@code long grepLog(String containerId, long sinceMillis, String... tokens)} instead
   */
  public static long grepLog(Container container, long sinceMillis, String... tokens) {
    return grepLog(container.getContainerName(), sinceMillis, LONG_TIMEOUT, TimeUnit.SECONDS, tokens);
  }

  /**
   * @param containerId container id.
   * @param sinceMillis UTC millis to filter logs; it's mandatory for outputing log-entries since only that timestamp.
   * @param tokens tokens to match; will be quoted and joined with together '.*'.
   * @return millis corresponding to matched log line.
   */
  public static long grepLog(String containerId, long sinceMillis, String... tokens) {
    return grepLog(containerId, sinceMillis, LONG_TIMEOUT, TimeUnit.SECONDS, tokens);
  }

  /**
   * @deprecated Use
   *             {@code long grepLog(String containerId, long sinceMillis, long timeout, TimeUnit timeUnit, String... tokens)}
   *             instead
   */
  public static long grepLog(Container container, long sinceMillis, long timeout, TimeUnit timeUnit, String... tokens) {
    return grepLog(container.getContainerName(), sinceMillis, timeout, timeUnit, tokens);
  }

  /**
   * @param containerId container id.
   * @param sinceMillis UTC millis to filter logs; it's mandatory for outputing log-entries since only that timestamp.
   * @param timeout timeout to wait for operation completion.
   * @param timeUnit timeunit for {@code timeout}.
   * @param tokens tokens to match; will be quoted and joined with together '.*'.
   * @return millis corresponding to matched log line.
   */
  public static long grepLog(String containerId, long sinceMillis, long timeout, TimeUnit timeUnit, String... tokens) {
    String matchedRecord = grepLog0(containerId, sinceMillis, timeout, timeUnit, tokens);
    LOGGER.debug("Matched: {}, at container: {}", matchedRecord, containerId);
    return getLogRecordMillis(matchedRecord);
  }

  public static String grepTokenStr(String containerId, String... tokens) {
    return grepTokenStr(containerId, 0l, LONG_TIMEOUT, TimeUnit.SECONDS, tokens);
  }

  /**
   * @param containerId container id.
   * @param timeout timeout to wait for operation completion.
   * @param timeUnit timeunit for {@code timeout}.
   * @param tokens tokens to match; will be quoted and joined with together '.*'.
   * @return millis corresponding to matched log line.
   */
  public static String grepTokenStr(String containerId, long since, long timeout, TimeUnit timeUnit, String... tokens) {
    String matchedRecord = grepLog0(containerId, since, timeout, timeUnit, tokens);
    LOGGER.debug("Matched: {}, at container: {}", matchedRecord, containerId);
    return matchedRecord;
  }

  /*
   * Will return matched log record
   */
  private static String grepLog0(String containerId, long sinceMillis, long timeout,
      TimeUnit timeUnit, String... tokens) {
    checkArgument(tokens.length > 0, "at least 1 search token is required");
    ContainerLogInputMatchCallback callback = new ContainerLogInputMatchCallback(tokens, sinceMillis);
    int unixTime = (int) (sinceMillis / 1000);
    try {
      boolean completion =
          dockerClient
              .logContainerCmd(containerId)
              .withStdOut(true)
              .withStdErr(true)
              .withTimestamps(true)
              .withSince(unixTime)
              .withFollowStream(true)
              .exec(callback)
              .awaitCompletion(timeout, timeUnit);
      if (!completion || callback.matchedRecord == null) {
        long lastSeenRecordMillis = getLogRecordMillis(callback.lastSeenRec);
        throw new DockerClientException(
            "Timeout, token(s): "
                + Arrays.toString(tokens)
                + " not found at container: "
                + containerId
                + ", lookup interval: ["
                + UTC_LOG_CFG_DATE_TIME_FORMATTER.print(sinceMillis)
                + " - "
                + UTC_LOG_CFG_DATE_TIME_FORMATTER.print(lastSeenRecordMillis)
                + "]");
      }
      return callback.matchedRecord;
    } catch (Exception e) {
      throw Throwables.propagate(e);
    }
  }

  public static void disconnectFromNetwork(Container container) {
    dockerClient
        .disconnectFromNetworkCmd()
        .withContainerId(container.getContainerName())
        .withNetworkId(container.getNetwork())
        .exec();
  }

  public static void connectToNetwork(Container container) {
    ContainerNetwork.Ipam ipam = new ContainerNetwork.Ipam().withIpv4Address(container.getIpAddr());

    ContainerNetwork containerNetwork =
        new ContainerNetwork()
            .withNetworkID(container.getNetwork())
            .withIpv4Address(container.getIpAddr())
            .withIpamConfig(ipam);

    dockerClient
        .connectToNetworkCmd()
        .withContainerId(container.getContainerName())
        .withNetworkId(container.getNetwork())
        .withContainerNetwork(containerNetwork)
        .exec();
  }

  private static String pullImage(String registryHost, String imageName, String tag) {
    checkArgument(!Strings.isNullOrEmpty(registryHost));
    checkArgument(!Strings.isNullOrEmpty(imageName));

    String tag1 = Strings.isNullOrEmpty(tag) ? "latest" : tag;
    String fullQualifiedImageName = String.format("%s/%s:%s", registryHost, imageName, tag1);
    dockerClient
        .pullImageCmd(fullQualifiedImageName)
        .exec(new PullImageResultCallback())
        .awaitSuccess();

    String fullQualifiedImageId =
        dockerClient.inspectImageCmd(fullQualifiedImageName).exec().getId();
    fullQualifiedImageId = fullQualifiedImageId.substring(fullQualifiedImageId.indexOf(":") + 1);
    return fullQualifiedImageId.substring(0, IMAGE_ID_LENGTH);
  }

  private static String getImage(String imageName) {
    checkArgument(!Strings.isNullOrEmpty(imageName));
    String fullQualifiedImageId =
        dockerClient.inspectImageCmd(String.format("%s:%s", imageName, "latest")).exec().getId();
    fullQualifiedImageId = fullQualifiedImageId.substring(fullQualifiedImageId.indexOf(":") + 1);
    return fullQualifiedImageId.substring(0, IMAGE_ID_LENGTH);
  }

  private static void copyContainerLog(InputStream is, String containerLogHostPath) {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    try (OutputStream os = new BufferedOutputStream(baos)) {
      ByteStreams.copy(is, os);
      os.flush();
    } catch (IOException e) {
      throw new DockerClientException("Can't copy stream, cause: " + e);
    }

    Path path = Paths.get(Paths.get(containerLogHostPath).toUri());
    try (TarArchiveInputStream in =
        new TarArchiveInputStream(new ByteArrayInputStream(baos.toByteArray()))) {
      TarArchiveEntry entry = in.getNextTarEntry();
      while (entry != null) {
        File file = path.resolve(entry.getName()).toFile();
        if (entry.isDirectory()) {
          file.mkdirs();
        } else {
          try (OutputStream out = new FileOutputStream(file)) {
            ByteStreams.copy(in, out);
            out.flush();
          }
        }
        entry = in.getNextTarEntry();
      }
    } catch (IOException e) {
      throw new DockerClientException("Can't copy stream, cause: " + e);
    }
  }

  private static long getLogRecordMillis(String logRecord) {
    long parsedMillis = -1l;
    try {

      String isoDateAndTime = logRecord.substring(0, logRecord.indexOf('Z') + 1);
      parsedMillis = UTC_ISO_DATE_TIME_FORMATTER.parseMillis(isoDateAndTime);
    } catch (Throwable t) {
      LOGGER.warn("Failed to parse log record's timestamp. Record: {}, Cause: {}", logRecord, t);
    }
    return parsedMillis;
  }


  private static class ContainerLogInputMatchCallback
      extends ResultCallbackTemplate<LogContainerResultCallback, Frame> {
    private final Pattern pattern;
    private final long sinceMillis;
    private String matchedRecord;
    private String lastSeenRec;

    private ContainerLogInputMatchCallback(String[] tokens, long sinceMillis) {
      List<String> quotedTokens =
          Arrays.stream(tokens).map(s -> Pattern.quote(s)).collect(Collectors.toList());
      int flags = Pattern.MULTILINE | Pattern.DOTALL | Pattern.CASE_INSENSITIVE;
      this.pattern = Pattern.compile(".*" + Joiner.on(".*").join(quotedTokens) + ".*", flags);
      this.sinceMillis = sinceMillis;
    }

    @Override
    public void onNext(Frame item) {
      String msg = new String(item.getPayload());
      lastSeenRec = msg;
      long parsedMillis = getLogRecordMillis(msg);
      if (parsedMillis > 0 && parsedMillis >= sinceMillis) {
        if (pattern.matcher(msg).matches()) {
          matchedRecord = msg;
          super.onComplete();
        }
      }
    }
  }


  private static class ExecResultOutputMatchCallback
      extends ResultCallbackTemplate<ExecStartResultCallback, Frame> {
    private final Collection<String> soutResult = new ArrayList<>();
    private final Collection<String> serrResult = new ArrayList<>();

    @Override
    public void onNext(Frame frame) {
      if (frame != null) {
        switch (frame.getStreamType()) {
          case STDOUT:
          case RAW:
            soutResult.add(new String(frame.getPayload()));
            break;
          case STDERR:
            serrResult.add(new String(frame.getPayload()));
            break;
          default:
            LOGGER.error("Unknown stream type: " + frame.getStreamType());
        }
      }
    }

    String soutAsString() {
      return Joiner.on("").join(soutResult);
    }

    String serrAsString() {
      return Joiner.on("").join(serrResult);
    }
  }
}
