package io.scalecube.docker.utils;

import static com.google.common.base.Preconditions.checkNotNull;
import static io.scalecube.docker.utils.Containers.execCommand;

import com.github.dockerjava.api.exception.DockerClientException;
import com.google.common.base.Strings;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public final class Container {
  private static final Logger LOGGER = LoggerFactory.getLogger(Container.class);

  /** Image name. This's part of qualified image name, which is: {@code [imageRegistryHost/]imageName[:tag]} */
  private String imageName;
  /** Helper parameter denoting test in which container being executed. */
  private String domain;
  /** Container name. Set to {@code imageName[-tag]-ipAddr} */
  private String containerName;
  /** Network id or name. */
  private String network;
  /** Ip4 addr. */
  private String ipAddr;
  /** Host path of the container logs. */
  private String containerLogHostPath;
  /** Image registry host. */
  private String registryHost;
  /** Image tag name. */
  private String tag;
  /** Path to application's log folder on container instance. */
  private String containerLogPath;
  /**
   * Path to host machine's base dir where logs from container would be put to. The final log's destination depends on
   * {@link #domain} property.
   * <p>
   * For instance, given:
   * </p>
   * <p>
   * {@code containerLogHostBaseDir = container-log}
   * </p>
   * <p>
   * {@code domain = com.domain.name}
   * </p>
   * <p>
   * The logs for such container would be copied to host machine under: <b>{@code
   * workingDir}/container-log/com/domain/name}</b>
   * </p>
   */
  private String containerLogHostBaseDir = "container-log";

  private Container(Builder builder) {
    this.imageName = builder.imageName;
    this.domain = builder.domain;
    this.containerName = builder.containerName;
    this.network = builder.network;
    this.ipAddr = builder.ipAddr;
    this.containerLogHostPath = builder.containerLogHostPath;
    this.registryHost = builder.registryHost;
    this.tag = builder.tag;
    this.containerLogPath = builder.containerLogPath;
    this.containerLogHostBaseDir = builder.containerLogHostBaseDir;
  }

  public static Container.Builder builder() {
    return new Builder();
  }

  public static Container.Builder from(String image) {
    return new Builder(image);
  }

  public static Builder from(Container container) {
    return new Builder(container);
  }

  public String getImageName() {
    return imageName;
  }

  public String getDomain() {
    return domain;
  }

  public String getContainerName() {
    return containerName;
  }

  public String getNetwork() {
    return network;
  }

  public String getIpAddr() {
    return ipAddr;
  }

  public String getContainerLogPath() {
    return this.containerLogPath;
  }

  public String getContainerLogHostPath() {
    return containerLogHostPath;
  }

  public String getRegistryHost() {
    return registryHost;
  }

  public String getTag() {
    return tag;
  }

  public long getTime() {
    String isoDateAndTime = execCommand(this, "date", "-u", "+%Y-%m-%dT%H:%M:%S.%NZ");
    return Containers.UTC_ISO_DATE_TIME_FORMATTER.parseMillis(isoDateAndTime.replaceAll("\"", "").replaceAll("\n", ""));
  }

  public Container runContainer(String... env) {
    Containers.runContainer(this, env);
    return this;
  }

  public Container restartContainer() {
    Containers.restartContainer(this);
    return this;
  }

  public void removeContainer() {
    removeContainerQuietly();
  }

  public void removeContainerQuietly() {
    try {
      Containers.removeContainer(this);
    } catch (Exception e) {
      LOGGER.error("Can't remove container: {}, cause: {}", getContainerName(), e);
    }
  }

  /**
   * Mutes TCP traffic (IN and OUT) from container to given {@code ip} by virtue of iptables rules
   * 
   * @param ip - remote address of server the traffic should be muted
   * @return the instance of container
   */
  public Container muteTcpBetween(String ip) {
    execCommand(this, "iptables", "-I", "INPUT", "-p", "tcp", "-s", ip, "-j", "DROP");
    execCommand(this, "iptables", "-I", "OUTPUT", "-p", "tcp", "-d", ip, "-j", "DROP");
    return this;
  }

  /**
   * Mutes multicast traffic (IN and OUT) for the container by virtue of iptables rules
   * 
   * @return the instance of container
   */
  public Container muteMulticast() {
    execCommand(this, "iptables", "-I", "INPUT", "-m", "pkttype", "--pkt-type", "multicast", "-j", "DROP");
    execCommand(this, "iptables", "-I", "OUTPUT", "-m", "pkttype", "--pkt-type", "multicast", "-j", "DROP");
    return this;
  }

  /**
   * Discards all iptables rules previously applied to container
   * 
   * @return the instance of container
   */
  public Container removeNetworkRules() {
    execCommand(this, "iptables", "-F");
    return this;
  }

  /**
   * Runs tpckill util to given address (packages form and to would be dropped)
   * 
   * @param ip address to pass as a parameter to tcpkill
   * @return the instance of container
   */
  public Container setTcpKill(String ip) {
    execCommand(this, "tcpkill", "-i", "any", "host", ip);
    return this;
  }

  /**
   * Kills all tcpkill jobs running on container
   * 
   * @return the instance of container
   */
  public Container removeTcpKill() {
    execCommand(this, "pkill", "-9", "tcpkill");
    return this;
  }

  /**
   * Disconnects container from virtual network it was connected to
   * 
   * @return the instance of container
   */
  public Container disconnectFromNetwork() {
    Containers.disconnectFromNetwork(this);
    return this;
  }

  public Container connectToNetwork() {
    Containers.connectToNetwork(this);
    return this;
  }

  public static class Builder {
    /** See {@link Container#containerLogPath} */
    public String containerLogPath;
    /** See {@link Container#containerLogHostBaseDir} */
    public String containerLogHostBaseDir;
    /** See {@link Container#imageName} */
    private String imageName;
    /** See {@link Container#domain} */
    private String domain;
    /** See {@link Container#containerName} */
    private String containerName;
    /** See {@link Container#network} */
    private String network;
    /** See {@link Container#ipAddr} */
    private String ipAddr;
    /** See {@link Container#containerLogHostPath} */
    private String containerLogHostPath;
    /** See {@link Container#registryHost} */
    private String registryHost;
    /** See {@link Container#tag} */
    private String tag;

    private Builder() {}

    private Builder(String imageName) {
      this.imageName = imageName;
    }

    private Builder(Container container) {
      this.imageName = container.imageName;
      this.domain = container.domain;
      this.containerName = container.containerName;
      this.network = container.network;
      this.ipAddr = container.ipAddr;
      this.containerLogHostPath = container.containerLogHostPath;
      this.registryHost = container.registryHost;
      this.tag = container.tag;
      this.containerLogPath = container.containerLogPath;
      this.containerLogHostBaseDir = container.containerLogHostBaseDir;
    }

    public Builder withImageName(String image) {
      this.imageName = image;
      return this;
    }

    public Builder withNetwork(String network) {
      this.network = network;
      return this;
    }

    public Builder withDomain(String domain) {
      this.domain = domain;
      return this;
    }

    public Builder withIpAddr(String ipAddr) {
      this.ipAddr = ipAddr;
      return this;
    }

    public Builder withRegistryHost(String registryHost) {
      this.registryHost = registryHost;
      return this;
    }

    public Builder withTag(String tag) {
      this.tag = tag;
      return this;
    }

    public Builder withContainerLogPath(String containerLogPath) {
      this.containerLogPath = containerLogPath;
      return this;
    }

    public Builder withContainerLogHostBaseDir(String containerLogHostBaseDir) {
      this.containerLogHostBaseDir = containerLogHostBaseDir;
      return this;
    }

    public Container build() {
      if (!Strings.isNullOrEmpty(imageName) && !Strings.isNullOrEmpty(ipAddr)) {
        this.containerName = prepareContainerName();
      }
      if (!Strings.isNullOrEmpty(containerName) && !Strings.isNullOrEmpty(containerLogHostBaseDir)) {
        this.containerLogHostPath = prepareContainerLogHostPath().toString();
      }
      return new Container(this);
    }

    private String prepareContainerName() {
      String replace = imageName.replace('/', '.');
      if (!Strings.isNullOrEmpty(tag)) {
        replace += "-" + tag;
      }
      return replace + "_" + ipAddr;
    }

    private Path prepareContainerLogHostPath() {
      Path result;
      URL targetClasses = Thread.currentThread().getContextClassLoader().getResource("");
      checkNotNull(targetClasses, "Working dir couldn't be found!");
      try {
        String replace = domain.replace('.', '/');
        result =
            Paths.get(targetClasses.toURI()).resolve(containerLogHostBaseDir).resolve(replace).resolve(containerName);
        Files.createDirectories(result); // prepare container host's log directory structure
      } catch (URISyntaxException e) {
        throw new DockerClientException(e.getMessage());
      } catch (IOException e) {
        throw new DockerClientException("Can't create container host's log directory", e);
      }
      return result;
    }
  }
}
