package invirt.data.mongodb

import com.mongodb.client.model.IndexModel
import com.mongodb.client.model.Indexes
import com.mongodb.kotlin.client.MongoCollection
import io.github.oshai.kotlinlogging.KotlinLogging
import kotlin.reflect.KClass
import kotlin.reflect.full.findAnnotation
import kotlin.reflect.full.isSubclassOf
import kotlin.reflect.full.memberProperties

private val log = KotlinLogging.logger {}

@Target(AnnotationTarget.PROPERTY)
@Retention(AnnotationRetention.RUNTIME)
@MustBeDocumented
annotation class Indexed(
    val order: Order = Order.ASC,
    vararg val fields: String
) {

    enum class Order {
        ASC,
        DESC
    }
}

@Target(AnnotationTarget.PROPERTY)
@Retention(AnnotationRetention.RUNTIME)
@MustBeDocumented
annotation class TextIndexed(
    vararg val fields: String
)

fun <T : MongoEntity> MongoCollection<T>.createIndexes() {
    createPropertyIndexes(this.documentClass.kotlin)

    // Handling Timestamped explicitly, because annotations are not inherited in Kotlin
    // so we cannot simply add @Indexed on Timestamped.createdAt
    if (this.documentClass.kotlin.isSubclassOf(TimestampedEntity::class)) {
        createPropertyIndexes(TimestampedEntity::class)
    }
}

private fun MongoCollection<*>.createPropertyIndexes(cls: KClass<*>) {
    val textIndexedFields = mutableListOf<String>()
    val indexes = mutableListOf<IndexModel>()

    cls.memberProperties.forEach { property ->

        // Indexed properties
        property.findAnnotation<Indexed>()?.let { annotation ->
            val fields = if (annotation.fields.isEmpty()) listOf(property.name) else annotation.fields.toList()
            fields.forEach { field ->
                val index = when (annotation.order) {
                    Indexed.Order.ASC -> Indexes.ascending(field)
                    Indexed.Order.DESC -> Indexes.descending(field)
                }
                indexes.add(IndexModel(index))
                log.info { "Creating ${annotation.order} index for ${cls.simpleName}.${field}" }
            }
        }

        // Text indexed properties need collecting into one index as Mongo only supports one text index
        property.findAnnotation<TextIndexed>()?.let { annotation ->
            val fields = if (annotation.fields.isEmpty()) listOf(property.name) else annotation.fields.toList()
            textIndexedFields.addAll(fields)
        }
    }

    if (textIndexedFields.isNotEmpty()) {
        indexes.add(IndexModel(Indexes.compoundIndex(textIndexedFields.map { Indexes.text(it) })))
        log.info { "Creating text index for ${textIndexedFields.joinToString(", ") { cls.simpleName + "." + it }}" }
    }

    if (indexes.isNotEmpty()) {
        log.info { "Creating ${indexes.size} indexes for ${cls.simpleName}" }
        createIndexes(indexes)
    }
}
