package io.pythagoras.messagebus.core;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.pythagoras.messagebus.core.config.MessageBusProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;

@Service
public class MessageBusService implements IMessageBus {

    private static final Logger logger = LoggerFactory.getLogger(MessageBusService.class);

    private MessageBusAdapterProvider messageBusAdapterProvider;

    private MessageContractProvider messageContractProvider;

    private MessageFactory messageFactory;

    private MessageHandlerProvider messageHandlerProvider;

    private ObjectMapper mapper;

    private ExecutorService executorService;

    private MessageBusProperties properties;

    private LinkedBlockingQueue<Runnable> queue = new LinkedBlockingQueue<>();

    @Autowired
    public MessageBusService(
            MessageBusAdapterProvider messageBusAdapterProvider,
            MessageContractProvider messageContractProvider,
            MessageHandlerProvider messageHandlerProvider,
            MessageFactory messageFactory,
            MessageBusProperties properties) {
        this.messageBusAdapterProvider = messageBusAdapterProvider;
        this.messageHandlerProvider = messageHandlerProvider;
        this.messageContractProvider = messageContractProvider;
        this.messageFactory = messageFactory;
        this.mapper = new ObjectMapper();
        this.properties = properties;
    }

    @PostConstruct
    public void init() {
        if (!properties.isEnabled()) {
            return;
        }

        List<String> messageCodes = this.messageContractProvider.getCodeList();
        List<String> handlerCodes = this.messageHandlerProvider.getHandlerContractCodes();
        Map<String, List<Class<IMessageContract>>> messageCodesAndMessageContracts = this.messageContractProvider.getCodeListWithMessageContracts();

        this.messageBusAdapterProvider.getAdapter().initialize(messageCodes, handlerCodes, messageCodesAndMessageContracts);
        if (properties.isReceiveEnabled()) {
            this.messageBusAdapterProvider.getAdapter().registerMessageBusService(this);
            this.messageBusAdapterProvider.getAdapter().start();
        }

        // Set up the outbound thread pool
        if (properties.isSendEnabled()) {
            this.executorService = new ThreadPoolExecutor(properties.getOutboundThreadCount(), properties.getOutboundThreadCount(),
                    0L, TimeUnit.MILLISECONDS,
                    this.queue);
        }
    }

    public int getOutboundWorkQueueSize() {
        return this.queue.size();
    }

    /**
     * Takes the specific message, turns it into a generic message, and send to the message bus. ASYNC
     *
     * @param message send message into the bus
     * @throws MessageSendingException on any failure
     */
    public void sendMessage(IMessageContract message) throws MessageSendingException {

        if (!properties.isEnabled()) {
            logger.warn("MessageBus is disabled.");
            return;
        }

        if (!properties.isSendEnabled()) {
            logger.warn("MessageBus is not enabled for sending.");
            return;
        }

        executorService.submit(() -> {
            sendImessageContract(message);
        });
    }

    /**
     * Takes the specific message, turns it into a generic message, and send to the message bus. BLOCKING
     *
     * @param message send message into the bus
     * @throws MessageSendingException on any failure
     */
    public void sendBlockingMessage(IMessageContract message) throws MessageSendingException {

        if (!properties.isEnabled()) {
            logger.warn("MessageBus is disabled.");
            return;
        }

        if (!properties.isSendEnabled()) {
            logger.warn("MessageBus is not enabled for sending.");
            return;
        }

        sendImessageContract(message);
    }

    private void sendImessageContract(IMessageContract message) {
        // Convert message to IBusMessage
        BusMessage busMessage = new BusMessage();
        busMessage.setCode(message.getCode());
        busMessage.setVersion(message.getVersion());

        // Serialize the body
        try {
            String body = mapper.writeValueAsString(message);
            busMessage.setPayload(body);
        } catch (JsonProcessingException e) {
            logger.error("Unable to convert message to json.", e);
            return;
        }

        // Send the Message using Adapter.sendMessage()
        IMessageBusAdapter adapter = this.messageBusAdapterProvider.getAdapter();
        adapter.sendMessage(busMessage);
    }

    /**
     * Handles a generic message coming in from the message Bus, and converts it into the
     * specific contract types and dispatches it.
     *
     * @param busMessage the message received from the bus
     * @throws HandleMessageFailureException on any failure
     */
    public void receiveMessage(IBusMessage busMessage) throws HandleMessageFailureException {

        if (!properties.isEnabled()) {
            throw new HandleMessageFailureException("MessageBus is disabled.");
        }

        if (!properties.isReceiveEnabled()) {
            throw new HandleMessageFailureException("MessageBus receiving is not enabled.");
        }

        IMessageContract message = null;
        List<IMessageHandler> handlers = null;
        try {
            message = this.convertFromBusMessage(busMessage);
            handlers = this.messageHandlerProvider.getMessageHandlersForCodeAndVersion(message.getCode(), message.getVersion());
        } catch (Exception e) {
            logger.error("Message Unwrapping Error: " + e.getMessage(), e, busMessage);
            throw new HandleMessageFailureException("Message Unwrapping Error: "+e.getMessage(),e);
        }
        for (IMessageHandler handler : handlers) {
            //executorService.submit(() -> {
            try {
                handler.handleMessage(message);
                if(IMessageHandler2.class.isInstance(handler)) {
                    IMessageHandler2 handler2 = (IMessageHandler2) handler;
                    handler2.handleMessage(message, busMessage.getPayload(), busMessage.getSentTime());

                    // TODO: Remove this with version 1.8.0
                    handler2.handleMessage(message,busMessage.getSentTime());
                    handler2.handleRawMessage(busMessage.getPayload(), busMessage.getSentTime());
                }
            } catch (Exception e) {
                logger.error("Message Handling Error: " + e.getMessage(), e);
                throw new HandleMessageFailureException("Message Handling Error: "+e.getMessage(),e);
            }
            //});
        }
    }


    private IMessageContract convertFromBusMessage(IBusMessage busMessage) {
        Class klass = messageContractProvider.get(busMessage.getCode(), busMessage.getVersion());
        IMessageContract obj = messageFactory.make(klass);

        try {
            return mapper.readValue(busMessage.getPayload(), obj.getClass());
        } catch (IOException e) {
            logger.error("Unable to convert json to message.", e, busMessage.getPayload());
        }
        return null;
    }

    @Override
    public void disableMessageBus() throws MessageBusStateException {
        // Check if already disabled.
        if (!this.properties.isEnabled()) {
            return;
        }

        // Stop sending
        this.properties.setSendEnabled(false);

        // Tell the adapter to disable/stop.
        this.messageBusAdapterProvider.getAdapter().stop();

        // Stop whole queue
        this.properties.setReceiveEnabled(false);
        this.properties.setEnabled(false);
    }

    @Override
    public void enableMessageBus() throws MessageBusStateException {
        this.properties.setEnabled(true);
        this.properties.setReceiveEnabled(true);
        this.properties.setSendEnabled(true);

        try {
            this.init();
        } catch (MessageBusInitializationException e) {
            throw new MessageBusStateException("Unable to enable message bus.", e);
        }
    }

    @PreDestroy
    public void preDestroy() {
        this.messageBusAdapterProvider.getAdapter().cleanResources();
    }
}
