package io.pythagoras.messagebus.core;

import java.util.List;
import java.util.Map;

public interface IMessageBusAdapter {

    /**
     *
     * @param messageCodes The codes for all the messages in scope (Exchanges)
     * @param handlerCodes The codes the there are handlers for (Exchanges we need a queue which we are subscribed to)
     * @throws MessageBusInitializationException On failure to initialize
     */
    default void initialize(List<String> messageCodes, List<String> handlerCodes) throws MessageBusInitializationException{
        // new or existing adapters can use new signature of initialize method ->
        // initialize(List<String>, List<String>, HashMap<String, HashMap<Integer, Class<IMessageContract>>>, HashMap<String, List<IMessageHandler>>)
        // when new signature is used there is no need to have implementation of this method so default empty implementation is added
    }

    /**
     * @param messageCodes         The codes for all the messages in scope (Exchanges)
     * @param handlerCodes         The codes the there are handlers for (Exchanges we need a queue which we are subscribed to)
     * @param messageCodesAndMessageContracts Hash map of codes and message contracts
     * @throws MessageBusInitializationException On failure to initialize
     */
    default void initialize(List<String> messageCodes, List<String> handlerCodes, Map<String, List<Class<IMessageContract>>> messageCodesAndMessageContracts) throws MessageBusInitializationException {
        // default implementation in order to maintain backward compatibility
        initialize(messageCodes, handlerCodes);
    }

    /**
     * Sends a message out to the bus implementation.
     * @param message Message to send to the Bus
     * @throws MessageSendingException If an error occurs
     */
    void sendMessage(IBusMessage message) throws MessageSendingException;

    /**
     * Registers the messageBusService.
     * @param messageBus the message bus service.
     */
    void registerMessageBusService(IMessageBus messageBus);

    /**
     * Stops consuming incoming messages.
     * @throws MessageBusStateException
     */
    void stop() throws MessageBusStateException;

    /**
     * Starts Consuming messages.
     * @throws MessageBusStateException
     */
    void start() throws MessageBusStateException;

    /**
     * Clean resources before service shutdown.
     */
    default void cleanResources() {}
}
