package io.pythagoras.messagebus.core;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import io.pythagoras.messagebus.core.config.MessageBusProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.List;

@Service
public class MessageBusService implements IMessageBus {

    private MessageBusAdapterProvider messageBusAdapterProvider;

    private MessageContractProvider messageContractProvider;

    private MessageFactory messageFactory;

    private MessageHandlerProvider messageHandlerProvider;

    private Gson gson;

    private Boolean enabled;

    @Autowired
    public MessageBusService(
            MessageBusAdapterProvider messageBusAdapterProvider,
            MessageContractProvider messageContractProvider,
            MessageHandlerProvider messageHandlerProvider,
            MessageFactory messageFactory,
            MessageBusProperties properties) {
        this.messageBusAdapterProvider = messageBusAdapterProvider;
        this.messageHandlerProvider = messageHandlerProvider;
        this.messageContractProvider = messageContractProvider;
        this.messageFactory = messageFactory;
        this.gson = new GsonBuilder().create();
        this.enabled = properties.isEnabled();
    }

    @PostConstruct
    public void init() {
        if(!enabled) {
            return;
        }

        List<String> messageCodes = this.messageContractProvider.getCodeList();
        List<String> handlerCodes = this.messageHandlerProvider.getHandlerContractCodes();

        this.messageBusAdapterProvider.getAdapter().initialize(messageCodes,handlerCodes);
        this.messageBusAdapterProvider.getAdapter().registerMessageBusService(this);

    }

    /**
     * Takes the specific message, turns it into a generic message, and send to the message bus.
     * @param message send message into the bus
     * @throws MessageSendingException on any failure
     */
    public void sendMessage(IMessageContract message) throws MessageSendingException {

        if(!enabled) {
            throw new MessageSendingException("MessageBus is disabled.");
        }

        // Convert message to IBusMessage
        BusMessage busMessage = new BusMessage();
        busMessage.setCode(message.getCode());
        busMessage.setVersion(message.getVersion());

        // Serialize the body
        String body = gson.toJson(message);
        busMessage.setPayload(body);

        // Send the Message using Adapter.sendMessage()
        this.messageBusAdapterProvider.getAdapter().sendMessage(busMessage);
    }

    /**
     * Handles a generic message coming in from the message Bus, and converts it into the
     * specific contract types and dispatches it.
     * @param busMessage the message received from the bus
     * @throws HandleMessageFailureException on any failure
     */
    public void receiveMessage(IBusMessage busMessage) throws HandleMessageFailureException {

        if(!enabled) {
            throw new HandleMessageFailureException("MessageBus is disabled.");
        }

        IMessageContract message = this.convertFromBusMessage(busMessage);

        IMessageHandler handler = this.messageHandlerProvider.getMessageHandlerForCodeAndVersion(message.getCode(),message.getVersion());

        handler.handleMessage(message);
    }


    private IMessageContract convertFromBusMessage(IBusMessage busMessage) {
        Class klass = messageContractProvider.get(busMessage.getCode(),busMessage.getVersion());
        IMessageContract obj = messageFactory.make(klass);

        IMessageContract message = gson.fromJson(busMessage.getPayload(),obj.getClass());

        return message;
    }




}
