/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.codec.compression;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.util.ByteProcessor;
import io.netty.util.CharsetUtil;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.internal.EmptyArrays;
import io.netty.util.test.DisabledForSlowLeakDetection;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Random;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

public abstract class AbstractIntegrationTest {
    protected static final Random rand = new Random();
    protected EmbeddedChannel encoder;
    protected EmbeddedChannel decoder;

    protected abstract EmbeddedChannel createEncoder();

    protected abstract EmbeddedChannel createDecoder();

    public void initChannels() {
        this.encoder = this.createEncoder();
        this.decoder = this.createDecoder();
    }

    public void closeChannels() {
        Object msg;
        this.encoder.close();
        while ((msg = this.encoder.readOutbound()) != null) {
            ReferenceCountUtil.release((Object)msg);
        }
        this.decoder.close();
        while ((msg = this.decoder.readInbound()) != null) {
            ReferenceCountUtil.release((Object)msg);
        }
    }

    @Test
    public void testEmpty() throws Exception {
        this.testIdentity(EmptyArrays.EMPTY_BYTES, true);
        this.testIdentity(EmptyArrays.EMPTY_BYTES, false);
    }

    @Test
    public void testOneByte() throws Exception {
        byte[] data = new byte[]{65};
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testTwoBytes() throws Exception {
        byte[] data = new byte[]{66, 65};
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testRegular() throws Exception {
        byte[] data = "Netty is a NIO client server framework which enables quick and easy development of network applications such as protocol servers and clients.".getBytes(CharsetUtil.UTF_8);
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testLargeRandom() throws Exception {
        byte[] data = new byte[0x100000];
        rand.nextBytes(data);
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testPartRandom() throws Exception {
        byte[] data = new byte[10240];
        rand.nextBytes(data);
        for (int i = 0; i < 1024; ++i) {
            data[i] = 2;
        }
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testCompressible() throws Exception {
        byte[] data = new byte[10240];
        for (int i = 0; i < data.length; ++i) {
            data[i] = i % 4 != 0 ? (byte)0 : (byte)rand.nextInt();
        }
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testLongBlank() throws Exception {
        byte[] data = new byte[102400];
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testLongSame() throws Exception {
        byte[] data = new byte[102400];
        Arrays.fill(data, (byte)123);
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    @Test
    public void testSequential() throws Exception {
        byte[] data = new byte[1024];
        for (int i = 0; i < data.length; ++i) {
            data[i] = (byte)i;
        }
        this.testIdentity(data, true);
        this.testIdentity(data, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void testIdentity(byte[] data, boolean heapBuffer) {
        this.initChannels();
        ByteBuf in = heapBuffer ? Unpooled.wrappedBuffer((byte[])data) : Unpooled.directBuffer((int)data.length).writeBytes(data);
        CompositeByteBuf compressed = Unpooled.compositeBuffer();
        CompositeByteBuf decompressed = Unpooled.compositeBuffer();
        try {
            ByteBuf msg;
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.encoder.writeOutbound(new Object[]{in.retain()}));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.encoder.finish());
            while ((msg = (ByteBuf)this.encoder.readOutbound()) != null) {
                compressed.addComponent(true, msg);
            }
            this.decoder.writeInbound(new Object[]{compressed.retain()});
            org.junit.jupiter.api.Assertions.assertFalse((boolean)compressed.isReadable());
            while ((msg = (ByteBuf)this.decoder.readInbound()) != null) {
                decompressed.addComponent(true, msg);
            }
            in.readerIndex(0);
            org.junit.jupiter.api.Assertions.assertEquals((Object)in, (Object)decompressed);
        }
        finally {
            compressed.release();
            decompressed.release();
            in.release();
            this.closeChannels();
        }
    }

    @Test
    @DisabledForSlowLeakDetection
    public void testHugeDecompress() throws Exception {
        int chunkSize = 0x100000;
        int numberOfChunks = 256;
        int memoryLimit = chunkSize * 128;
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        EmbeddedChannel compressChannel = this.createEncoder();
        ByteBuf compressed = compressChannel.alloc().buffer();
        for (int i = 0; i <= numberOfChunks; ++i) {
            ByteBuf buf;
            if (i < numberOfChunks) {
                ByteBuf in = compressChannel.alloc().buffer(chunkSize);
                for (int j = 0; j < chunkSize; ++j) {
                    byte byteValue = (byte)(i + (j & 0xA0));
                    in.writeByte((int)byteValue);
                    digest.update(byteValue);
                }
                compressChannel.writeOutbound(new Object[]{in});
            } else {
                compressChannel.close();
            }
            while ((buf = (ByteBuf)compressChannel.readOutbound()) != null) {
                compressed.writeBytes(buf);
                buf.release();
            }
        }
        byte[] expectedData = digest.digest();
        PooledByteBufAllocator allocator = new PooledByteBufAllocator(false);
        HugeDecompressIncomingHandler endHandler = new HugeDecompressIncomingHandler(memoryLimit, digest);
        EmbeddedChannel decompressChannel = this.createDecoder();
        decompressChannel.pipeline().addLast(new ChannelHandler[]{endHandler});
        decompressChannel.config().setAllocator((ByteBufAllocator)allocator);
        decompressChannel.writeInbound(new Object[]{compressed});
        decompressChannel.finishAndReleaseAll();
        org.junit.jupiter.api.Assertions.assertEquals((long)((long)chunkSize * (long)numberOfChunks), (long)endHandler.total);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])expectedData, (byte[])digest.digest());
    }

    private static final class HugeDecompressIncomingHandler
    extends ChannelInboundHandlerAdapter
    implements ByteProcessor {
        final int memoryLimit;
        final MessageDigest digest;
        long total;

        HugeDecompressIncomingHandler(int memoryLimit, MessageDigest digest) {
            this.memoryLimit = memoryLimit;
            this.digest = digest;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void channelRead(ChannelHandlerContext ctx, Object msg) {
            ByteBuf buf = (ByteBuf)msg;
            this.total += (long)buf.readableBytes();
            try {
                buf.forEachByte((ByteProcessor)this);
                PooledByteBufAllocator allocator = (PooledByteBufAllocator)ctx.alloc();
                Assertions.assertThat((long)allocator.metric().usedHeapMemory()).isLessThan((long)this.memoryLimit);
                Assertions.assertThat((long)allocator.metric().usedDirectMemory()).isLessThan((long)this.memoryLimit);
            }
            finally {
                buf.release();
            }
        }

        public boolean process(byte value) throws Exception {
            this.digest.update(value);
            return true;
        }
    }
}

