/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package io.milvus.v2.service.partition.request;

import java.util.ArrayList;
import java.util.List;

public class LoadPartitionsReq {
    private String databaseName;
    private String collectionName;
    private List<String> partitionNames;
    private Integer numReplicas;
    private Boolean sync; // wait the partitions to be fully loaded
    private Long timeout; // timeout value for waiting the partitions to be fully loaded
    private Boolean refresh;
    private List<String> loadFields;
    private Boolean skipLoadDynamicField;
    private List<String> resourceGroups;

    private LoadPartitionsReq(LoadPartitionsReqBuilder builder) {
        this.databaseName = builder.databaseName;
        this.collectionName = builder.collectionName;
        this.partitionNames = builder.partitionNames;
        this.numReplicas = builder.numReplicas;
        this.sync = builder.sync;
        this.timeout = builder.timeout;
        this.refresh = builder.refresh;
        this.loadFields = builder.loadFields;
        this.skipLoadDynamicField = builder.skipLoadDynamicField;
        this.resourceGroups = builder.resourceGroups;
    }

    public String getDatabaseName() {
        return databaseName;
    }

    public void setDatabaseName(String databaseName) {
        this.databaseName = databaseName;
    }

    public String getCollectionName() {
        return collectionName;
    }

    public void setCollectionName(String collectionName) {
        this.collectionName = collectionName;
    }

    public List<String> getPartitionNames() {
        return partitionNames;
    }

    public void setPartitionNames(List<String> partitionNames) {
        this.partitionNames = partitionNames;
    }

    public Integer getNumReplicas() {
        return numReplicas;
    }

    public void setNumReplicas(Integer numReplicas) {
        this.numReplicas = numReplicas;
    }

    public Boolean getSync() {
        return sync;
    }

    public void setSync(Boolean sync) {
        this.sync = sync;
    }

    public Long getTimeout() {
        return timeout;
    }

    public void setTimeout(Long timeout) {
        this.timeout = timeout;
    }

    public Boolean getRefresh() {
        return refresh;
    }

    public void setRefresh(Boolean refresh) {
        this.refresh = refresh;
    }

    public List<String> getLoadFields() {
        return loadFields;
    }

    public void setLoadFields(List<String> loadFields) {
        this.loadFields = loadFields;
    }

    public Boolean getSkipLoadDynamicField() {
        return skipLoadDynamicField;
    }

    public void setSkipLoadDynamicField(Boolean skipLoadDynamicField) {
        this.skipLoadDynamicField = skipLoadDynamicField;
    }

    public List<String> getResourceGroups() {
        return resourceGroups;
    }

    public void setResourceGroups(List<String> resourceGroups) {
        this.resourceGroups = resourceGroups;
    }

    @Override
    public String toString() {
        return "LoadPartitionsReq{" +
                "databaseName='" + databaseName + '\'' +
                ", collectionName='" + collectionName + '\'' +
                ", partitionNames=" + partitionNames +
                ", numReplicas=" + numReplicas +
                ", sync=" + sync +
                ", timeout=" + timeout +
                ", refresh=" + refresh +
                ", loadFields=" + loadFields +
                ", skipLoadDynamicField=" + skipLoadDynamicField +
                ", resourceGroups=" + resourceGroups +
                '}';
    }

    public static LoadPartitionsReqBuilder builder() {
        return new LoadPartitionsReqBuilder();
    }

    public static class LoadPartitionsReqBuilder {
        private String databaseName;
        private String collectionName;
        private List<String> partitionNames = new ArrayList<>();
        private Integer numReplicas = 1;
        private Boolean sync = Boolean.TRUE; // wait the partitions to be fully loaded
        private Long timeout = 60000L; // timeout value for waiting the partitions to be fully loaded
        private Boolean refresh = Boolean.FALSE;
        private List<String> loadFields = new ArrayList<>();
        private Boolean skipLoadDynamicField = Boolean.FALSE;
        private List<String> resourceGroups = new ArrayList<>();

        private LoadPartitionsReqBuilder() {
        }

        public LoadPartitionsReqBuilder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public LoadPartitionsReqBuilder collectionName(String collectionName) {
            this.collectionName = collectionName;
            return this;
        }

        public LoadPartitionsReqBuilder partitionNames(List<String> partitionNames) {
            this.partitionNames = partitionNames;
            return this;
        }

        public LoadPartitionsReqBuilder numReplicas(Integer numReplicas) {
            this.numReplicas = numReplicas;
            return this;
        }

        public LoadPartitionsReqBuilder sync(Boolean sync) {
            this.sync = sync;
            return this;
        }

        public LoadPartitionsReqBuilder timeout(Long timeout) {
            this.timeout = timeout;
            return this;
        }

        public LoadPartitionsReqBuilder refresh(Boolean refresh) {
            this.refresh = refresh;
            return this;
        }

        public LoadPartitionsReqBuilder loadFields(List<String> loadFields) {
            this.loadFields = loadFields;
            return this;
        }

        public LoadPartitionsReqBuilder skipLoadDynamicField(Boolean skipLoadDynamicField) {
            this.skipLoadDynamicField = skipLoadDynamicField;
            return this;
        }

        public LoadPartitionsReqBuilder resourceGroups(List<String> resourceGroups) {
            this.resourceGroups = resourceGroups;
            return this;
        }

        public LoadPartitionsReq build() {
            return new LoadPartitionsReq(this);
        }
    }
}
