/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package io.milvus.param.dml.ranker;

import com.google.gson.JsonObject;
import io.milvus.common.utils.JsonUtils;
import io.milvus.exception.ParamException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The Average Weighted Scoring reranking strategy, which prioritizes vectors based on relevance,
 * averaging their significance.
 */
public class WeightedRanker extends BaseRanker {
    private final List<Float> weights;

    private WeightedRanker(Builder builder) {
        // Replace @NonNull logic with explicit null check
        if (builder == null) {
            throw new IllegalArgumentException("builder cannot be null");
        }
        this.weights = builder.weights;
    }

    // Getter method to replace @Getter annotation
    public List<Float> getWeights() {
        return weights;
    }

    // toString method to replace @ToString annotation
    @Override
    public String toString() {
        return "WeightedRanker{" +
                "weights=" + weights +
                '}';
    }

    @Override
    public Map<String, String> getProperties() {
        JsonObject params = new JsonObject();
        params.add("weights", JsonUtils.toJsonTree(this.weights).getAsJsonArray());

        Map<String, String> props = new HashMap<>();
        props.put("strategy", "weighted");
        props.put("params", params.toString());
        return props;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    /**
     * Builder for {@link WeightedRanker} class.
     */
    public static class Builder {
        private List<Float> weights = new ArrayList<>();

        private Builder() {
        }

        /**
         * Assign weights for each AnnSearchParam. The length of weights must be equal to number of AnnSearchParam.
         * You can assign any float value for weight, the sum of weight values can exceed 1.
         * The distance/similarity values of each field will be mapped into a range of [0,1],
         * and score = sum(weights[i] * distance_i_in_[0,1])
         *
         * @param weights weight values
         * @return <code>Builder</code>
         */
        public Builder withWeights(List<Float> weights) {
            // Replace @NonNull logic with explicit null check
            if (weights == null) {
                throw new IllegalArgumentException("weights cannot be null");
            }
            this.weights = weights;
            return this;
        }

        /**
         * Verifies parameters and creates a new {@link WeightedRanker} instance.
         *
         * @return {@link WeightedRanker}
         */
        public WeightedRanker build() throws ParamException {
            return new WeightedRanker(this);
        }
    }
}
