package io.lumigo.core.parsers.v1;

import static io.lumigo.core.utils.StringUtils.dynamodbItemToHash;

import com.amazonaws.AmazonWebServiceRequest;
import com.amazonaws.Request;
import com.amazonaws.Response;
import com.amazonaws.services.dynamodbv2.model.*;
import io.lumigo.models.HttpSpan;
import java.util.List;
import java.util.Map;

public class DynamoDBV1Parser implements AwsSdkV1Parser {
    @Override
    public String getParserType() {
        return DynamoDBV1Parser.class.getName();
    }

    @Override
    public void parse(HttpSpan span, Request request, Response response) {
        String messageId = extractMessageId(request.getOriginalRequest());
        if (messageId != null) span.getInfo().setMessageId(messageId);
        String tableName = extractTableName(request.getOriginalRequest());
        if (tableName != null) span.getInfo().setResourceName(tableName);
    }

    private String extractMessageId(AmazonWebServiceRequest request) {
        if (request instanceof PutItemRequest) {
            return dynamodbItemToHash(((PutItemRequest) request).getItem());
        } else if (request instanceof UpdateItemRequest) {
            return dynamodbItemToHash(((UpdateItemRequest) request).getKey());
        } else if (request instanceof DeleteItemRequest) {
            return dynamodbItemToHash(((DeleteItemRequest) request).getKey());
        } else if (request instanceof BatchWriteItemRequest) {
            Map<String, List<WriteRequest>> requests =
                    ((BatchWriteItemRequest) request).getRequestItems();
            WriteRequest firstRequest = requests.entrySet().iterator().next().getValue().get(0);
            if (firstRequest.getPutRequest() != null) {
                return dynamodbItemToHash(firstRequest.getPutRequest().getItem());
            } else if (firstRequest.getDeleteRequest() != null) {
                return dynamodbItemToHash(firstRequest.getDeleteRequest().getKey());
            }
        }
        return null;
    }

    private String extractTableName(AmazonWebServiceRequest request) {
        if (request instanceof PutItemRequest) {
            return ((PutItemRequest) request).getTableName();
        } else if (request instanceof UpdateItemRequest) {
            return ((UpdateItemRequest) request).getTableName();
        } else if (request instanceof DeleteItemRequest) {
            return ((DeleteItemRequest) request).getTableName();
        } else if (request instanceof BatchWriteItemRequest) {
            Map<String, List<WriteRequest>> requests =
                    ((BatchWriteItemRequest) request).getRequestItems();
            return requests.entrySet().iterator().next().getKey();
        } else if (request instanceof BatchGetItemRequest) {
            Map<String, KeysAndAttributes> requests =
                    ((BatchGetItemRequest) request).getRequestItems();
            return requests.entrySet().iterator().next().getKey();
        } else if (request instanceof GetItemRequest) {
            return ((GetItemRequest) request).getTableName();
        }
        return null;
    }
}
