package io.lsn.spring.rest;

import io.lsn.spring.rest.helper.LoggingInterceptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.http.*;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.net.URI;

@Component
public class RestClient {

    private RestTemplateBuilder restTemplateBuilder;

    @Autowired
    public RestClient(RestTemplateBuilder restTemplateBuilder) {
        this.restTemplateBuilder = restTemplateBuilder;
    }

    protected RestTemplate buildRestTemplate() {
        SimpleClientHttpRequestFactory requestFactory = new SimpleClientHttpRequestFactory();
        BufferingClientHttpRequestFactory bufferingClientHttpRequestFactory = new BufferingClientHttpRequestFactory(requestFactory);
        requestFactory.setOutputStreaming(false);

        RestTemplate restTemplate = restTemplateBuilder
                .additionalMessageConverters(new MappingJackson2HttpMessageConverter())
                .interceptors(new LoggingInterceptor())
                .build();

        restTemplate.setRequestFactory(bufferingClientHttpRequestFactory);
        return restTemplate;
    }

    /**
     * @param url           - url of endpoint
     * @param responseClass - response class*
     * @param <K>           - object request type
     * @param <T>           - object response type
     * @return ResponseEntity<T>, responseEntity.getBody() (to get the body of the response)
     */
    public <K, T> ResponseEntity<T> get(String url, Class<T> responseClass) {
        return buildRestTemplate().exchange(URI.create(url), HttpMethod.GET, buildHttpEntity(null), responseClass);
    }

    /**
     * @param url           - url of endpoint
     * @param request       - request object
     * @param responseClass - response class*
     * @param <K>           - object request type
     * @param <T>           - object response type
     * @return ResponseEntity<T>, responseEntity.getBody() (to get the body of the response)
     */
    public <K, T> ResponseEntity<T> post(String url, K request, Class<T> responseClass) {
        return buildRestTemplate().exchange(URI.create(url), HttpMethod.POST, buildHttpEntity(request), responseClass);
    }

    /**
     * @param url           - url of endpoint
     * @param request       - request object
     * @param responseClass - response class*
     * @param <K>           - object request type
     * @param <T>           - object response type
     * @return ResponseEntity<T>, responseEntity.getBody() (to get the body of the response)
     */
    public <K, T> ResponseEntity<T> put(String url, K request, Class<T> responseClass) {
        return buildRestTemplate().exchange(URI.create(url), HttpMethod.PUT, buildHttpEntity(request), responseClass);
    }

    protected <T> HttpEntity<T> buildHttpEntity(T requestClass) {

        HttpHeaders headers = new HttpHeaders();
        headers.set(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        if (requestClass != null) {
            return new HttpEntity<>(requestClass, headers);
        }
        return new HttpEntity<>(headers);

    }
}
