package io.liveqa.web;

import io.liveqa.Config;
import io.liveqa.LiveQA;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;

import static java.util.Arrays.asList;

public class LiveQAFilter implements Filter {

  private static final ThreadLocal<Map> STORE = new ThreadLocal<>();
  private static final String TRACKER_NAME = "liveqa_tracker_id";

  private Map<String, Object> store;
  private HttpServletRequest request;

  public static Map getStore() {
    return STORE.get();
  }

  public LiveQAFilter() { }

  public LiveQAFilter(Config config) {
    LiveQA.configure(config);
  }

  @Override
  public void init(FilterConfig filterConfig) throws ServletException {
    String accountToken = filterConfig.getInitParameter("accountToken");
    String spaceName = filterConfig.getInitParameter("spaceName");
    String environmentName = filterConfig.getInitParameter("environmentName");

    if (accountToken != null) {
      Config config = Config.builder(accountToken, spaceName, environmentName)
          .build();
      LiveQA.configure(config);
    }
  }

  @Override
  public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
    this.store = new HashMap<>();
    this.request = (HttpServletRequest) req;
    HttpServletResponse response = (HttpServletResponse) res;
    STORE.set(this.store);

    storeTrackerId();
    storeRequestData();

    response.addCookie(new Cookie(TRACKER_NAME, store.get("sessionTrackerId").toString()));

    chain.doFilter(req, res);

    STORE.remove();
  }

  @Override
  public void destroy() {
    LiveQA.shutdown();
  }

  private void storeTrackerId() {
    String value = null;

    if (request.getCookies() != null) {
      for (Cookie cookie : request.getCookies()) {
        if (TRACKER_NAME.equals(cookie.getName())) {
          value = cookie.getValue();
        }
      }
    }

    store.put(
        "sessionTrackerId",
        (value == null || value.isEmpty()) ? UUID.randomUUID() : value
    );
  }

  private void storeRequestData() {
    Map<String, Object> data = new HashMap<>();

    data.put("url", getFullUrl());
    data.put("ssl", httpsRequest());
    data.put("host", request.getServerName());
    data.put("port", request.getServerPort());
    data.put("path", request.getRequestURI());
    data.put("referrer", request.getHeader("Referer"));
    data.put("method", request.getMethod());
    data.put("xhr", "XMLHttpRequest".equals(request.getHeader("X-Requested-With")));
    data.put("userAgent", request.getHeader("User-Agent"));
    data.put("ip", request.getRemoteAddr());
    data.put("headers", getHeaders());

    if (request.getMethod().equals("POST")) {
      data.put("postParams", getParams());
    } else {
      data.put("getParams", getParams());
    }

    store.put("request", data);
  }

  private Map<String, String> getHeaders() {
    Map<String, String> headers = new HashMap<>();
    String[] skipHeaders = { "host", "user-agent", "cookie" };

    Enumeration<String> headerNames = request.getHeaderNames();
    while (headerNames.hasMoreElements()) {
      String headerName = headerNames.nextElement();

      if (Arrays.stream(skipHeaders).anyMatch(headerName::equals)) {
        continue;
      }

      headers.put(headerName, request.getHeader(headerName));
    }

    return headers;
  }

  private Map<String, List<String>> getParams() {
    Map<String, List<String>> params = new HashMap<>();

    Map<String, String[]> paramNames = request.getParameterMap();

    for (Map.Entry<String, String[]> param : paramNames.entrySet()) {
      if (param.getValue() != null && param.getValue().length > 0) {
        params.put(param.getKey(), asList(param.getValue()));
      }
    }

    return params;
  }

  private boolean httpsRequest() {
    if ("https".equals(request.getHeader("X-Forwarded-Proto"))) { return true; }
    if ("https".equals(request.getScheme())) { return true; }

    return false;
  }

  public String getFullUrl() {
    StringBuilder requestUrl = new StringBuilder(request.getRequestURL().toString());
    String queryString = request.getQueryString();

    if (queryString == null) {
      return requestUrl.toString();
    } else {
      return requestUrl.append('?').append(queryString).toString();
    }
  }
}
