package io.linkrunner.sdk.work

import androidx.work.*
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import java.util.concurrent.TimeUnit

/**
 * Work request class for handling background API calls with retry mechanism
 */
data class LinkRunnerWorkRequest(
    val apiMethod: String,
    val parameters: Map<String, Any>
) {
    companion object {
        // API method constants
        const val KIND_INITIALIZE = "initialize"
        const val KIND_SEND_EVENT = "send_event"
        const val KIND_TRACK_INSTALL = "track_install"
        const val KIND_SIGNUP = "signup"
        const val KIND_CAPTURE_PAYMENT = "capture_payment"
        const val KIND_REMOVE_PAYMENT = "remove_payment"
        const val KIND_SET_USER_DATA = "set_user_data"
        const val KIND_DEEPLINK_TRIGGERED = "deeplink_triggered"
        const val KIND_ATTRIBUTION_DATA = "attribution_data"
        const val KIND_UPDATE_PUSH_TOKEN = "update_push_token"
        const val KIND_LOG = "log"
        const val KIND_UPDATE_INTEGRATIONS = "update_integrations"
        
        // Work data keys
        private const val WORK_DATA_API_METHOD = "api_method"
        private const val WORK_DATA_PARAMETERS = "parameters"
        private const val WORK_TAG = "linkrunner_api_work"
        
        private val gson = Gson()
        
        /**
         * Enqueue a work request for immediate execution with background retry fallback
         * @param workRequest The work request to enqueue
         * @param immediate Whether to try immediate execution first (default: true)
         * @throws Exception if enqueue fails
         */
        fun enqueue(workRequest: LinkRunnerWorkRequest, immediate: Boolean = true) {
            // Serialize parameters to JSON
            val parametersJson = gson.toJson(workRequest.parameters)
            
            // Create input data
            val inputData = workDataOf(
                WORK_DATA_API_METHOD to workRequest.apiMethod,
                WORK_DATA_PARAMETERS to parametersJson
            )
            
            // Create constraints
            val constraints = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build()
            
            // Create one-time work request with exponential backoff
            val workRequestBuilder = OneTimeWorkRequestBuilder<ApiWorker>()
                .setBackoffCriteria(
                    BackoffPolicy.EXPONENTIAL,
                    1, // Initial delay of 1 seconds
                    TimeUnit.SECONDS
                )
                .setConstraints(constraints)
                .setInputData(inputData)
                .addTag(WORK_TAG)
            
            // If immediate execution is requested, set no initial delay
            if (immediate) {
                workRequestBuilder.setInitialDelay(0, TimeUnit.SECONDS)
            }
            
            val builtWorkRequest = workRequestBuilder.build()
            WorkManager.getInstance().enqueue(builtWorkRequest)
        }
        
        /**
         * Extract work data from WorkerParameters
         * @param workerParameters The worker parameters containing input data
         * @return LinkRunnerWorkRequest or null if data is invalid
         */
        internal fun fromWorkerParameters(workerParameters: WorkerParameters): LinkRunnerWorkRequest? {
            return try {
                val apiMethod = workerParameters.inputData.getString(WORK_DATA_API_METHOD)
                val parametersJson = workerParameters.inputData.getString(WORK_DATA_PARAMETERS)
                
                if (apiMethod.isNullOrBlank() || parametersJson.isNullOrBlank()) {
                    return null
                }
                
                // Deserialize parameters from JSON
                val type = object : TypeToken<Map<String, Any>>() {}.type
                val parameters: Map<String, Any> = gson.fromJson(parametersJson, type) ?: emptyMap()

                android.util.Log.d("LinkRunnerWorkRequest", "API Method: $apiMethod")
                android.util.Log.d("LinkRunnerWorkRequest", "Parameters: $parameters")
                
                LinkRunnerWorkRequest(apiMethod, parameters)
            } catch (e: Exception) {
                android.util.Log.e("LinkRunnerWorkRequest", "Failed to parse work data", e)
                null
            }
        }
        
        /**
         * Cancel all pending work requests
         */
        fun cancelAll() {
            WorkManager.getInstance().cancelAllWorkByTag(WORK_TAG)
        }
    }
}