package io.linkrunner.sdk.work

import android.content.Context
import androidx.work.*
import androidx.work.CoroutineWorker
import io.linkrunner.sdk.LinkRunner
import io.linkrunner.sdk.models.request.*
import io.linkrunner.sdk.network.ApiClient
import io.linkrunner.sdk.network.ApiExecutor
import retrofit2.HttpException
import java.io.IOException
import java.net.SocketTimeoutException
import kotlinx.coroutines.delay

/**
 * Background worker for handling API calls with retry mechanism
 */
class ApiWorker(
    private val context: Context,
    private val workerParams: WorkerParameters
) : CoroutineWorker(context, workerParams) {
    
    companion object {
        private const val TAG = "ApiWorker"
        private const val MAX_RETRIES = 3
    }
    
    override suspend fun doWork(): Result {
        // Check retry count - stop if we've exceeded max retries
        if (runAttemptCount >= MAX_RETRIES) {
            android.util.Log.w(TAG, "Max retries ($MAX_RETRIES) exceeded for work")
            return Result.failure(createErrorOutput("Max retries exceeded"))
        }
        
        val workRequest = LinkRunnerWorkRequest.fromWorkerParameters(workerParams)

        android.util.Log.d(TAG, "Work request: $workRequest")

        if (workRequest == null) {
            android.util.Log.e(TAG, "Invalid work request data")
            return Result.failure(createErrorOutput("Invalid work request data"))
        }
        
        return try {
            android.util.Log.d(TAG, "Executing work request: ${workRequest.apiMethod}, attempt: ${runAttemptCount + 1}")
            
            // Execute the appropriate API call - this will throw exceptions for HTTP errors
            executeApiCall(workRequest)
            
            android.util.Log.d(TAG, "Work request completed successfully: ${workRequest.apiMethod}")
            Result.success(createSuccessOutput())
            
        } catch (exception: Exception) {
            android.util.Log.e(TAG, "Work request exception: ${workRequest.apiMethod}", exception)
            handleException(exception)
        }
    }
    
    
    /**
     * Execute the appropriate API call based on the work request
     * Uses shared ApiExecutor to avoid code duplication
     * @throws HttpException for HTTP errors that should be handled by retry logic
     * @throws IOException for network errors that should be handled by retry logic
     */
    private suspend fun executeApiCall(workRequest: LinkRunnerWorkRequest): Boolean {
        val apiExecutor = ApiExecutor(context)
        return apiExecutor.executeApiCall(workRequest.apiMethod, workRequest.parameters)
    }
    
    /**
     * Handle exceptions and determine retry behavior
     */
    private fun handleException(exception: Exception): Result {
        return when (exception) {
            is IOException, is SocketTimeoutException -> {
                // Network-related issues - retry
                android.util.Log.w(TAG, "Network error, will retry: ${exception.message}")
                if (runAttemptCount < MAX_RETRIES - 1) {
                    Result.retry()
                } else {
                    Result.failure(createErrorOutput(exception.message ?: "Network error"))
                }
            }
            is HttpException -> {
                when (exception.code()) {
                    in 500..599, 408, 429 -> {
                        // Server errors, request timeout, too many requests - retry
                        android.util.Log.w(TAG, "Retriable HTTP error ${exception.code()}, will retry")
                        if (runAttemptCount < MAX_RETRIES - 1) {
                            Result.retry()
                        } else {
                            Result.failure(createErrorOutput("HTTP error ${exception.code()}"))
                        }
                    }
                    else -> {
                        // Client errors (4xx except 408/429) - don't retry
                        android.util.Log.e(TAG, "Non-retriable HTTP error ${exception.code()}")
                        Result.failure(createErrorOutput("HTTP error ${exception.code()}"))
                    }
                }
            }
            else -> {
                // Other exceptions - don't retry
                android.util.Log.e(TAG, "Non-retriable error: ${exception.message}")
                Result.failure(createErrorOutput(exception.message ?: "Unknown error"))
            }
        }
    }
    
    
    /**
     * Create error output data
     */
    private fun createErrorOutput(error: String): Data {
        return workDataOf(
            "success" to false,
            "error" to error,
            "attempt" to runAttemptCount + 1
        )
    }
    
    /**
     * Create success output data
     */
    private fun createSuccessOutput(): Data {
        return workDataOf(
            "success" to true,
            "attempt" to runAttemptCount + 1
        )
    }
}