package io.linkrunner.sdk.work

import io.linkrunner.sdk.models.request.CapturePaymentRequest
import io.linkrunner.sdk.models.request.RemovePaymentRequest
import io.linkrunner.sdk.models.request.UserDataRequest
import io.linkrunner.sdk.models.IntegrationData
import io.linkrunner.sdk.network.ApiExecutor
import android.content.Context


/**
 * Scheduler for API work requests with background retry mechanism
 */
object ApiWorkScheduler {
    
    private const val TAG = "ApiWorkScheduler"
    
    /**
     * Enqueue an initialization work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueInitialize(
        context: Context,
        token: String,
        packageVersion: String,
        appVersion: String,
        deviceData: Map<String, Any>,
        platform: String,
        installInstanceId: String,
        link: String? = null,
        source: String? = null,
        debug: Boolean? = null
    ): Boolean {
        val parameters = mutableMapOf<String, Any>(
            "token" to token,
            "packageVersion" to packageVersion,
            "appVersion" to appVersion,
            "deviceData" to deviceData,
            "platform" to platform,
            "installInstanceId" to installInstanceId
        )
        
        link?.let { parameters["link"] = it }
        source?.let { parameters["source"] = it }
        debug?.let { parameters["debug"] = it }
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_INITIALIZE,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued initialize work")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue initialize work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_INITIALIZE, parameters)
                android.util.Log.d(TAG, "Direct API call successful for initialize")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for initialize", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue a send event work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param eventName Name of the event to track
     * @param eventData Optional additional event data
     * @param eventId Optional unique event identifier for deduplication
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueSendEvent(context: Context, eventName: String, eventData: Map<String, Any>? = null, eventId: String? = null): Boolean {
        val parameters = mutableMapOf<String, Any>(
            "eventName" to eventName
        )
        
        if (eventData != null) {
            parameters["eventData"] = eventData
        }
        if (eventId != null) {
            parameters["eventId"] = eventId
        }
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_SEND_EVENT,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued send event work: $eventName")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue send event work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_SEND_EVENT, parameters)
                android.util.Log.d(TAG, "Direct API call successful for send event: $eventName")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for send event: $eventName", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue a signup work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param userData User data to be associated with the signup
     * @param additionalData Additional custom data to be sent with the event
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueSignup(context: Context, userData: UserDataRequest, additionalData: Map<String, Any>? = null): Boolean {
        val parameters = mutableMapOf<String, Any>(
            "userData" to userDataToMap(userData)
        )
        
        if (additionalData != null) {
            parameters["additionalData"] = additionalData
        }
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_SIGNUP,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued signup work for user: ${userData.id}")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue signup work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_SIGNUP, parameters)
                android.util.Log.d(TAG, "Direct API call successful for signup: ${userData.id}")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for signup: ${userData.id}", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue a capture payment work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param paymentData Payment data to be tracked
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueCapturePayment(context: Context, paymentData: CapturePaymentRequest): Boolean {
        val parameters = mapOf<String, Any>(
            "paymentData" to capturePaymentToMap(paymentData)
        )
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_CAPTURE_PAYMENT,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued capture payment work: ${paymentData.paymentId}")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue capture payment work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_CAPTURE_PAYMENT, parameters)
                android.util.Log.d(TAG, "Direct API call successful for capture payment: ${paymentData.paymentId}")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for capture payment: ${paymentData.paymentId}", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue a remove payment work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param removeData Payment data to be removed
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueRemovePayment(context: Context, removeData: RemovePaymentRequest): Boolean {
        val parameters = mapOf<String, Any>(
            "removeData" to removePaymentToMap(removeData)
        )
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_REMOVE_PAYMENT,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued remove payment work: ${removeData.paymentId}")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue remove payment work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_REMOVE_PAYMENT, parameters)
                android.util.Log.d(TAG, "Direct API call successful for remove payment: ${removeData.paymentId}")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for remove payment: ${removeData.paymentId}", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue a set user data work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param userData User data to be associated with the session
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueSetUserData(context: Context, userData: UserDataRequest): Boolean {
        val parameters = mapOf<String, Any>(
            "userData" to userDataToMap(userData)
        )
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_SET_USER_DATA,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued set user data work for user: ${userData.id}")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue set user data work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_SET_USER_DATA, parameters)
                android.util.Log.d(TAG, "Direct API call successful for set user data: ${userData.id}")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for set user data: ${userData.id}", fallbackException)
                false
            }
        }
    }
   
    
    /**
     * Cancel all pending work requests
     */
    fun cancelAllWork() {
        try {
            LinkRunnerWorkRequest.cancelAll()
            android.util.Log.d(TAG, "Cancelled all pending work requests")
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to cancel work requests", e)
        }
    }
    
    // Helper methods to convert objects to maps for serialization
    
    private fun userDataToMap(userData: UserDataRequest): Map<String, Any?> {
        return mapOf(
            "id" to userData.id,
            "name" to userData.name,
            "email" to userData.email,
            "phone" to userData.phone,
            "mixpanelDistinctId" to userData.mixpanelDistinctId,
            "amplitudeDeviceId" to userData.amplitudeDeviceId,
            "posthogDistinctId" to userData.posthogDistinctId,
            "brazeDeviceId" to userData.brazeDeviceId,
            "gaAppInstanceId" to userData.gaAppInstanceId,
            "userCreatedAt" to userData.userCreatedAt,
            "isFirstTimeUser" to userData.isFirstTimeUser
        )
    }
    
    private fun capturePaymentToMap(paymentData: CapturePaymentRequest): Map<String, Any?> {
        return mapOf(
            "paymentId" to paymentData.paymentId,
            "userId" to paymentData.userId,
            "amount" to paymentData.amount,
            "type" to paymentData.type.value,
            "status" to paymentData.status.value
        )
    }
    
    private fun removePaymentToMap(removeData: RemovePaymentRequest): Map<String, Any?> {
        return mapOf(
            "paymentId" to removeData.paymentId,
            "userId" to removeData.userId
        )
    }
    
    /**
     * Enqueue an integration data work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param integrationData Integration data to be sent
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueSetAdditionalData(context: Context, integrationData: IntegrationData): Boolean {
        val parameters = mapOf<String, Any>(
            "integrationData" to integrationData.toMap()
        )
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_UPDATE_INTEGRATIONS,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued set additional data work")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue set additional data work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_UPDATE_INTEGRATIONS, parameters)
                android.util.Log.d(TAG, "Direct API call successful for set additional data")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for set additional data", fallbackException)
                false
            }
        }
    }
    
    /**
     * Enqueue an update push token work request with fallback to direct API call
     * @param context Application context for fallback API call
     * @param pushToken Push notification token to be updated
     * @return true if successful (either enqueued or direct API call succeeded), false otherwise
     */
    suspend fun enqueueUpdatePushToken(context: Context, pushToken: String): Boolean {
        val parameters = mapOf<String, Any>(
            "pushToken" to pushToken
        )
        
        val workRequest = LinkRunnerWorkRequest(
            LinkRunnerWorkRequest.KIND_UPDATE_PUSH_TOKEN,
            parameters
        )
        
        return try {
            LinkRunnerWorkRequest.enqueue(workRequest)
            android.util.Log.d(TAG, "Enqueued update push token work")
            true
        } catch (e: Exception) {
            android.util.Log.w(TAG, "Failed to enqueue update push token work, falling back to direct API call", e)
            // Fallback: Direct API call using ApiExecutor
            try {
                val apiExecutor = ApiExecutor(context)
                apiExecutor.executeApiCall(LinkRunnerWorkRequest.KIND_UPDATE_PUSH_TOKEN, parameters)
                android.util.Log.d(TAG, "Direct API call successful for update push token")
                true
            } catch (fallbackException: Exception) {
                android.util.Log.e(TAG, "Both enqueue and direct API call failed for update push token", fallbackException)
                false
            }
        }
    }
}