package io.linkrunner.sdk.utils
import android.content.Context
import android.util.Log
import okhttp3.Interceptor
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.Response
import okio.Buffer
import java.io.IOException
import java.util.concurrent.ExecutorService
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.ThreadFactory
import java.util.concurrent.ThreadPoolExecutor
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import java.util.concurrent.TimeUnit
import kotlin.math.max

class HmacSigningInterceptor(
    private val context: Context,
    private val bundleId: String,
    private val secretKey: String,
    private val keyId: String,
    private val signingExecutor: ExecutorService = createSigningExecutor()
) : Interceptor {

    private val TAG = "HmacSigning"

    companion object {
        private const val AUTHORIZATION_HEADER = "Authorization"
        private const val TIMESTAMP_HEADER = "x-Timestamp"
        private const val SIGNATURE_HEADER = "x-Signature"
        private const val KEY_ID_HEADER = "x-Key-Id"

        private fun createSigningExecutor(): ExecutorService {
            val coreThreads = max(2, Runtime.getRuntime().availableProcessors() / 2)
            return ThreadPoolExecutor(
                coreThreads,
                coreThreads * 2,
                60L,
                TimeUnit.SECONDS,
                LinkedBlockingQueue(100),
                ThreadFactory { r ->
                    Thread(r, "hmac-signing-${Thread.currentThread().id}").apply {
                        isDaemon = true
                    }
                },
                ThreadPoolExecutor.CallerRunsPolicy()
            )
        }
    }

    private val signatureGenerator: HmacSignatureGenerator? by lazy {

        if (keyId.isBlank() || secretKey.isBlank()) {
            return@lazy null
        }
        try {
            SecureKeyManager.reconstructSigningKey(context, bundleId, secretKey)?.let { keyBytes ->
                HmacSignatureGenerator(keyBytes, keyId)
            } ?: run {
                null
            }
        } catch (e: Exception) {
            null
        }
    }

    override fun intercept(chain: Interceptor.Chain): Response {


        val originalRequest = chain.request()

        val generator = signatureGenerator ?: run {
            return chain.proceed(originalRequest)
        }

        return try {
            val requestBody = originalRequest.body?.let { readRequestBody(it) }

            val signedRequest = generator.signRequest(requestBody) {}

            val finalRequest = addSignatureHeaders(
                originalRequest,
                signedRequest.signature,
                signedRequest.timestamp
            )
            chain.proceed(finalRequest)
        } catch (e: Exception) {
            // If signing fails, proceed with the original request without signature
            chain.proceed(originalRequest)
        }
    }

    private fun addSignatureHeaders(
        request: Request,
        signature: String,
        timestamp: Long
    ): Request {
        return request.newBuilder()
            .header(AUTHORIZATION_HEADER, "HMAC")
            .header(KEY_ID_HEADER, keyId)
            .header(TIMESTAMP_HEADER, timestamp.toString())
            .header(SIGNATURE_HEADER, signature)
            .build()
    }

    private fun readRequestBody(requestBody: RequestBody): String {
        return try {
            val buffer = Buffer()
            requestBody.writeTo(buffer)
            buffer.readUtf8()
        } catch (e: IOException) {
            ""
        }
    }
}
