package io.linkrunner.sdk.utils

import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.os.Build
import android.os.Looper
import android.util.Log
import android.provider.Settings
import android.telephony.TelephonyManager
import android.util.DisplayMetrics
import android.view.WindowManager
import android.webkit.WebSettings
import android.webkit.WebView
import com.android.installreferrer.api.InstallReferrerClient
import com.android.installreferrer.api.InstallReferrerStateListener
import com.android.installreferrer.api.ReferrerDetails
import com.google.android.gms.ads.identifier.AdvertisingIdClient
//import com.google.android.gms.common.GooglePlayServicesNotAvailableException
//import com.google.android.gms.common.GooglePlayServicesRepairableException
import io.linkrunner.sdk.BuildConfig
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import java.io.IOException
import java.net.InetAddress
import java.net.NetworkInterface
import java.util.*
import kotlin.coroutines.resume

/**
 * Utility class to provide device information
 */
internal class DeviceInfoProvider(private val context: Context) {
    
    private val packageManager: PackageManager = context.packageManager
    private val packageName: String = context.packageName
    private val preferenceManager = PreferenceManager(context)
    
    companion object {
        private const val TAG = "DeviceInfoProvider"
        private const val INSTALL_REFERRER_CACHE_KEY = "cached_install_referrer_data"
        private const val INSTALL_REFERRER_CACHE_TIMESTAMP = "cached_install_referrer_timestamp"
        private const val CACHE_VALIDITY_HOURS = 24 // Cache for 24 hours
    }
    
    /**
     * Get device information as a map
     */
    @SuppressLint("HardwareIds")
    suspend fun getDeviceInfo(): Map<String, Any> = withContext(Dispatchers.IO) {
        val deviceInfo = mutableMapOf<String, Any>()
        
        try {
            // App info
            val packageInfo: PackageInfo = packageManager.getPackageInfo(packageName, 0)
            deviceInfo["application_name"] = getApplicationName()
            deviceInfo["app_version"] = packageInfo.versionName ?: ""
            deviceInfo["build_number"] = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                packageInfo.longVersionCode.toString()
            } else {
                @Suppress("DEPRECATION")
                packageInfo.versionCode.toString()
            }
            deviceInfo["bundle_id"] = packageName
            deviceInfo["version"] = packageInfo.versionName ?: ""

            // Device info
            deviceInfo["device_id"] = getDeviceId()
            deviceInfo["device_name"] = "${Build.MANUFACTURER} ${Build.MODEL}"
            deviceInfo["manufacturer"] = Build.MANUFACTURER
            deviceInfo["brand"] = Build.BRAND
            deviceInfo["system_version"] = Build.VERSION.RELEASE ?: ""
            deviceInfo["version"] = packageInfo.versionName ?: ""
            deviceInfo["connectivity"] = getNetworkType()
            deviceInfo["user_agent"] = getUserAgent()
            deviceInfo["gaid"] = getAdvertisingId() ?: ""
            // idfa and idfv are iOS only, set as empty string for Android
            deviceInfo["idfa"] = ""
            deviceInfo["idfv"] = ""

            // Carrier info as array
            val carrierName = getCarrierName()
            deviceInfo["carrier"] = if (carrierName.isNotEmpty()) listOf(carrierName) else listOf<String>()

            // IP Address
            getIpAddress()?.let { ipAddress ->
                deviceInfo["device_ip"] = ipAddress
            }

            // Play Store details (install referrer) - non-blocking
            val installReferrerData = getInstallReferrerData()
            deviceInfo.putAll(installReferrerData)
           
        } catch (e: Exception) {
            if (BuildConfig.DEBUG) {
                e.printStackTrace()
            }
        }
        
        return@withContext deviceInfo
    }
    
    /**
     * Get install referrer data - this includes cached data, fresh data, or error info
     * This method never throws exceptions that would block API calls
     */
    private suspend fun getInstallReferrerData(): Map<String, Any> {
        val referrerData = mutableMapOf<String, Any>()
        
        try {
            // First check if we have cached data
            val cachedData = getCachedInstallReferrerData()
            if (cachedData != null) {
                Log.d(TAG, "Using cached install referrer data")
                referrerData.putAll(cachedData)
                return referrerData
            }
            
            // If no cached data, try to get fresh data
            Log.d(TAG, "No cached install referrer data, attempting fresh fetch")
            val installReferrerInfo = getInstallReferrerInfo()
            
            if (installReferrerInfo != null) {
                // Got fresh data - cache it and use it
                val freshData = extractReferrerData(installReferrerInfo)
                cacheInstallReferrerData(freshData)
                referrerData.putAll(freshData)
                Log.d(TAG, "Successfully fetched and cached fresh install referrer data")
            } else {
                // Failed to get data - include error info if available
                getInstallReferrerErrorInfo()?.let { errorInfo ->
                    referrerData["install_referrer_error"] = errorInfo
                    Log.d(TAG, "Including install referrer error info in device data")
                }
            }
            
        } catch (e: Exception) {
            // Store this exception too
            storeInstallReferrerException(e)
            getInstallReferrerErrorInfo()?.let { errorInfo ->
                referrerData["install_referrer_error"] = errorInfo
            }
        }
        
        return referrerData
    }
    
    /**
     * Get cached install referrer data if valid
     */
    private fun getCachedInstallReferrerData(): Map<String, Any>? {
        return try {
            val cachedJson = preferenceManager.getString(INSTALL_REFERRER_CACHE_KEY)
            val cacheTimestamp = preferenceManager.getString(INSTALL_REFERRER_CACHE_TIMESTAMP).toLongOrNull() ?: 0
            
            if (cachedJson.isNotEmpty() && cacheTimestamp > 0) {
                val cacheAgeHours = (System.currentTimeMillis() - cacheTimestamp) / (1000 * 60 * 60)
                if (cacheAgeHours < CACHE_VALIDITY_HOURS) {
                    val gson = com.google.gson.Gson()
                    @Suppress("UNCHECKED_CAST")
                    return gson.fromJson(cachedJson, Map::class.java) as? Map<String, Any>
                } else {
                    // Clear stale cache
                    preferenceManager.remove(INSTALL_REFERRER_CACHE_KEY)
                    preferenceManager.remove(INSTALL_REFERRER_CACHE_TIMESTAMP)
                }
            }
            null
        } catch (e: Exception) {
            Log.w(TAG, "Failed to read cached install referrer data: ${e.message}")
            null
        }
    }
    
    /**
     * Cache install referrer data
     */
    private fun cacheInstallReferrerData(data: Map<String, Any>) {
        try {
            val gson = com.google.gson.Gson()
            val jsonData = gson.toJson(data)
            preferenceManager.saveString(INSTALL_REFERRER_CACHE_KEY, jsonData)
            preferenceManager.saveString(INSTALL_REFERRER_CACHE_TIMESTAMP, System.currentTimeMillis().toString())
            Log.d(TAG, "Cached install referrer data successfully")
        } catch (e: Exception) {
            Log.w(TAG, "Failed to cache install referrer data: ${e.message}")
        }
    }
    
    /**
     * Extract referrer data from ReferrerDetails
     */
    private fun extractReferrerData(referrerInfo: ReferrerDetails): Map<String, Any> {
        val data = mutableMapOf<String, Any>()
        
        data["install_ref"] = referrerInfo.installReferrer ?: ""
        val installReferrerStr = referrerInfo.installReferrer ?: ""
        data["install_ref_url"] = try {
            java.net.URL(installReferrerStr).toString()
        } catch (e: Exception) {
            ""
        }
        
        data["install_ref_hashCode"] = installReferrerStr.hashCode()
        data["install_ref_install_version"] = referrerInfo.installVersion ?: ""
        data["install_ref_installBeginTimestampSeconds"] = referrerInfo.installBeginTimestampSeconds
        data["install_ref_referrerClickTimestampSeconds"] = referrerInfo.referrerClickTimestampSeconds
        data["installBeginTimestampServerSeconds"] = referrerInfo.installBeginTimestampServerSeconds
        data["referrerClickTimestampServerSeconds"] = referrerInfo.referrerClickTimestampServerSeconds
        data["install_ref_googlePlayInstantParam"] = referrerInfo.googlePlayInstantParam
        
        return data
    }
    
    /**
     * Get Play Store install referrer information with retries
     * This method uses retries only for connection issues, never blocks API calls
     */
    private suspend fun getInstallReferrerInfo(): ReferrerDetails? = withContext(Dispatchers.IO) {
        val currentThread = Thread.currentThread()
        Log.d(TAG, "Install referrer executing on thread: ${currentThread.name} (ID: ${currentThread.id})")
        Log.d(TAG, "Is main thread: ${currentThread == Looper.getMainLooper().thread}")
        
        val maxRetries = 2
        val baseDelayMs = 1000L // 1 second base delay
        
        for (attempt in 1..maxRetries) {
            Log.d(TAG, "Install referrer attempt $attempt of $maxRetries")
            
            val result = tryGetInstallReferrer(attempt)
            when {
                result.isSuccess -> {
                    Log.d(TAG, "Install referrer successful on attempt $attempt")
                    return@withContext result.getOrNull()
                }
                attempt < maxRetries -> {
                    val delayMs = baseDelayMs * (1L shl (attempt - 1)) // Exponential backoff: 1s, 2s
                    Log.w(TAG, "Install referrer attempt $attempt failed, retrying in ${delayMs}ms. Error: ${result.exceptionOrNull()?.message}")
                    delay(delayMs)
                }
                else -> {
                    val exception = result.exceptionOrNull()
                    Log.w(TAG, "Install referrer failed after $maxRetries attempts. Final error: ${exception?.message}")
                    
                    // Store the exception for inclusion in API requests
                    storeInstallReferrerException(exception)
                    return@withContext null
                }
            }
        }
        
        return@withContext null
    }
    
    /**
     * Attempt to get install referrer with timeout - runs on separate thread
     */
    private suspend fun tryGetInstallReferrer(attempt: Int): Result<ReferrerDetails?> = withContext(Dispatchers.IO) {
        suspendCancellableCoroutine { continuation ->
            val timeoutMs = 10000L // 10 seconds timeout
            
            try {
                val referrerClient = InstallReferrerClient.newBuilder(context).build()
                
                // Setup timeout handler
                val timeoutHandler = android.os.Handler(Looper.getMainLooper())
                val timeoutRunnable = Runnable {
                    Log.w(TAG, "Install referrer connection timed out after ${timeoutMs}ms")
                    try {
                        referrerClient.endConnection()
                    } catch (e: Exception) {
                        Log.w(TAG, "Error ending timed-out connection: ${e.message}")
                    }
                    if (continuation.isActive) {
                        continuation.resume(Result.success(null))
                    }
                }
                
                timeoutHandler.postDelayed(timeoutRunnable, timeoutMs)
                
                referrerClient.startConnection(object : InstallReferrerStateListener {
                    override fun onInstallReferrerSetupFinished(responseCode: Int) {
                        
                        // Cancel Timeout
                        timeoutHandler.removeCallbacks(timeoutRunnable)
                        
                        when (responseCode) {
                            InstallReferrerClient.InstallReferrerResponse.OK -> {
                                try {
                                    val response = referrerClient.installReferrer
                                    if (continuation.isActive) {
                                        continuation.resume(Result.success(response))
                                    }
                                } catch (e: Exception) {
                                    Log.w(TAG, "Error retrieving install referrer data: ${e.message}")
                                    if (continuation.isActive) {
                                        continuation.resume(Result.success(null))
                                    }
                                } finally {
                                    try {
                                        referrerClient.endConnection()
                                    } catch (e: Exception) {
                                        Log.w(TAG, "Error ending connection: ${e.message}")
                                    }
                                }
                            }
                            InstallReferrerClient.InstallReferrerResponse.FEATURE_NOT_SUPPORTED -> {
                                Log.w(TAG, "Install referrer feature not supported")
                                try {
                                    referrerClient.endConnection()
                                } catch (e: Exception) {
                                    Log.w(TAG, "Error ending connection: ${e.message}")
                                }
                                if (continuation.isActive) {
                                    continuation.resume(Result.success(null))
                                }
                            }
                            InstallReferrerClient.InstallReferrerResponse.SERVICE_UNAVAILABLE -> {
                                Log.w(TAG, "Install referrer service unavailable")
                                try {
                                    referrerClient.endConnection()
                                } catch (e: Exception) {
                                    Log.w(TAG, "Error ending connection: ${e.message}")
                                }
                                if (continuation.isActive) {
                                    continuation.resume(Result.success(null))
                                }
                            }
                            else -> {
                                Log.w(TAG, "Install referrer unknown response code: $responseCode")
                                try {
                                    referrerClient.endConnection()
                                } catch (e: Exception) {
                                    Log.w(TAG, "Error ending connection: ${e.message}")
                                }
                                if (continuation.isActive) {
                                    continuation.resume(Result.success(null))
                                }
                            }
                        }
                    }
                    
                    override fun onInstallReferrerServiceDisconnected() {
                        // Cancel timeout
                        timeoutHandler.removeCallbacks(timeoutRunnable)
                        
                        if (continuation.isActive) {
                            continuation.resume(Result.success(null))
                        }
                    }
                })
                
                continuation.invokeOnCancellation {
                    Log.d(TAG, "Install referrer operation cancelled")
                    timeoutHandler.removeCallbacks(timeoutRunnable)
                    try {
                        referrerClient.endConnection()
                    } catch (e: Exception) {
                        Log.w(TAG, "Error ending connection on cancellation: ${e.message}")
                    }
                }
                
            } catch (e: Exception) {
                Log.w(TAG, "Error starting install referrer connection: ${e.message}")
                if (continuation.isActive) {
                    continuation.resume(Result.success(null))
                }
            }
        }
    }
    
    /**
     * Store install referrer exception for inclusion in API requests
     * Only stores the error message in a simple, safe format
     */
    private fun storeInstallReferrerException(exception: Throwable?) {
        if (exception == null) return
        
        try {
            val sharedPrefs = context.getSharedPreferences("linkrunner_referrer_errors", Context.MODE_PRIVATE)
            
            // Extract only the error message, with safe fallbacks
            val errorMessage = when {
                exception.message?.isNotBlank() == true -> exception.message!!
                exception.cause?.message?.isNotBlank() == true -> exception.cause!!.message!!
                else -> "Install referrer connection failed"
            }
            
            // Store just the error message - no JSON formatting to avoid errors
            sharedPrefs.edit()
                .putString("last_install_referrer_error", errorMessage)
                .putLong("error_timestamp", System.currentTimeMillis())
                .apply()
                
            Log.d(TAG, "Stored install referrer error message: $errorMessage")
            
        } catch (e: Exception) {
            // Fallback: store a generic error message if everything fails
            try {
                val sharedPrefs = context.getSharedPreferences("linkrunner_referrer_errors", Context.MODE_PRIVATE)
                sharedPrefs.edit()
                    .putString("last_install_referrer_error", "Install referrer connection error")
                    .putLong("error_timestamp", System.currentTimeMillis())
                    .apply()
            } catch (fallbackException: Exception) {
                // If even the fallback fails, just log and continue
                Log.w(TAG, "Failed to store install referrer error info")
            }
        }
    }
    
    /**
     * Get stored install referrer exception info for API requests as a clean JSON object
     * Returns only the error message in a safe format that never causes exceptions
     */
    fun getInstallReferrerErrorInfo(): Map<String, Any>? {
        return try {
            val sharedPrefs = context.getSharedPreferences("linkrunner_referrer_errors", Context.MODE_PRIVATE)
            val errorMessage = sharedPrefs.getString("last_install_referrer_error", null)
            val errorTimestamp = sharedPrefs.getLong("error_timestamp", 0)
            
            if (!errorMessage.isNullOrBlank() && errorTimestamp > 0) {
                // Only include errors from the last 24 hours to avoid stale error data
                val twentyFourHoursAgo = System.currentTimeMillis() - (24 * 60 * 60 * 1000)
                if (errorTimestamp > twentyFourHoursAgo) {
                    // Create a simple, safe JSON object with just the error message
                    val errorInfo = mapOf(
                        "error" to errorMessage.trim()
                    )
                    return errorInfo
                } else {
                    // Clear stale error data
                    try {
                        sharedPrefs.edit().clear().apply()
                    } catch (clearException: Exception) {
                        // Ignore cleanup errors
                    }
                }
            }
            
            null
        } catch (e: Exception) {
            // Return a fallback error object if everything fails
            Log.w(TAG, "Failed to retrieve install referrer error info, using fallback")
            return try {
                mapOf("error" to "Install referrer error")
            } catch (fallbackException: Exception) {
                // If even the fallback fails, return null to avoid breaking API calls
                null
            }
        }
    }
    
    private fun getApplicationName(): String {
        return try {
            val applicationInfo = packageManager.getApplicationInfo(packageName, 0)
            packageManager.getApplicationLabel(applicationInfo).toString()
        } catch (e: Exception) {
            ""
        }
    }
    
    @SuppressLint("HardwareIds")
    private fun getDeviceId(): String {
        return Settings.Secure.getString(
            context.contentResolver,
            Settings.Secure.ANDROID_ID
        ) ?: UUID.randomUUID().toString()
    }
    
    private fun getNetworkType(): String {
        return try {
            val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as android.net.ConnectivityManager
            
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                val network = connectivityManager.activeNetwork ?: return "Not Connected"
                val capabilities = connectivityManager.getNetworkCapabilities(network) ?: return "Unknown"
                
                return when {
                    capabilities.hasTransport(android.net.NetworkCapabilities.TRANSPORT_WIFI) -> "Wi-Fi"
                    capabilities.hasTransport(android.net.NetworkCapabilities.TRANSPORT_CELLULAR) -> "Mobile Network"
                    capabilities.hasTransport(android.net.NetworkCapabilities.TRANSPORT_ETHERNET) -> "Ethernet"
                    else -> "Unknown"
                }
            } else {
                @Suppress("DEPRECATION")
                val networkInfo = connectivityManager.activeNetworkInfo
                @Suppress("DEPRECATION")
                when (networkInfo?.type) {
                    android.net.ConnectivityManager.TYPE_WIFI -> "Wi-Fi"
                    android.net.ConnectivityManager.TYPE_MOBILE -> "Mobile Network"
                    else -> "Unknown"
                }
            }
        } catch (e: Exception) {
            "Unknown"
        }
    }
    
    private suspend fun getAdvertisingId(): String? = withContext(Dispatchers.IO) {
        return@withContext try {
            val advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(context)
            if (advertisingIdInfo.isLimitAdTrackingEnabled) {
                null
            } else {
                advertisingIdInfo.id
            }
        } catch (e: Exception) {
            if (BuildConfig.DEBUG) {
                e.printStackTrace()
            }
            null
        }
    }
    
    /**
     * Get display information of the device
     */
    private fun getDisplayInfo(): String {
        return try {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val metrics = DisplayMetrics()
            @Suppress("DEPRECATION")
            windowManager.defaultDisplay.getMetrics(metrics)
            
            "${metrics.widthPixels}x${metrics.heightPixels} (${metrics.densityDpi} dpi)"
        } catch (e: Exception) {
            ""
        }
    }
    
    /**
     * Get device type - phone, tablet, etc.
     */
    private fun getDeviceType(): String {
        return try {
            val metrics = context.resources.displayMetrics
            val widthInches = metrics.widthPixels / metrics.xdpi
            val heightInches = metrics.heightPixels / metrics.ydpi
            val diagonalInches = Math.sqrt((widthInches * widthInches + heightInches * heightInches).toDouble())
            
            // Tablet is typically a device with screen larger than 7 inches diagonally
            if (diagonalInches >= 7.0) "Tablet" else "Phone"
        } catch (e: Exception) {
            "Unknown"
        }
    }
    
    /**
     * Get device user agent
     */
    private fun getUserAgent(): String {
        return try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                WebSettings.getDefaultUserAgent(context)
            } else {
                @Suppress("DEPRECATION")
                WebView(context).settings.userAgentString ?: ""
            }
        } catch (e: Exception) {
            // Fallback user agent if WebView is not available
            "Mozilla/5.0 (Linux; Android ${Build.VERSION.RELEASE}; ${Build.MODEL} Build/${Build.ID})"
        }
    }
    
    /**
     * Get mobile carrier name
     */
    private fun getCarrierName(): String {
        return try {
            val telephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
            telephonyManager.networkOperatorName ?: ""
        } catch (e: Exception) {
            ""
        }
    }
    
    /**
     * Get device IP address
     */
    private fun getIpAddress(): String? {
        return try {
            // Check Wi-Fi first
            val wifiManager = context.applicationContext.getSystemService(Context.WIFI_SERVICE)
            if (wifiManager != null) {
                val wifiInfo = Class.forName("android.net.wifi.WifiManager").getMethod("getConnectionInfo").invoke(wifiManager)
                if (wifiInfo != null) {
                    val ipAddress = Class.forName("android.net.wifi.WifiInfo").getMethod("getIpAddress").invoke(wifiInfo) as Int
                    if (ipAddress != 0) {
                        return String.format(
                            Locale.US,
                            "%d.%d.%d.%d",
                            ipAddress and 0xff,
                            ipAddress shr 8 and 0xff,
                            ipAddress shr 16 and 0xff,
                            ipAddress shr 24 and 0xff
                        )
                    }
                }
            }
            
            // Otherwise try to get IP from network interfaces
            val interfaces = NetworkInterface.getNetworkInterfaces()
            while (interfaces.hasMoreElements()) {
                val networkInterface = interfaces.nextElement()
                val addresses = networkInterface.inetAddresses
                while (addresses.hasMoreElements()) {
                    val address = addresses.nextElement()
                    if (!address.isLoopbackAddress && address is InetAddress) {
                        return address.hostAddress
                    }
                }
            }
            null
        } catch (e: Exception) {
            null
        }
    }
}