package io.linkrunner.sdk.network

import android.content.Context
import io.linkrunner.sdk.models.request.*
import io.linkrunner.sdk.utils.DeviceInfoProvider
import io.linkrunner.sdk.utils.PreferenceManager
import retrofit2.HttpException

/**
 * Shared API execution logic to avoid code duplication between ApiWorker and LinkRunner
 */
class ApiExecutor(private val context: Context) {
    
    companion object {
        private const val TAG = "ApiExecutor"
    }
    
    /**
     * Execute an API call based on the method and parameters
     * @param apiMethod The API method to execute
     * @param parameters The parameters for the API call
     * @throws HttpException for HTTP errors
     * @throws Exception for other errors
     */
    suspend fun executeApiCall(apiMethod: String, parameters: Map<String, Any>): Boolean {
        // Get common required parameters from preferences
        val preferenceManager = PreferenceManager(context)
        val token = preferenceManager.getString("linkrunner_token")
        val installInstanceId = preferenceManager.getString("install_instance_id")
        val configPlatform = preferenceManager.getString("linkrunner_platform").takeIf { it.isNotEmpty() } ?: "ANDROID"
        val configPackageVersion = preferenceManager.getString("linkrunner_package_version").takeIf { it.isNotEmpty() } ?: "3.3.1"
        val configDebug = preferenceManager.getBoolean("linkrunner_debug")
        
        if (token.isEmpty()) {
            android.util.Log.e(TAG, "Token not found in preferences")
            throw IllegalStateException("Token not found in preferences")
        }
        
        // Create ApiClient and get device info
        val apiClient = ApiClient(context)
        
        // Set SDK signature if credentials are available in preferences
        val keyId = preferenceManager.getString("linkrunner_key_id")
        val secretKey = preferenceManager.getString("linkrunner_secret_key")
        if (keyId.isNotEmpty() && secretKey.isNotEmpty()) {
            apiClient.setSDKSignature(keyId, secretKey)
            android.util.Log.d(TAG, "Set SDK signature for ApiExecutor ApiClient")
        }
        
        val deviceInfoProvider = DeviceInfoProvider(context)
        val deviceData = deviceInfoProvider.getDeviceInfo()
        
        return when (apiMethod) {
            "initialize" -> {
                // Parse initialization parameters
                val packageVersion = parameters["packageVersion"] as? String ?: ""
                val appVersion = parameters["appVersion"] as? String ?: ""
                val platform = parameters["platform"] as? String ?: "ANDROID"
                val link = parameters["link"] as? String
                val source = parameters["source"] as? String
                val debug = parameters["debug"] as? Boolean
                
                // Create InitRequest
                val request = InitRequest(
                    token = token,
                    package_version = packageVersion,
                    app_version = appVersion,
                    device_data = deviceData,
                    platform = platform,
                    install_instance_id = installInstanceId,
                    link = link,
                    source = source,
                    debug = debug
                )
                
                android.util.Log.d(TAG, "Initialize request: ${request}")
                val response = apiClient.apiService.initialize(request)
                android.util.Log.d(TAG, "Initialize response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "send_event" -> {
                val eventName = parameters["eventName"] as? String ?: throw IllegalArgumentException("eventName is required")
                val eventData = parameters["eventData"] as? Map<String, Any>
                val eventId = parameters["eventId"] as? String
                
                val request = TrackEventApiRequest(
                    token = token,
                    event_name = eventName,
                    event_data = eventData,
                    event_id = eventId,
                    platform = configPlatform,
                    install_instance_id = installInstanceId
                )
                android.util.Log.d(TAG, "Track event request: ${request}")
                val response = apiClient.apiService.trackEvent(request)
                android.util.Log.d(TAG, "Track event response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "signup" -> {
                val userData = parseUserDataRequest(parameters["userData"] as? Map<String, Any>) ?: throw IllegalArgumentException("userData is required")
                val additionalData = parameters["additionalData"] as? Map<String, Any> ?: emptyMap()
                
                val userDataMap = mapOf(
                    "id" to userData.id,
                    "name" to userData.name,
                    "email" to userData.email,
                    "phone" to userData.phone,
                    "mixpanel_distinct_id" to userData.mixpanelDistinctId,
                    "amplitude_device_id" to userData.amplitudeDeviceId,
                    "posthog_distinct_id" to userData.posthogDistinctId,
                    "braze_device_id" to userData.brazeDeviceId,
                    "ga_app_instance_id" to userData.gaAppInstanceId,
                    "user_created_at" to userData.userCreatedAt,
                    "is_first_time_user" to userData.isFirstTimeUser
                ).filterValues { it != null }
                
                val request = TriggerRequest(
                    token = token,
                    user_data = userDataMap,
                    platform = "ANDROID",
                    data = deviceData + additionalData,
                    install_instance_id = installInstanceId
                )
                android.util.Log.d(TAG, "Signup request: ${request}")
                val response = apiClient.apiService.trigger(request)
                android.util.Log.d(TAG, "Signup response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "capture_payment" -> {
                val paymentData = parseCapturePaymentRequest(parameters["paymentData"] as? Map<String, Any>) ?: throw IllegalArgumentException("paymentData is required")
                
                val request = CapturePaymentApiRequest(
                    token = token,
                    platform = configPlatform,
                    payment_id = paymentData.paymentId.toString(),
                    user_id = paymentData.userId,
                    amount = paymentData.amount,
                    type = paymentData.type.name,
                    status = paymentData.status.name,
                    install_instance_id = installInstanceId
                )
                android.util.Log.d(TAG, "Capture payment request: ${request}")
                val response = apiClient.apiService.capturePayment(request)
                android.util.Log.d(TAG, "Capture payment response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "remove_payment" -> {
                val removeData = parseRemovePaymentRequest(parameters["removeData"] as? Map<String, Any>) ?: throw IllegalArgumentException("removeData is required")
                
                val request = RemovePaymentApiRequest(
                    token = token,
                    platform = configPlatform,
                    payment_id = removeData.paymentId ?: "",
                    user_id = removeData.userId ?: "",
                    install_instance_id = installInstanceId
                )
                android.util.Log.d(TAG, "Remove payment request: ${request}")
                val response = apiClient.apiService.removePayment(request)
                android.util.Log.d(TAG, "Remove payment response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "set_user_data" -> {
                val userData = parseUserDataRequest(parameters["userData"] as? Map<String, Any>) ?: throw IllegalArgumentException("userData is required")
                
                val userDataMap = mapOf(
                    "id" to userData.id,
                    "name" to userData.name,
                    "email" to userData.email,
                    "phone" to userData.phone,
                    "mixpanel_distinct_id" to userData.mixpanelDistinctId,
                    "amplitude_device_id" to userData.amplitudeDeviceId,
                    "posthog_distinct_id" to userData.posthogDistinctId,
                    "braze_device_id" to userData.brazeDeviceId,
                    "ga_app_instance_id" to userData.gaAppInstanceId,
                    "user_created_at" to userData.userCreatedAt,
                    "is_first_time_user" to userData.isFirstTimeUser
                ).filterValues { it != null }
                
                val request = SetUserDataRequest(
                    token = token,
                    user_data = userDataMap,
                    platform = configPlatform,
                    install_instance_id = installInstanceId
                )
                android.util.Log.d(TAG, "Set user data request: ${request}")
                val response = apiClient.apiService.setUserData(request)
                android.util.Log.d(TAG, "Set user data response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "update_integrations" -> {
                val integrationDataMap = parameters["integrationData"] as? Map<String, Any> ?: throw IllegalArgumentException("integrationData is required")
                
                val request = IntegrationRequest(
                    token = token,
                    installInstanceId = installInstanceId,
                    integrationInfo = integrationDataMap,
                    platform = configPlatform
                )
                
                android.util.Log.d(TAG, "Update integrations request: ${request}")
                val response = apiClient.apiService.updateIntegrations(request)
                android.util.Log.d(TAG, "Update integrations response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            "update_push_token" -> {
                val pushToken = parameters["pushToken"] as? String ?: throw IllegalArgumentException("pushToken is required")
                
                val request = UpdatePushTokenRequest(
                    token = token,
                    push_token = pushToken,
                    platform = "ANDROID",
                    install_instance_id = installInstanceId
                )
                
                android.util.Log.d(TAG, "Update push token request: ${request}")
                val response = apiClient.apiService.updatePushToken(request)
                android.util.Log.d(TAG, "Update push token response: ${response}")
                if (response.isSuccessful) {
                    true
                } else {
                    throw HttpException(response)
                }
            }
            
            else -> {
                android.util.Log.w(TAG, "Unknown API method: $apiMethod")
                throw IllegalArgumentException("Unknown API method: $apiMethod")
            }
        }
    }
    
    /**
     * Parse UserDataRequest from Map
     */
    private fun parseUserDataRequest(data: Map<String, Any>?): UserDataRequest? {
        if (data == null) return null
        
        return try {
            val id = data["id"] as? String
            if (id == null) {
                android.util.Log.e(TAG, "UserDataRequest id is required but was null")
                return null
            }
            
            UserDataRequest(
                id = id,
                name = data["name"] as? String,
                email = data["email"] as? String,
                phone = data["phone"] as? String,
                mixpanelDistinctId = data["mixpanelDistinctId"] as? String,
                amplitudeDeviceId = data["amplitudeDeviceId"] as? String,
                posthogDistinctId = data["posthogDistinctId"] as? String,
                brazeDeviceId = data["brazeDeviceId"] as? String,
                gaAppInstanceId = data["gaAppInstanceId"] as? String,
                userCreatedAt = data["userCreatedAt"] as? String,
                isFirstTimeUser = data["isFirstTimeUser"] as? Boolean
            )
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to parse UserDataRequest", e)
            null
        }
    }
    
    /**
     * Parse CapturePaymentRequest from Map
     */
    private fun parseCapturePaymentRequest(data: Map<String, Any>?): CapturePaymentRequest? {
        if (data == null) return null
        
        return try {
            CapturePaymentRequest(
                paymentId = data["paymentId"] as? String ?: "",
                userId = data["userId"] as? String ?: "",
                amount = (data["amount"] as? Number)?.toDouble() ?: 0.0,
                type = parsePaymentType(data["type"] as? String),
                status = parsePaymentStatus(data["status"] as? String)
            )
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to parse CapturePaymentRequest", e)
            null
        }
    }
    
    /**
     * Parse RemovePaymentRequest from Map
     */
    private fun parseRemovePaymentRequest(data: Map<String, Any>?): RemovePaymentRequest? {
        if (data == null) return null
        
        return try {
            RemovePaymentRequest(
                paymentId = data["paymentId"] as? String,
                userId = data["userId"] as? String
            )
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to parse RemovePaymentRequest", e)
            null
        }
    }
    
    /**
     * Parse PaymentType from string
     */
    private fun parsePaymentType(typeString: String?): io.linkrunner.sdk.models.PaymentType {
        return when (typeString?.uppercase()) {
            "SUBSCRIPTION_CREATED" -> io.linkrunner.sdk.models.PaymentType.SUBSCRIPTION_CREATED
            "SUBSCRIPTION_RENEWED" -> io.linkrunner.sdk.models.PaymentType.SUBSCRIPTION_RENEWED
            "ONE_TIME" -> io.linkrunner.sdk.models.PaymentType.ONE_TIME
            "RECURRING" -> io.linkrunner.sdk.models.PaymentType.RECURRING
            "FIRST_PAYMENT" -> io.linkrunner.sdk.models.PaymentType.FIRST_PAYMENT
            "WALLET_TOPUP" -> io.linkrunner.sdk.models.PaymentType.WALLET_TOPUP
            "FUNDS_WITHDRAWAL" -> io.linkrunner.sdk.models.PaymentType.FUNDS_WITHDRAWAL
            else -> io.linkrunner.sdk.models.PaymentType.DEFAULT // default
        }
    }
    
    /**
     * Parse PaymentStatus from string
     */
    private fun parsePaymentStatus(statusString: String?): io.linkrunner.sdk.models.PaymentStatus {
        return when (statusString?.uppercase()) {
            "PAYMENT_COMPLETED" -> io.linkrunner.sdk.models.PaymentStatus.PAYMENT_COMPLETED
            "PAYMENT_FAILED" -> io.linkrunner.sdk.models.PaymentStatus.PAYMENT_FAILED
            "PAYMENT_INITIATED" -> io.linkrunner.sdk.models.PaymentStatus.PAYMENT_INITIATED
            "PAYMENT_CANCELLED" -> io.linkrunner.sdk.models.PaymentStatus.PAYMENT_CANCELLED
            else -> io.linkrunner.sdk.models.PaymentStatus.PAYMENT_COMPLETED // default
        }
    }
}