package io.linkrunner.sdk

import android.content.Context
import android.net.Uri
import androidx.annotation.Keep
import io.linkrunner.sdk.models.request.*
import io.linkrunner.sdk.models.response.BaseResponse
import io.linkrunner.sdk.models.request.CapturePaymentRequest
import io.linkrunner.sdk.models.request.RemovePaymentRequest
import io.linkrunner.sdk.models.IntegrationData
import io.linkrunner.sdk.models.request.IntegrationRequest
import io.linkrunner.sdk.models.response.AttributionDataResponse
import io.linkrunner.sdk.network.ApiClient
import io.linkrunner.sdk.network.ApiExecutor
import io.linkrunner.sdk.utils.DeviceInfoLogger
import io.linkrunner.sdk.utils.DeviceInfoProvider
import io.linkrunner.sdk.utils.PreferenceManager
import io.linkrunner.sdk.utils.SHA256
import kotlinx.coroutines.*
import java.util.*
import android.util.Log
import io.linkrunner.sdk.models.response.AttributionData
import io.linkrunner.sdk.work.ApiWorkScheduler

/**
 * Main entry point for the LinkRunner SDK.
 * This class provides methods to initialize and interact with the LinkRunner service.
 */
@Keep
class LinkRunner private constructor() {
    private var TAG = "Linkrunner SDK"
    // Token is now accessed via getter/setter with SharedPreferences persistence
    private var _token: String? = null
    var token: String?
        get() {
            if (_token == null) {
                _token = applicationContext?.let {
                    val preferenceManager = PreferenceManager(it)
                    preferenceManager.getString(KEY_TOKEN)
                }
            }
            return _token
        }
        set(value) {
            _token = value
            applicationContext?.let {
                val preferenceManager = PreferenceManager(it)
                if (value != null) {
                    preferenceManager.saveString(KEY_TOKEN, value)
                }
            }
        }
    
    private var packageVersion = "3.3.1"
    private var platform = "ANDROID"
    private var debug: Boolean = false
    
    // Configuration option for PII hashing
    private var hashPII: Boolean = false
    
    private val baseUrl = "https://api.linkrunner.io"
    private var applicationContext: Context? = null
    
    // SDK-managed coroutine scope with SupervisorJob for proper lifecycle management
    private val sdkScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
    
    // ApiClient singleton management
    private var _apiClient: ApiClient? = null
    private val apiClientLock = Any()
    
    companion object {
        @Volatile
        private var instance: LinkRunner? = null
        private const val KEY_INSTALL_ID = "install_instance_id"
        private const val KEY_DEEPLINK_URL = "deeplink_url"
        private const val KEY_HASH_PII = "hash_pii_enabled"
        private const val KEY_TOKEN = "linkrunner_token"
        private const val KEY_PLATFORM = "linkrunner_platform"
        private const val KEY_PACKAGE_VERSION = "linkrunner_package_version"
        private const val KEY_DEBUG = "linkrunner_debug"
        private const val KEY_SECRET_KEY = "linkrunner_secret_key"
        private const val KEY_KEY_ID = "linkrunner_key_id"
        
        /**
         * Get the singleton instance of LinkRunner.
         */
        @JvmStatic
        fun getInstance(): LinkRunner {
            return instance ?: synchronized(this) {
                instance ?: LinkRunner().also { instance = it }
            }
        }
        
        /**
         * Configure SDK for cross-platform usage (Flutter, React Native, etc.)
         * This method should be called by other SDK wrappers to set platform-specific values
         * @param platform Platform identifier (e.g., "FLUTTER", "REACT_NATIVE")
         * @param packageVersion Package version from the wrapper SDK
         */
        @JvmStatic
        fun configureSDK(platform: String, packageVersion: String) {
            val instance = getInstance()
            instance.platform = platform
            instance.packageVersion = packageVersion
            
            instance.applicationContext?.let { context ->
                val preferenceManager = PreferenceManager(context)
                preferenceManager.saveString(KEY_PLATFORM, platform)
                preferenceManager.saveString(KEY_PACKAGE_VERSION, packageVersion)
                android.util.Log.d("LinkRunner", "SDK configured for platform: $platform, version: $packageVersion")
            }
        }
        
        private fun generateInstallId(): String = UUID.randomUUID().toString()
    }
    
    /**
     * Initialize dependencies for the SDK
     * @param context Application context
     * @param token Your LinkRunner API token
     * @param link Optional deeplink URL
     * @param source Optional source parameter
     * @param secretKey Optional secret key for SDK signature
     * @param keyId Optional key ID for SDK signature
     * @param debug Optional debug flag to enable debug mode
     * @return Result containing the initialization response or an exception
     */
    suspend fun init(context: Context, token: String, link: String? = null, source: String? = null, secretKey: String? = null, keyId: String? = null, debug: Boolean? = false): Result<BaseResponse> {
        this.applicationContext = context.applicationContext
        
        // Set the token - this will also persist it to SharedPreferences
        this.token = token
        
        // Store debug flag and other settings
        this.debug = debug ?: false
        val preferenceManager = PreferenceManager(context.applicationContext)
        preferenceManager.saveBoolean(KEY_DEBUG, this.debug)
        
        // Initialize default platform and packageVersion in preferences if not already set
        if (preferenceManager.getString(KEY_PLATFORM).isEmpty()) {
            preferenceManager.saveString(KEY_PLATFORM, platform)
        }
        if (preferenceManager.getString(KEY_PACKAGE_VERSION).isEmpty()) {
            preferenceManager.saveString(KEY_PACKAGE_VERSION, packageVersion)
        }
        
        // Load saved hashing preference
        this.hashPII = preferenceManager.getBoolean(KEY_HASH_PII)
        
        // Initialize dependencies
        initializeDependencies(context.applicationContext)

        // Store and set SDK signature if secretKey and keyId are provided
        if (!secretKey.isNullOrEmpty() && !keyId.isNullOrEmpty()) {
            // Store signing credentials in preferences for use by other ApiClient instances
            preferenceManager.saveString(KEY_SECRET_KEY, secretKey)
            preferenceManager.saveString(KEY_KEY_ID, keyId)
            
            // Set signature on the main ApiClient instance
            getApiClient().setSDKSignature(keyId, secretKey)
            android.util.Log.d(TAG, "SDK signature configured and stored in preferences")
        }
        
        return initApi(link, source, debug)
    }
    
    /**
     * Initialize SDK dependencies
     */
    private fun initializeDependencies(context: Context) {
        try {
            // Initialize ApiClient singleton with the application context
            synchronized(apiClientLock) {
                if (_apiClient == null) {
                    _apiClient = ApiClient(context)
                }
            }
        } catch (e: Exception) {
            Log.e(TAG, "Error initializing SDK dependencies: ${e.message}")
        }
    }
    
    /**
     * Get the ApiClient singleton instance
     * @return ApiClient instance
     * @throws IllegalStateException if ApiClient is not initialized
     */
    private fun getApiClient(): ApiClient {
        return _apiClient ?: throw IllegalStateException("ApiClient not initialized.")
    }
    
    private fun getOrCreateInstallId(): String {
        val preferenceManager = PreferenceManager(applicationContext ?: throw IllegalStateException("Context not set"))
        return preferenceManager.getString(KEY_INSTALL_ID).takeIf { it.isNotEmpty() } ?: run {
            val newId = generateInstallId()
            preferenceManager.saveString(KEY_INSTALL_ID, newId)
            newId
        }
    }
    
    private fun saveDeeplinkUrl(url: String) {
        val preferenceManager = PreferenceManager(applicationContext ?: throw IllegalStateException("Context not set"))
        preferenceManager.saveString(KEY_DEEPLINK_URL, url)
    }
    
    private fun loadDeeplinkUrl(): String? {
        val preferenceManager = PreferenceManager(applicationContext ?: throw IllegalStateException("Context not set"))
        return preferenceManager.getString(KEY_DEEPLINK_URL).takeIf { it.isNotEmpty() }
    }
    
    private suspend fun initApi(link: String? = null, source: String? = null, debug: Boolean? = false): Result<BaseResponse> {
        return try {
            val deviceInfoProvider = DeviceInfoProvider(applicationContext ?: throw IllegalStateException("Context not set"))
            val deviceInfo = deviceInfoProvider.getDeviceInfo()
            
            val installId = getOrCreateInstallId()
            
            // Get the app version from device info
            val appVersion = deviceInfo["app_version"] as? String ?: ""
            
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueInitialize(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                token = token ?: "",
                packageVersion = packageVersion,
                appVersion = appVersion,
                deviceData = deviceInfo,
                platform = platform,
                installInstanceId = installId,
                link = link,
                source = source,
                debug = debug
            )
            
            if (success) {
                Result.success(createGenericResponse())
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for initialize"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Both enqueue and direct API call failed for initialize", e)
            Result.failure(e)
        }
    }
    

    
    /**
     * Track a user signup event
     * @param userData User data to be associated with the signup
     * @param additionalData Additional custom data to be sent with the event
     * @return Result containing the base response or an exception
     */
    suspend fun signup(
        userData: UserDataRequest,
        additionalData: Map<String, Any>? = null
    ): Result<BaseResponse> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueSignup(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                userData = userData,
                additionalData = additionalData
            )
            
            if (success) {
                Result.success(createGenericResponse())
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for signup"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to signup: ${userData.id}", e)
            Result.failure(e)
        }
    }

    /**
     * Track a payment event
     * @param paymentData Payment data to be tracked
     */
    suspend fun capturePayment(paymentData: CapturePaymentRequest): Result<Unit> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueCapturePayment(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                paymentData = paymentData
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for capturePayment"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to capture payment: ${paymentData.paymentId}", e)
            Result.failure(e)
        }
    }
    
    /**
     * Remove a previously captured payment
     * @param removePayment Payment data to be removed
     */
    suspend fun removePayment(removePayment: RemovePaymentRequest): Result<Unit> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueRemovePayment(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                removeData = removePayment
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for removePayment"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to remove payment: ${removePayment.paymentId}", e)
            Result.failure(e)
        }
    }
    
    /**
     * Track a custom event
     * @param eventName Name of the event to track
     * @param eventData Optional additional event data
     * @param eventId Optional unique event identifier for deduplication
     */
    suspend fun trackEvent(
        eventName: String,
        eventData: Map<String, Any>? = null,
        eventId: String? = null
    ): Result<Unit> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        if (eventName.isEmpty()) {
            return Result.failure(Exception("Event name is required"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueSendEvent(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                eventName = eventName,
                eventData = eventData,
                eventId = eventId
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for trackEvent: $eventName"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to track event: $eventName", e)
            Result.failure(e)
        }
    }
    
    /**
     * Set user data for the current session
     * @param userData User data to be associated with the session
     */
    suspend fun setUserData(userData: UserDataRequest): Result<Unit> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueSetUserData(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                userData = userData
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for setUserData"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to set user data: ${userData.id}", e)
            Result.failure(e)
        }
    }
    
    /**
     * Update the push notification token for the current user
     * @param pushToken The push notification token to be associated with the user
     */
    suspend fun setPushToken(pushToken: String): Result<Unit> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        if (pushToken.isBlank()) {
            return Result.failure(Exception("Push token cannot be empty"))
        }
        
        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueUpdatePushToken(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                pushToken = pushToken
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for setPushToken"))
            }
        } catch (e: Exception) {
            android.util.Log.e(TAG, "Failed to set push token", e)
            Result.failure(e)
        }
    }
    
    
    /**
     * Get attribution data for the current installation
     * @return Result containing the attribution data or an exception
     */
    suspend fun getAttributionData(): Result<AttributionData> {
        if (token == null) {
            return Result.failure(Exception("Linkrunner token not initialized"))
        }
        
        return try {
            val deviceInfoProvider = DeviceInfoProvider(applicationContext ?: throw IllegalStateException("Context not set"))
            val deviceInfo = deviceInfoProvider.getDeviceInfo()
            val installId = getOrCreateInstallId()
            
            val appVersion = deviceInfo["app_version"] as? String ?: ""
            
            val initRequest = InitRequest(
                token = token ?: "",
                package_version = packageVersion,
                app_version = appVersion,
                device_data = deviceInfo,
                platform = platform,
                install_instance_id = installId,
                debug = debug
            )
            
            val response = getApiClient().apiService.getAttributionData(initRequest)
            
            if (response.isSuccessful) {
                val attributionResponse = response.body()
                if (attributionResponse != null) {
                    Result.success(attributionResponse.data)
                } else {
                    Result.failure(Exception("Failed to get attribution data"))
                }
            } else {
                val errorBody = response.errorBody()?.string()
                Result.failure(Exception("Failed to get attribution data: ${response.code()} - $errorBody"))
            }
        } catch (e: Exception) {
            Result.failure(e)
        }
    }


    /**
     * Sends integration data to the server
     * @param integrationData Object containing integration data values
     * @return Result with API response data or error if failed
     */
    suspend fun setAdditionalData(integrationData: IntegrationData): Result<Unit> {
        if (token == null) {
            Log.e("LinkRunner", "Setting integration data failed, token not initialized")
            return Result.failure(Exception("Linkrunner token not initialized"))
        }

        if (integrationData.isEmpty()) {
            Log.e("LinkRunner", "Integration data is required")
            return Result.failure(Exception("Integration data is required"))
        }

        return try {
            // Try to enqueue with fallback handled in ApiWorkScheduler
            val success = ApiWorkScheduler.enqueueSetAdditionalData(
                context = applicationContext ?: throw IllegalStateException("Context not set"),
                integrationData = integrationData
            )
            
            if (success) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Both enqueue and direct API call failed for setAdditionalData"))
            }
        } catch (e: Exception) {
            Log.e("LinkRunner", "Failed to set additional data: ${e.message}")
            Result.failure(e)
        }
    }
    

    /**
     * Enable or disable PII hashing
     * @param enabled Whether PII hashing should be enabled
     */
    fun enablePIIHashing(enabled: Boolean = true) {
        this.hashPII = enabled
        val preferenceManager = PreferenceManager(applicationContext ?: throw IllegalStateException("Context not set"))
        preferenceManager.saveBoolean(KEY_HASH_PII, enabled)
    }
    
    /**
     * Check if PII hashing is currently enabled
     * @return true if PII hashing is enabled, false otherwise
     */
    fun isPIIHashingEnabled(): Boolean {
        applicationContext?.let {
            val preferenceManager = PreferenceManager(it)
            this.hashPII = preferenceManager.getBoolean(KEY_HASH_PII)
        }
        return this.hashPII
    }
    
    /**
     * Hash a string using SHA-256 algorithm
     * @param input The string to hash
     * @return Hashed string in hexadecimal format
     */
    fun hashWithSHA256(input: String): String {
        return SHA256.hash(input)
    }
    
    /**
     * Clean up resources when the SDK is no longer needed
     */
    fun destroy() {
        try {
            // Cancel all ongoing coroutines
            sdkScope.cancel("LinkRunner SDK destroyed")
            
            // Clean up ApiClient singleton
            synchronized(apiClientLock) {
                _apiClient = null
            }
        } catch (e: Exception) {
            Log.e(TAG, "Error during LinkRunner cleanup: ${e.message}")
        } finally {
            instance = null
        }
    }
    
    /**
     * Create a generic success response for WorkManager-handled requests
     */
    private fun createGenericResponse(): BaseResponse {
        return BaseResponse(
            success = true,
            message = "Request enqueued for processing"
        )
    }
    
}
