@file:OptIn(ExperimentalJsExport::class)

import io.kform.*

/**
 * [Validation issue severity][ValidationIssueSeverity] representation for use from JavaScript
 * (`"error" | "warning"`).
 */
public typealias ValidationIssueSeverityJs = String

internal fun ValidationIssueSeverityJs.toValidationIssueSeverityKt(): ValidationIssueSeverity =
    when (this) {
        "error" -> ValidationIssueSeverity.Error
        "warning" -> ValidationIssueSeverity.Warning
        else -> error("Invalid validation issue severity: $this")
    }

internal fun ValidationIssueSeverity.toJs(): ValidationIssueSeverityJs =
    toString().replaceFirstChar { it.lowercase() }

/** [Validation issue][ValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationIssue")
public sealed class ValidationIssueJs {
    internal abstract val issueKt: ValidationIssue

    public val code: String
        get() = issueKt.code

    public val data: RecordTs<String, String?>?
        get() = issueKt.data?.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is ValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: ValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

/** [Validation error][ValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationError")
public class ValidationErrorJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationError =
        when (issueKtOrCode) {
            is ValidationError -> issueKtOrCode
            else -> ValidationError(issueKtOrCode.toString(), jsObjectToMap(data))
        }
}

/** [Validation warning][ValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationWarning")
public class ValidationWarningJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationWarning =
        when (issueKtOrCode) {
            is ValidationWarning -> issueKtOrCode
            else -> ValidationWarning(issueKtOrCode.toString(), jsObjectToMap(data))
        }
}

/** [Validation exception error][ValidationExceptionError] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationExceptionError")
public class ValidationExceptionErrorJs internal constructor(issueKt: Any) : ValidationIssueJs() {
    override val issueKt: ValidationExceptionError =
        when (issueKt) {
            is ValidationExceptionError -> issueKt
            else -> error("Invalid argument.")
        }
}

/** [Located validation issue][LocatedValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationIssue")
public sealed class LocatedValidationIssueJs {
    internal abstract val issueKt: LocatedValidationIssue

    public val path: AbsolutePathJs
        get() = issueKt.path.cachedToJs()

    public val code: String
        get() = issueKt.code

    public val data: RecordTs<String, String?>?
        get() = issueKt.data?.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is LocatedValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: LocatedValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

internal fun LocatedValidationIssue.cachedToJs(): LocatedValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is LocatedValidationExceptionError -> LocatedValidationExceptionErrorJs(this)
            is LocatedValidationError -> LocatedValidationErrorJs(this)
            is LocatedValidationWarning -> LocatedValidationWarningJs(this)
        }
    }

/** [Located validation error][LocatedValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationError")
public class LocatedValidationErrorJs(
    issueKtOrPath: Any,
    codeOrError: Any? = null,
    data: RecordTs<String, String?>? = null
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationError =
        if (issueKtOrPath is LocatedValidationError) issueKtOrPath
        else if (codeOrError is ValidationErrorJs)
            LocatedValidationError(PathJs(issueKtOrPath).pathKt, codeOrError.issueKt)
        else
            LocatedValidationError(
                PathJs(issueKtOrPath).pathKt,
                codeOrError!!.toString(),
                jsObjectToMap(data)
            )
}

/** [Located validation warning][LocatedValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationWarning")
public class LocatedValidationWarningJs(
    issueKtOrPath: Any,
    codeOrWarning: Any? = null,
    data: RecordTs<String, String?>? = null
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationWarning =
        if (issueKtOrPath is LocatedValidationWarning) issueKtOrPath
        else if (codeOrWarning is ValidationWarningJs)
            LocatedValidationWarning(PathJs(issueKtOrPath).pathKt, codeOrWarning.issueKt)
        else
            LocatedValidationWarning(
                PathJs(issueKtOrPath).pathKt,
                codeOrWarning!!.toString(),
                jsObjectToMap(data)
            )
}

/**
 * [Located exception validation error][LocatedValidationExceptionError] wrapper for use from
 * JavaScript.
 */
@JsExport
@JsName("LocatedValidationExceptionError")
public class LocatedValidationExceptionErrorJs
internal constructor(
    issueKt: Any,
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationExceptionError =
        when (issueKt) {
            is LocatedValidationExceptionError -> issueKt
            else -> error("Invalid argument.")
        }
}

/**
 * Function which converts a [ValidationIssue] into a wrapped [ValidationIssueJs] object to be used
 * from JavaScript, while caching the conversion in the process.
 */
internal fun ValidationIssue.cachedToJs(): ValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is ValidationError -> ValidationErrorJs(this)
            is ValidationWarning -> ValidationWarningJs(this)
            is ValidationExceptionError -> ValidationExceptionErrorJs(this)
        }
    }
