@file:OptIn(ExperimentalJsExport::class)

import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext
import kotlin.coroutines.cancellation.CancellationException
import kotlin.js.Promise
import kotlinx.coroutines.*

/** A JavaScript promise-like object that can be cancelled. */
@JsExport
public class CancellablePromise<T>
internal constructor(private val deferred: Any, private val promise: Promise<T>) {
    public fun <TResult> then(
        onFulfilled: ((T) -> TResult)?,
        onRejected: ((Throwable) -> TResult)?
    ): CancellablePromise<TResult> =
        CancellablePromise(
            deferred,
            promise.then(onFulfilled, onRejected?.let { mapOnRejectedCancellation(onRejected) })
        )

    public fun <TResult> catch(onRejected: (Throwable) -> TResult): CancellablePromise<TResult> =
        CancellablePromise(deferred, promise.catch(mapOnRejectedCancellation(onRejected)))

    public fun finally(onFinally: () -> Unit): CancellablePromise<T> =
        CancellablePromise(deferred, promise.finally(onFinally))

    public fun cancel(causeMessage: String? = null): Unit =
        (deferred as Deferred<*>).cancel(PromiseCancellationException(causeMessage))

    private fun <TResult> mapOnRejectedCancellation(
        onRejected: (Throwable) -> TResult
    ): (Throwable) -> TResult = { ex ->
        if (ex is CancellationException) onRejected(PromiseCancellationException(ex.message))
        else onRejected(ex)
    }
}

/** Exception emitted when a [cancellable promise][CancellablePromise] is cancelled. */
@JsExport
public class PromiseCancellationException(override val message: String? = null) :
    CancellationException(message)

internal fun <T> CoroutineScope.cancellablePromise(
    context: CoroutineContext = EmptyCoroutineContext,
    start: CoroutineStart = CoroutineStart.DEFAULT,
    block: suspend CoroutineScope.() -> T
): CancellablePromise<T> = async(context, start, block).asCancellablePromise()

internal fun <T> Deferred<T>.asCancellablePromise(): CancellablePromise<T> =
    CancellablePromise(this, this.asPromise())
