@file:OptIn(ExperimentalJsExport::class, DelicateCoroutinesApi::class)

import io.kform.FormManager
import io.kform.Schema
import io.kform.ValidationMode
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.onSubscription

/**
 * [Validation mode][ValidationMode] representation for use from JavaScript (`"auto" | "manual"`).
 */
public typealias ValidationModeJs = String

internal fun ValidationModeJs.toValidationModeKt(): ValidationMode =
    when (this) {
        "auto" -> ValidationMode.Auto
        "manual" -> ValidationMode.Manual
        else -> error("Invalid validation mode: $this")
    }

/** [Form manager][FormManager] wrapper for use from JavaScript. */
@JsExport
@JsName("FormManager")
public class FormManagerJs(
    schema: Any,
    externalContexts: RecordTs<String, Any>? = null,
    validationMode: ValidationModeJs = "auto",
    autoInit: Boolean = true
) {
    private val formManager =
        FormManager(
            when (schema) {
                is Schema<*> -> schema
                is SchemaJs<*> -> schema.schemaKt
                else -> error("Invalid schema: $schema")
            },
            jsObjectToMap(externalContexts),
            validationMode.toValidationModeKt(),
            autoInit = autoInit
        )

    public fun init(
        externalContexts: RecordTs<String, Any>? = null,
        validationMode: ValidationModeJs = "auto"
    ): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.init(jsObjectToMap(externalContexts), validationMode.toValidationModeKt())
            undefined
        }

    public fun destroy(): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.destroy()
            undefined
        }

    public val autoValidationStatus: AutoValidationStatusJs
        get() = formManager.autoValidationStatus.value.toJs()

    public fun onAutoValidationStatusChange(
        statusChangeHandler: (status: AutoValidationStatusJs) -> Any?,
        onSubscription: (() -> Any?)? = null
    ): CancellablePromise<() -> CancellablePromise<Nothing?>> =
        GlobalScope.cancellablePromise {
            val subscribed = CompletableDeferred<Unit>()
            val job =
                GlobalScope.launch {
                    formManager.autoValidationStatus
                        .onSubscription {
                            onSubscription?.invoke().maybeAwait()
                            subscribed.complete(Unit)
                        }
                        .collect { statusChangeHandler(it.toJs()).maybeAwait() }
                }
            subscribed.join()
            return@cancellablePromise {
                GlobalScope.cancellablePromise {
                    job.cancelAndJoin()
                    undefined
                }
            }
        }

    public fun setValidationMode(validationMode: ValidationModeJs): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setValidationMode(validationMode.toValidationModeKt())
            undefined
        }

    public fun isValidPath(path: PathJs): Boolean = formManager.isValidPath(path.pathKt)

    public fun schemaInfo(path: PathJs = AbsolutePathJs.MATCH_ALL): IterableJs<SchemaInfoJs<*>> =
        formManager.schemaInfo(path.pathKt).toIterableJs { it.cachedToJs() }

    public fun valueInfo(
        path: PathJs = AbsolutePathJs.MATCH_ALL,
        infoHandler: (infoIterable: AsyncIterableJs<ValueInfoJs<*>>) -> Any?
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.valueInfo(path.pathKt) { infoFlow ->
                infoHandler(infoFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
            }
        }

    public fun info(
        path: PathJs = AbsolutePathJs.MATCH_ALL,
        infoHandler: (infoIterable: AsyncIterableJs<InfoJs<*>>) -> Any?
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.info(path.pathKt) { infoFlow ->
                infoHandler(infoFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
            }
        }

    public fun schema(path: PathJs = AbsolutePathJs.ROOT): SchemaJs<*> =
        formManager.schema(path.pathKt).cachedToJs()

    public fun has(path: PathJs): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.has(path.pathKt) }

    public fun <T> get(
        path: PathJs = AbsolutePathJs.ROOT,
        valueHandler: (value: T) -> Any?
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.get(path.pathKt) { value: T -> valueHandler(value).maybeAwait() }
        }

    public fun <T> getClone(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise { formManager.getClone<T>(path.pathKt) }

    public fun set(path: PathJs = AbsolutePathJs.ROOT, toSet: Any?): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.set(path.pathKt, toSet)
            undefined
        }

    public fun reset(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.reset(path.pathKt)
            undefined
        }

    public fun remove(path: PathJs): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.remove(path.pathKt)
            undefined
        }

    public fun getExternalContext(
        externalContextName: String,
        externalContextHandler: (externalContext: Any?) -> Any?
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.getExternalContext<Any?, Any?>(externalContextName) {
                externalContextHandler(it).maybeAwait()
            }
        }

    public fun <T> setExternalContext(
        externalContextName: String,
        externalContext: T
    ): CancellablePromise<T?> =
        GlobalScope.cancellablePromise {
            formManager.setExternalContext(externalContextName, externalContext)
        }

    public fun <T> removeExternalContext(externalContextName: String): CancellablePromise<T?> =
        GlobalScope.cancellablePromise { formManager.removeExternalContext(externalContextName) }

    public fun validate(
        path: PathJs = AbsolutePathJs.MATCH_ALL,
        issuesHandler: ((issuesIterable: AsyncIterableJs<LocatedValidationIssueJs>) -> Any?)? = null
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            if (issuesHandler != null)
                formManager.validate(path.pathKt) { issuesFlow ->
                    issuesHandler(issuesFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
                }
            else formManager.validate(path.pathKt).map { it.cachedToJs() }.toTypedArray()
        }

    public fun isValid(path: PathJs = AbsolutePathJs.MATCH_ALL): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isValid(path.pathKt) }

    public fun isDirty(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isDirty(path.pathKt) }

    public fun isPristine(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isPristine(path.pathKt) }

    public fun setDirty(path: PathJs = AbsolutePathJs.MATCH_ALL): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setDirty(path.pathKt)
            undefined
        }

    public fun setPristine(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setPristine(path.pathKt)
            undefined
        }

    public fun isTouched(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isTouched(path.pathKt) }

    public fun isUntouched(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isUntouched(path.pathKt) }

    public fun setTouched(path: PathJs = AbsolutePathJs.MATCH_ALL): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setTouched(path.pathKt)
            undefined
        }

    public fun setUntouched(path: PathJs = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setUntouched(path.pathKt)
            undefined
        }

    public fun subscribe(
        path: PathJs = AbsolutePathJs.MATCH_ALL,
        eventHandler: (event: FormManagerEventJs<*>) -> Any?,
        onSubscription: (() -> Any?)? = null
    ): CancellablePromise<() -> CancellablePromise<Nothing?>> =
        GlobalScope.cancellablePromise {
            val unsubscribe =
                formManager.subscribe(
                    path.pathKt,
                    onSubscription?.let { { onSubscription().maybeAwait() } }
                ) {
                    eventHandler(it.cachedToJs()).maybeAwait()
                }
            return@cancellablePromise {
                GlobalScope.cancellablePromise {
                    unsubscribe()
                    undefined
                }
            }
        }
}
