import io.kform.*

/**
 * [Validation issue severity][ValidationIssueSeverity] representation for use from JavaScript
 * (`"error" | "warning"`).
 */
public typealias ValidationIssueSeverityJs = String

internal fun ValidationIssueSeverityJs.toValidationIssueSeverityKt(): ValidationIssueSeverity =
    when (this) {
        "error" -> ValidationIssueSeverity.Error
        "warning" -> ValidationIssueSeverity.Warning
        else -> error("Invalid validation issue severity: '$this'.")
    }

internal fun ValidationIssueSeverity.toJs(): ValidationIssueSeverityJs =
    toString().replaceFirstChar { it.lowercase() }

/** [Validation issue][ValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationIssue")
public sealed class ValidationIssueJs {
    internal abstract val issueKt: ValidationIssue

    public val code: String
        get() = issueKt.code

    public val data: RecordTs<String, String?>
        get() = issueKt.data.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is ValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: ValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

/** [Validation error][ValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationError")
public class ValidationErrorJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationError =
        when (issueKtOrCode) {
            is ValidationError -> issueKtOrCode
            else -> ValidationError(issueKtOrCode.toString(), jsObjectToMap(data) ?: emptyMap())
        }
}

/** [Validation warning][ValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationWarning")
public class ValidationWarningJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationWarning =
        when (issueKtOrCode) {
            is ValidationWarning -> issueKtOrCode
            else -> ValidationWarning(issueKtOrCode.toString(), jsObjectToMap(data) ?: emptyMap())
        }
}

/** [Validation exception error][ValidationExceptionError] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationExceptionError")
public class ValidationExceptionErrorJs internal constructor(issueKt: Any) : ValidationIssueJs() {
    override val issueKt: ValidationExceptionError =
        when (issueKt) {
            is ValidationExceptionError -> issueKt
            else -> error("Invalid argument.")
        }
}

/** [Located validation issue][LocatedValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationIssue")
public sealed class LocatedValidationIssueJs {
    internal abstract val issueKt: LocatedValidationIssue

    public val path: AbsolutePathJs
        get() = issueKt.path.cachedToJs()

    public val code: String
        get() = issueKt.code

    public val dependencies: Array<AbsolutePathJs>
        get() = issueKt.dependencies.cachedToJs { it.cachedToJs() }

    public val dependsOnDescendants: Boolean
        get() = issueKt.dependsOnDescendants

    public val externalContextDependencies: Array<String>
        get() = issueKt.externalContextDependencies.cachedToJs()

    public val data: RecordTs<String, String?>
        get() = issueKt.data.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is LocatedValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: LocatedValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

internal fun LocatedValidationIssue.cachedToJs(): LocatedValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is LocatedValidationExceptionError -> LocatedValidationExceptionErrorJs(this)
            is LocatedValidationError -> LocatedValidationErrorJs(this)
            is LocatedValidationWarning -> LocatedValidationWarningJs(this)
        }
    }

/** Options available when building a [LocatedValidationErrorJs] or [LocatedValidationWarningJs]. */
public external interface LocatedValidationIssueOptionsJs {
    public val dependencies: Array<Any>?
    public val dependsOnDescendants: Boolean?
    public val externalContextDependencies: Array<String>?
    public val data: RecordTs<String, String?>?
}

/** [Located validation error][LocatedValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationError")
public class LocatedValidationErrorJs(
    issueKtOrPath: Any,
    code: String? = null,
    options: LocatedValidationIssueOptionsJs? = null,
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationError =
        if (issueKtOrPath is LocatedValidationError) issueKtOrPath
        else
            LocatedValidationError(
                issueKtOrPath.toPathKt(),
                code!!.toString(),
                options?.dependencies?.map { it.toPathKt() } ?: emptySet(),
                options?.dependsOnDescendants ?: false,
                options?.externalContextDependencies?.toSet() ?: emptySet(),
                jsObjectToMap(options?.data) ?: emptyMap(),
            )
}

/** [Located validation warning][LocatedValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationWarning")
public class LocatedValidationWarningJs(
    issueKtOrPath: Any,
    code: String? = null,
    options: LocatedValidationIssueOptionsJs? = null,
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationWarning =
        if (issueKtOrPath is LocatedValidationWarning) issueKtOrPath
        else
            LocatedValidationWarning(
                issueKtOrPath.toPathKt(),
                code!!.toString(),
                options?.dependencies?.map { it.toPathKt() } ?: emptySet(),
                options?.dependsOnDescendants ?: false,
                options?.externalContextDependencies?.toSet() ?: emptySet(),
                jsObjectToMap(options?.data) ?: emptyMap(),
            )
}

/**
 * [Located exception validation error][LocatedValidationExceptionError] wrapper for use from
 * JavaScript.
 */
@JsExport
@JsName("LocatedValidationExceptionError")
public class LocatedValidationExceptionErrorJs internal constructor(issueKt: Any) :
    LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationExceptionError =
        when (issueKt) {
            is LocatedValidationExceptionError -> issueKt
            else -> error("Invalid argument.")
        }
}

/**
 * Function which converts a [ValidationIssue] into a wrapped [ValidationIssueJs] object to be used
 * from JavaScript, while caching the conversion in the process.
 */
internal fun ValidationIssue.cachedToJs(): ValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is ValidationError -> ValidationErrorJs(this)
            is ValidationWarning -> ValidationWarningJs(this)
            is ValidationExceptionError -> ValidationExceptionErrorJs(this)
        }
    }

/**
 * Function that converts a [LocatedValidationIssue] or a wrapped [LocatedValidationIssueJs] into a
 * [LocatedValidationIssue].
 */
internal fun Any.toLocatedValidationIssueKt(): LocatedValidationIssue =
    when (this) {
        is LocatedValidationIssue -> this
        is LocatedValidationIssueJs -> issueKt
        else -> error("Invalid located validation issue.")
    }
