import kotlin.js.Promise
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.launch

@JsName("IteratorResult")
public external interface IteratorResultJs<T> {
    public val done: Boolean?
    public val value: T?
}

@JsName("Iterator")
public external interface IteratorJs<T> {
    public fun next(): IteratorResultJs<T>

    public fun `return`(): IteratorResultJs<T>

    public fun `throw`(ex: Throwable?): IteratorResultJs<T>
}

@JsName("Iterable") public external interface IterableJs<@Suppress("unused") T>

@JsName("AsyncIterator")
public external interface AsyncIteratorJs<T> {
    public fun next(): Promise<IteratorResultJs<T>>

    public fun `return`(): Promise<IteratorResultJs<T>>

    public fun `throw`(ex: Throwable?): Promise<IteratorResultJs<T>>
}

@JsName("AsyncIterable") public external interface AsyncIterableJs<@Suppress("unused") T>

// JavaScript `Symbol`
private external val Symbol: dynamic

/** Whether the provided value is (likely) a JavaScript iterator. */
internal fun isJsIterator(value: dynamic): Boolean = jsTypeOf(value.next) == "function"

/** Iterates a (possibly async) iterator. */
@Suppress(
    "UNCHECKED_CAST_TO_EXTERNAL_INTERFACE",
    "UNCHECKED_CAST",
    "UnsafeCastFromDynamic",
    "USELESS_CAST"
)
internal suspend inline fun <T> iterateJsAsyncIterator(iterator: dynamic, cb: (value: T) -> Unit) {
    var result = (iterator.next() as Any?).maybeAwait() as IteratorResultJs<T>
    while (!result.done.asDynamic()) {
        cb(result.value as T)
        result = (iterator.next() as Any?).maybeAwait() as IteratorResultJs<T>
    }
}

/** Creates an iterator result. */
@Suppress("UNCHECKED_CAST_TO_EXTERNAL_INTERFACE", "UNUSED_PARAMETER")
internal fun <T> iteratorResult(done: Boolean = false, value: T? = undefined): IteratorResultJs<T> {
    return js("{ value: value, done: done }") as IteratorResultJs<T>
}

/**
 * Function returning a JavaScript iterator from a [Sequence], mapping each value via [valueMapper].
 */
@Suppress("UNCHECKED_CAST_TO_EXTERNAL_INTERFACE")
internal fun <T, TResult> Sequence<T>.toIterableJs(
    valueMapper: (value: T) -> TResult
): IterableJs<TResult> {
    val obj = js("{}")
    obj[Symbol.iterator] = { SequenceIteratorJs(this, valueMapper) }
    return obj as IterableJs<TResult>
}

/** Class implementing a JavaScript iterator from a given [Sequence]. */
@JsName("SequenceIterator")
public class SequenceIteratorJs<T, TResult>
internal constructor(sequence: Sequence<T>, private val valueMapper: (value: T) -> TResult) :
    IteratorJs<TResult> {
    private var iterator: Iterator<T>? = sequence.iterator()

    override fun next(): IteratorResultJs<TResult> =
        if (iterator != null && iterator!!.hasNext())
            iteratorResult(value = valueMapper(iterator!!.next()))
        else `return`()

    override fun `return`(): IteratorResultJs<TResult> {
        iterator = null
        return iteratorResult(done = true)
    }

    override fun `throw`(ex: Throwable?): IteratorResultJs<TResult> {
        iterator = null
        return iteratorResult(done = true)
    }
}

/** Converts a flow into a JavaScript async iterator. */
@Suppress("UNCHECKED_CAST_TO_EXTERNAL_INTERFACE")
internal fun <T, TResult> Flow<T>.toAsyncIterableJs(
    valueMapper: (value: T) -> Any?
): AsyncIterableJs<TResult> {
    val obj = js("{}")
    obj[Symbol.asyncIterator] = { FlowIteratorJs<T, TResult>(this, valueMapper) }
    return obj as AsyncIterableJs<TResult>
}

@JsName("FlowIterator")
internal class FlowIteratorJs<T, TResult>(
    private val flow: Flow<T>,
    private val valueMapper: (value: T) -> Any?
) : AsyncIteratorJs<TResult> {
    private var deferred: CompletableDeferred<Unit> = CompletableDeferred()
    private var resolve: ((value: IteratorResultJs<TResult>) -> Unit)? = null
    private var reject: ((ex: Throwable) -> Unit)? = null

    private val job =
        @OptIn(DelicateCoroutinesApi::class)
        GlobalScope.launch {
            try {
                deferred.await()
                flow.collect {
                    @Suppress("UNCHECKED_CAST")
                    resolve?.invoke(iteratorResult(value = valueMapper(it).maybeAwait() as TResult))
                    deferred = CompletableDeferred()
                    resolve = null
                    reject = null
                    deferred.await()
                }
                resolve?.invoke(iteratorResult(done = true))
            } catch (err: Throwable) {
                reject?.invoke(err)
            }
            resolve = null
            reject = null
        }

    override fun next(): Promise<IteratorResultJs<TResult>> =
        if (job.isActive)
            Promise { res, rej ->
                resolve = res
                reject = rej
                deferred.complete(Unit)
            }
        else Promise.resolve(iteratorResult(done = true))

    override fun `return`(): Promise<IteratorResultJs<TResult>> {
        job.cancel()
        return Promise.resolve(iteratorResult(done = true))
    }

    override fun `throw`(ex: Throwable?): Promise<IteratorResultJs<TResult>> {
        job.cancel()
        return Promise.reject(ex!!)
    }
}
