import io.kform.*
import io.kform.datatypes.Table
import io.kform.schemas.AbstractCollectionSchema
import io.kform.schemas.ListState
import io.kform.schemas.util.commonRestrictions
import io.kform.schemas.util.sizeBoundsRestrictions
import kotlin.math.min
import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.isActive

internal class ArraySchemaJs<T, TSchema : Schema<T>>(
    elementsSchema: TSchema,
    override val validations: List<Validation<Array<T>>> = emptyList(),
    override val initialValue: Array<T> = emptyArray()
) : AbstractCollectionSchema<Array<T>, T, TSchema>(elementsSchema) {
    override val typeInfo: TypeInfo =
        TypeInfo(
            Array::class,
            arguments = listOf(elementsSchema.typeInfo),
            restrictions = commonRestrictions(validations) + sizeBoundsRestrictions(validations)
        )

    override suspend fun clone(value: Array<T>): Array<T> =
        value.map { el -> elementsSchema.clone(el) }.toTypedArray()

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(initialValue) == true &&
            (if (type.classifier == Array::class)
                type.arguments[0].type == null ||
                    elementsSchema.assignableTo(type.arguments[0].type!!)
            else true)

    /** Whether the provided id fragment can be converted to an integer. */
    private fun isValidIndexId(fragment: AbsolutePathFragment.Id): Boolean =
        fragment.id.matches(INDEX_REGEX)

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id && isValidIndexId(fragment))

    override suspend fun isValidChildFragment(
        value: Array<T>,
        fragment: AbsolutePathFragment
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidIndexId(fragment) &&
            fragment.id.toInt() < value.size)

    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: Array<T>,
        fragment: AbsolutePathFragment
    ): Flow<ValueInfo<T>> = flow {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((i, elem) in value.withIndex()) {
                emit(
                    ValueInfo(
                        elem,
                        elementsSchema,
                        path.append(AbsolutePathFragment.Id(i)),
                        schemaPath.append(AbsolutePathFragment.Wildcard)
                    )
                )
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val idx = fragment.id.toInt()
            if (idx < value.size) { // `idx == value.size` is considered valid
                emit(
                    ValueInfo(
                        value[idx],
                        elementsSchema,
                        path.append(fragment),
                        schemaPath.append(AbsolutePathFragment.Wildcard)
                    )
                )
            }
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun fromAny(value: Any?): Array<T> =
        when (value) {
            is Array<*> -> value as Array<T>
            is Collection<*> -> (value as Collection<T>).toTypedArray()
            is Table<*> -> (value as Table<T>).values.toTypedArray()
            else -> throw IllegalArgumentException("Cannot convert value '$value' to Array.")
        }

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus
    ): Array<T> {
        val fromArray = fromAny(fromValue)
        val newValue = emptyArray<T>()
        for ((i, elem) in fromArray.withIndex()) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            newValue[i] =
                elementsSchema.init(path.append(AbsolutePathFragment.Id(i)), elem, eventsBus)
        }
        eventsBus.emit(ValueEvent.Init(newValue, path, this))
        return newValue
    }

    override suspend fun change(
        path: AbsolutePath,
        value: Array<T>,
        intoValue: Any?,
        eventsBus: SchemaEventsBus
    ): Array<T> {
        val intoArray = fromAny(intoValue)
        val curSize = value.size
        val newSize = intoArray.size
        for (i in 0 until min(curSize, newSize)) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            value[i] =
                elementsSchema.change(
                    path.append(AbsolutePathFragment.Id(i)),
                    value[i],
                    intoArray[i],
                    eventsBus
                )
        }
        if (curSize > newSize) {
            for (i in curSize - 1 downTo newSize) {
                if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
                val id = AbsolutePathFragment.Id(i)
                val oldChild = value.asDynamic().pop() as T
                eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
                elementsSchema.destroy(path.append(id), oldChild, eventsBus)
            }
        } else if (curSize < newSize) {
            for (i in curSize until newSize) {
                if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
                val id = AbsolutePathFragment.Id(i)
                val newChild = elementsSchema.init(path.append(id), intoArray[i], eventsBus)
                value.asDynamic().push(newChild)
                eventsBus.emit(ValueEvent.Add(value, newChild, id, path, this))
            }
        }
        return value
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: Array<T>,
        eventsBus: SchemaEventsBus
    ): Array<T> {
        eventsBus.emit(ValueEvent.Destroy(value, path, this))
        for (i in value.lastIndex downTo 0) {
            elementsSchema.destroy(path.append(AbsolutePathFragment.Id(i)), value[i], eventsBus)
        }
        return value
    }

    override suspend fun isValidSetFragment(
        value: Array<T>,
        fragment: AbsolutePathFragment
    ): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id &&
                isValidIndexId(fragment) &&
                fragment.id.toInt() < value.size)

    override suspend fun set(
        path: AbsolutePath,
        value: Array<T>,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus
    ) {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((idx, elem) in value.withIndex()) {
                value[idx] =
                    elementsSchema.change(
                        path.append(AbsolutePathFragment.Id(idx)),
                        elem,
                        childValue,
                        eventsBus
                    )
            }
        } else {
            val size = value.size
            val id =
                if (fragment is AbsolutePathFragment.CollectionEnd) AbsolutePathFragment.Id(size)
                else fragment
            val idx = (id as AbsolutePathFragment.Id).id.toInt()
            if (idx == size) {
                val newChild = elementsSchema.init(path.append(id), childValue, eventsBus)
                value.asDynamic().push(newChild)
                eventsBus.emit(ValueEvent.Add(value, newChild, id, path, this))
            } else {
                value[idx] =
                    elementsSchema.change(path.append(id), value[idx], childValue, eventsBus)
            }
        }
    }

    override suspend fun isValidRemoveFragment(
        value: Array<T>,
        fragment: AbsolutePathFragment
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidIndexId(fragment) &&
            fragment.id.toInt() < value.size)

    override suspend fun remove(
        path: AbsolutePath,
        value: Array<T>,
        fragment: AbsolutePathFragment,
        eventsBus: SchemaEventsBus
    ) {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for (i in value.lastIndex downTo 0) {
                val id = AbsolutePathFragment.Id(i)
                val oldChild = value.asDynamic().pop() as T
                eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
                elementsSchema.destroy(path.append(id), oldChild, eventsBus)
            }
        } else {
            val idx = (fragment as AbsolutePathFragment.Id).id.toInt()
            for (i in idx until value.lastIndex) {
                value[i] =
                    elementsSchema.change(
                        path.append(AbsolutePathFragment.Id(i)),
                        value[i],
                        value[i + 1],
                        eventsBus
                    )
            }
            val id = AbsolutePathFragment.Id(value.lastIndex)
            val oldChild = value.asDynamic().pop() as T
            eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
            elementsSchema.destroy(path.append(id), oldChild, eventsBus)
        }
    }

    override fun childrenStatesContainer(): CollectionState = ListState(elementsSchema)

    companion object {
        /** Regex used to determine if a certain identifier can be converted to an index. */
        private val INDEX_REGEX = Regex("^(0|[1-9]\\d*)\$")
    }
}

/** Schema representing a JavaScript array. */
@JsName("arraySchema")
@JsExport
public fun <T> arraySchemaJs(
    optionsOrElementsSchema: Any,
    elementsSchema: SchemaJs<T>? = null
): SchemaJs<Array<T>> =
    @Suppress("UNCHECKED_CAST", "UNCHECKED_CAST_TO_EXTERNAL_INTERFACE")
    when (elementsSchema) {
        null -> ArraySchemaJs((optionsOrElementsSchema as SchemaJs<T>).schemaKt)
        else -> {
            val options = optionsOrElementsSchema as SchemaOptionsJs<Array<T>>
            ArraySchemaJs(
                elementsSchema.schemaKt,
                options.validations?.map { it.validationKt } ?: emptyList(),
                options.initialValue ?: emptyArray()
            )
        }
    }.cachedToJs()
