import io.kform.Validation
import io.kform.ValidationContext
import io.kform.ValidationIssue
import io.kform.ValidationIssueSeverity
import io.kform.validations.Matches
import io.kform.validations.MatchesEmail
import kotlin.js.RegExp
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow

// Reimplementation using JavaScript's `RegExp` instead of `Kotlin's` Regex
private open class MatchesJs(
    val regex: RegExp,
    val code: String = Matches.DEFAULT_CODE,
    val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error
) : Validation<String>() {
    override fun toString(): String = "Matches($regex)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        // Validation only runs when the string is not empty
        if (value.isNotEmpty() && !regex.test(value)) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("actualValue" to value, "mustMatch" to "$regex")
                )
            )
        }
    }
}

/** Matches validation for use from JavaScript. */
@JsExport
@JsName("matches")
public fun matchesJs(
    regex: RegExp,
    code: String = Matches.DEFAULT_CODE,
    severity: ValidationIssueSeverityJs = "error",
): ValidationJs<String> =
    MatchesJs(regex, code, severity.toValidationIssueSeverityKt()).cachedToJs()

/** Matches email validation for use from JavaScript. */
@JsExport
@JsName("matchesEmail")
public fun matchesEmailJs(
    code: String = MatchesEmail.DEFAULT_CODE,
    severity: ValidationIssueSeverityJs = "error",
    emailRegex: RegExp? = null,
): ValidationJs<String> =
    MatchesJs(
            emailRegex ?: RegExp(MatchesEmail.EMAIL_REGEX.pattern),
            code,
            severity.toValidationIssueSeverityKt()
        )
        .cachedToJs()
