/** JavaScript's `Object` object. */
private external val Object: dynamic

@Suppress("UNCHECKED_CAST_TO_EXTERNAL_INTERFACE", "NOTHING_TO_INLINE")
internal inline fun <T> emptyJsObject(): RecordTs<String, T> = js("{}") as RecordTs<String, T>

/**
 * Transforms a JavaScript object into a [Map] with [String] keys, mapping the values in the
 * process.
 */
internal fun <T, TMapped> jsObjectToMap(
    obj: RecordTs<String, T>?,
    valueMapper: (value: T) -> TMapped
): Map<String, TMapped>? {
    obj ?: return null
    val keys = Object.keys(obj)
    return keys.reduce(
        { map: MutableMap<String, TMapped>, key: String ->
            map[key] = valueMapper(obj.asDynamic()[key] as T)
            map
        },
        LinkedHashMap<String, T>(keys.length as Int)
    ) as Map<String, TMapped>
}

/** Transforms a JavaScript object into a [Map] with [String] keys. */
@Suppress("NOTHING_TO_INLINE")
internal inline fun <T> jsObjectToMap(obj: RecordTs<String, T>?): Map<String, T>? =
    jsObjectToMap(obj) { it }

/**
 * Transforms a [Map] with [String] keys into a JavaScript object, mapping the values in the
 * process.
 */
internal fun <T, TMapped> mapToJsObject(
    map: Map<String, T>?,
    valueMapper: (value: T) -> TMapped
): RecordTs<String, TMapped>? {
    map ?: return null
    return map.entries.fold(emptyJsObject()) { obj, (key, value) ->
        obj.asDynamic()[key] = valueMapper(value)
        obj
    }
}

/** Transforms a [Map] with [String] keys into a JavaScript object. */
@Suppress("NOTHING_TO_INLINE")
internal inline fun <T> mapToJsObject(map: Map<String, T>?): RecordTs<String, T>? =
    mapToJsObject(map) { it }

internal fun <T> Map<String, T>.cachedToJs(): RecordTs<String, T> =
    getOrSetFromCache(this) { mapToJsObject(this)!! }

internal fun <T, TMapped> Map<String, T>.cachedToJs(
    valueMapper: (value: T) -> TMapped
): RecordTs<String, TMapped> = getOrSetFromCache(this) { mapToJsObject(this, valueMapper)!! }
