@file:OptIn(ExperimentalJsExport::class)

import io.kform.Validation
import io.kform.ValidationContext
import io.kform.ValidationIssue
import io.kform.paths.Path
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow

/** [Validation context][ValidationContext] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationContext")
public class ValidationContextJs<T>
internal constructor(private val validationContextKt: ValidationContext) {
    public val schema: SchemaJs<T>
        get() = validationContextKt.schema<T>().cachedToJs()

    public val value: T
        get() = validationContextKt.value()

    public fun <TDependency> dependencyInfoOrNull(
        dependencyKey: String
    ): ValueInfoJs<TDependency>? =
        validationContextKt.dependencyInfoOrNull<TDependency>(dependencyKey)?.cachedToJs()

    public fun <TDependency> dependencyInfo(dependencyKey: String): ValueInfoJs<TDependency> =
        validationContextKt.dependencyInfo<TDependency>(dependencyKey).cachedToJs()

    public fun dependencyPathOrNull(dependencyKey: String): AbsolutePathJs? =
        validationContextKt.dependencyPathOrNull(dependencyKey)?.cachedToJs()

    public fun dependencyPath(dependencyKey: String): AbsolutePathJs =
        validationContextKt.dependencyPath(dependencyKey).cachedToJs()

    public fun dependencySchemaOrNull(dependencyKey: String): SchemaJs<*>? =
        validationContextKt.dependencySchemaOrNull(dependencyKey)?.cachedToJs()

    public fun dependencySchema(dependencyKey: String): SchemaJs<*> =
        validationContextKt.dependencySchema(dependencyKey).cachedToJs()

    public fun <TDependency> dependencyOrNull(dependencyKey: String): TDependency? =
        validationContextKt.dependencyOrNull(dependencyKey)

    public fun <TDependency> dependency(dependencyKey: String): TDependency =
        validationContextKt.dependency(dependencyKey)

    public fun <TContext> externalContextOrNull(externalContextName: String): TContext? =
        validationContextKt.externalContextOrNull(externalContextName)

    public fun <TContext> externalContext(externalContextName: String): TContext =
        validationContextKt.externalContext(externalContextName)
}

/** Options passed when creating a new validation from JavaScript. */
public external interface ValidationOptionsJs {
    public val name: String?
    public val dependencies: RecordTs<String, Any>?
    public val dependsOnDescendants: Boolean?
    public val externalContextDependencies: Array<String>?
}

/** Validation function created from JavaScript. */
public typealias ValidationFunction<T> = (context: ValidationContextJs<T>) -> Any?

/**
 * [Validation] wrapper for use from JavaScript.
 *
 * A new custom validation can be created like so:
 * ```javascript
 * new Validation(
 *   { dependencies: { allowOdd: "../allowOdd" } },
 *   function* DisallowOdd(cx) {
 *     if (!cx.dependency("allowOdd") && cx.value % 2 !== 0) {
 *       yield new ValidationError("oddNotAllowed");
 *     }
 *   }
 * );
 * ```
 */
@JsExport
@JsName("Validation")
public open class ValidationJs<T>(
    validationKtOrOptionsOrValidate: Any,
    validate: ValidationFunction<T>? = null
) {
    @Suppress("UNCHECKED_CAST_TO_EXTERNAL_INTERFACE", "UNCHECKED_CAST")
    internal val validationKt: Validation<T> =
        when (validationKtOrOptionsOrValidate) {
            is Validation<*> -> validationKtOrOptionsOrValidate as Validation<T>
            else -> {
                val argType = jsTypeOf(validationKtOrOptionsOrValidate)
                val options =
                    (if (argType == "function") emptyJsObject<Any?>()
                    else validationKtOrOptionsOrValidate)
                        as ValidationOptionsJs
                val validateFn =
                    if (argType == "function")
                        validationKtOrOptionsOrValidate as ValidationFunction<T>
                    else validate!!
                JsValidationWrapper(options, validateFn)
            }
        }

    public val dependencies: RecordTs<String, PathJs>
        get() = validationKt.dependencies.cachedToJs { it.cachedToJs() }

    public val dependsOnDescendants: Boolean
        get() = validationKt.dependsOnDescendants

    public val externalContextDependencies: Array<String>
        get() = validationKt.externalContextDependencies.cachedToJs()

    public override fun toString(): String = validationKt.toString()
}

internal class JsValidationWrapper<T>(
    options: ValidationOptionsJs,
    private val validateFn: ValidationFunction<T>
) : Validation<T>() {
    private val name: String? = options.name

    override val dependencies: Map<String, Path> =
        jsObjectToMap(options.dependencies) { it.toPathKt() } ?: emptyMap()

    override val dependsOnDescendants: Boolean = options.dependsOnDescendants ?: false

    override val externalContextDependencies: Set<String> =
        options.externalContextDependencies?.toSet() ?: emptySet()

    @Suppress("USELESS_CAST")
    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val result = validateFn(ValidationContextJs(this@validate)).asDynamic()
        if (isJsIterator(result)) {
            // `Iterator<ValidationIssueJs> | AsyncIterator<ValidationIssueJs>`
            iterateJsAsyncIterator<ValidationIssueJs>(result) { emit(it.issueKt) }
        } else {
            // `ValidationIssueJs[] | Promise<ValidationIssueJs[]>`
            val issues = (result as Any?).maybeAwait() as Array<*>
            for (issue in issues) {
                emit((issue as ValidationIssueJs).issueKt)
            }
        }
    }

    override fun toString(): String =
        name ?: (validateFn.asDynamic().name as String).ifEmpty { super.toString() }
}

/**
 * Function which converts a [Validation] into a wrapped [ValidationJs] object to be used from
 * JavaScript, while caching the conversion in the process.
 */
internal fun <T> Validation<T>.cachedToJs(): ValidationJs<T> =
    getOrSetFromCache(this) { ValidationJs(this) }
