/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.kareldb;

import io.kcache.KafkaCacheConfig;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigDef.Importance;
import org.apache.kafka.common.config.ConfigDef.Type;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class KarelDbConfig extends KafkaCacheConfig {

    public static final String LISTENERS_CONFIG = "listeners";
    public static final String LISTENERS_DEFAULT = "http://0.0.0.0:8765";
    public static final String LISTENERS_DOC =
        "List of listeners. http and https are supported. Each listener must include the protocol, "
            + "hostname, and port. For example: http://myhost:8080, https://0.0.0.0:8081";

    public static final String CLUSTER_GROUP_ID_CONFIG = "cluster.group.id";
    public static final String CLUSTER_GROUP_ID_DEFAULT = "kareldb";
    public static final String CLUSTER_GROUP_ID_DOC =
        "The group ID used for leader election.";

    public static final String LEADER_ELIGIBILITY_CONFIG = "leader.eligibility";
    public static final boolean LEADER_ELIGIBILITY_DEFAULT = true;
    public static final String LEADER_ELIGIBILITY_DOC =
        "If true, this node can participate in leader election. In a multi-colo setup, turn this off "
            + "for clusters in the replica data center.";

    public static final String ROCKS_DB_ENABLE_CONFIG = "rocksdb.enable";
    public static final boolean ROCKS_DB_ENABLE_DEFAULT = true;
    public static final String ROCKS_DB_ENABLE_DOC =
        "Whether to enable RocksDB within KCache.";

    public static final String ROCKS_DB_ROOT_DIR_CONFIG = "rocksdb.root.dir";
    public static final String ROCKS_DB_ROOT_DIR_DEFAULT = "/tmp";
    public static final String ROCKS_DB_ROOT_DIR_DOC =
        "Root directory for RocksDB storage.";

    public static final String SSL_KEYSTORE_LOCATION_CONFIG = "ssl.keystore.location";
    public static final String SSL_KEYSTORE_LOCATION_DOC =
        "Location of the keystore file to use for SSL. This is required for HTTPS.";
    public static final String SSL_KEYSTORE_LOCATION_DEFAULT = "";

    public static final String SSL_KEYSTORE_PASSWORD_CONFIG = "ssl.keystore.password";
    public static final String SSL_KEYSTORE_PASSWORD_DOC =
        "The store password for the keystore file.";
    public static final String SSL_KEYSTORE_PASSWORD_DEFAULT = "";

    public static final String SSL_KEY_PASSWORD_CONFIG = "ssl.key.password";
    public static final String SSL_KEY_PASSWORD_DOC =
        "The password of the private key in the keystore file.";
    public static final String SSL_KEY_PASSWORD_DEFAULT = "";

    public static final String SSL_KEYSTORE_TYPE_CONFIG = "ssl.keystore.type";
    public static final String SSL_KEYSTORE_TYPE_DOC =
        "The type of keystore file.";
    public static final String SSL_KEYSTORE_TYPE_DEFAULT = "JKS";

    public static final String SSL_KEYMANAGER_ALGORITHM_CONFIG = "ssl.keymanager.algorithm";
    public static final String SSL_KEYMANAGER_ALGORITHM_DOC =
        "The algorithm used by the key manager factory for SSL connections. "
            + "Leave blank to use Jetty's default.";
    public static final String SSL_KEYMANAGER_ALGORITHM_DEFAULT = "";

    public static final String SSL_TRUSTSTORE_LOCATION_CONFIG = "ssl.truststore.location";
    public static final String SSL_TRUSTSTORE_LOCATION_DOC =
        "Location of the trust store. Required only to authenticate HTTPS clients.";
    public static final String SSL_TRUSTSTORE_LOCATION_DEFAULT = "";

    public static final String SSL_TRUSTSTORE_PASSWORD_CONFIG = "ssl.truststore.password";
    public static final String SSL_TRUSTSTORE_PASSWORD_DOC =
        "The store password for the trust store file.";
    public static final String SSL_TRUSTSTORE_PASSWORD_DEFAULT = "";

    public static final String SSL_TRUSTSTORE_TYPE_CONFIG = "ssl.truststore.type";
    public static final String SSL_TRUSTSTORE_TYPE_DOC =
        "The type of trust store file.";
    public static final String SSL_TRUSTSTORE_TYPE_DEFAULT = "JKS";

    public static final String SSL_TRUSTMANAGER_ALGORITHM_CONFIG = "ssl.trustmanager.algorithm";
    public static final String SSL_TRUSTMANAGER_ALGORITHM_DOC =
        "The algorithm used by the trust manager factory for SSL connections. "
            + "Leave blank to use Jetty's default.";
    public static final String SSL_TRUSTMANAGER_ALGORITHM_DEFAULT = "";

    public static final String SSL_PROTOCOL_CONFIG = "ssl.protocol";
    public static final String SSL_PROTOCOL_DOC =
        "The SSL protocol used to generate the SslContextFactory.";
    public static final String SSL_PROTOCOL_DEFAULT = "TLS";

    public static final String SSL_PROVIDER_CONFIG = "ssl.provider";
    public static final String SSL_PROVIDER_DOC =
        "The SSL security provider name. Leave blank to use Jetty's default.";
    public static final String SSL_PROVIDER_DEFAULT = "";

    public static final String SSL_CLIENT_AUTHENTICATION_CONFIG = "ssl.client.authentication";
    public static final String SSL_CLIENT_AUTHENTICATION_NONE = "NONE";
    public static final String SSL_CLIENT_AUTHENTICATION_REQUESTED = "REQUESTED";
    public static final String SSL_CLIENT_AUTHENTICATION_REQUIRED = "REQUIRED";
    public static final String SSL_CLIENT_AUTHENTICATION_DOC =
        "SSL mutual auth. Set to NONE to disable SSL client authentication, set to REQUESTED to "
            + "request but not require SSL client authentication, and set to REQUIRED to require SSL "
            + "client authentication.";
    public static final ConfigDef.ValidString SSL_CLIENT_AUTHENTICATION_VALIDATOR =
        ConfigDef.ValidString.in(
            SSL_CLIENT_AUTHENTICATION_NONE,
            SSL_CLIENT_AUTHENTICATION_REQUESTED,
            SSL_CLIENT_AUTHENTICATION_REQUIRED
        );

    public static final String SSL_ENABLED_PROTOCOLS_CONFIG = "ssl.enabled.protocols";
    public static final String SSL_ENABLED_PROTOCOLS_DOC =
        "The list of protocols enabled for SSL connections. Comma-separated list. "
            + "Leave blank to use Jetty's defaults.";
    public static final String SSL_ENABLED_PROTOCOLS_DEFAULT = "";

    public static final String SSL_CIPHER_SUITES_CONFIG = "ssl.cipher.suites";
    public static final String SSL_CIPHER_SUITES_DOC =
        "A list of SSL cipher suites. Leave blank to use Jetty's defaults.";
    public static final String SSL_CIPHER_SUITES_DEFAULT = "";

    public static final String SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_CONFIG =
        "ssl.endpoint.identification.algorithm";
    public static final String SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_DOC =
        "The endpoint identification algorithm to validate the server hostname using the "
            + "server certificate.";
    public static final String SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_DEFAULT = null;

    public static final String AUTHENTICATION_METHOD_CONFIG = "authentication.method";
    public static final String AUTHENTICATION_METHOD_NONE = "NONE";
    public static final String AUTHENTICATION_METHOD_BASIC = "BASIC";
    public static final String AUTHENTICATION_METHOD_DIGEST = "DIGEST";
    public static final String AUTHENTICATION_METHOD_DOC =
        "Method of authentication. Must be BASIC or DIGEST to enable authentication. "
            + "For BASIC or DIGEST, you must supply a valid JAAS config file for the "
            + "'java.security.auth.login.config' system property for the appropriate authentication "
            + "provider.";
    public static final ConfigDef.ValidString AUTHENTICATION_METHOD_VALIDATOR =
        ConfigDef.ValidString.in(
            AUTHENTICATION_METHOD_NONE,
            AUTHENTICATION_METHOD_BASIC,
            AUTHENTICATION_METHOD_DIGEST
        );

    public static final String AUTHENTICATION_REALM_CONFIG = "authentication.realm";
    public static final String AUTHENTICATION_REALM_DOC =
        "Security realm to be used in authentication.";

    public static final String AUTHENTICATION_ROLES_CONFIG = "authentication.roles";
    public static final String AUTHENTICATION_ROLES_DOC = "Valid roles to authenticate against.";
    public static final List<String> AUTHENTICATION_ROLES_DEFAULT =
        Collections.unmodifiableList(Arrays.asList("*"));

    private static final ConfigDef config;

    static {
        config = baseConfigDef()
            .define(
                LISTENERS_CONFIG,
                Type.LIST,
                LISTENERS_DEFAULT,
                Importance.HIGH,
                LISTENERS_DOC
            ).define(
                CLUSTER_GROUP_ID_CONFIG,
                Type.STRING,
                CLUSTER_GROUP_ID_DEFAULT,
                Importance.HIGH,
                CLUSTER_GROUP_ID_DOC
            ).define(
                LEADER_ELIGIBILITY_CONFIG,
                Type.BOOLEAN,
                LEADER_ELIGIBILITY_DEFAULT,
                Importance.MEDIUM,
                LEADER_ELIGIBILITY_DOC
            ).define(
                ROCKS_DB_ENABLE_CONFIG,
                Type.BOOLEAN,
                ROCKS_DB_ENABLE_DEFAULT,
                Importance.MEDIUM,
                ROCKS_DB_ENABLE_DOC
            ).define(
                ROCKS_DB_ROOT_DIR_CONFIG,
                Type.STRING,
                ROCKS_DB_ROOT_DIR_DEFAULT,
                Importance.MEDIUM,
                ROCKS_DB_ROOT_DIR_DOC
            ).define(
                SSL_KEYSTORE_LOCATION_CONFIG,
                Type.STRING,
                SSL_KEYSTORE_LOCATION_DEFAULT,
                Importance.HIGH,
                SSL_KEYSTORE_LOCATION_DOC
            ).define(
                SSL_KEYSTORE_PASSWORD_CONFIG,
                Type.PASSWORD,
                SSL_KEYSTORE_PASSWORD_DEFAULT,
                Importance.HIGH,
                SSL_KEYSTORE_PASSWORD_DOC
            ).define(
                SSL_KEY_PASSWORD_CONFIG,
                Type.PASSWORD,
                SSL_KEY_PASSWORD_DEFAULT,
                Importance.HIGH,
                SSL_KEY_PASSWORD_DOC
            ).define(
                SSL_KEYSTORE_TYPE_CONFIG,
                Type.STRING,
                SSL_KEYSTORE_TYPE_DEFAULT,
                Importance.MEDIUM,
                SSL_KEYSTORE_TYPE_DOC
            ).define(
                SSL_KEYMANAGER_ALGORITHM_CONFIG,
                Type.STRING,
                SSL_KEYMANAGER_ALGORITHM_DEFAULT,
                Importance.LOW,
                SSL_KEYMANAGER_ALGORITHM_DOC
            ).define(
                SSL_TRUSTSTORE_LOCATION_CONFIG,
                Type.STRING,
                SSL_TRUSTSTORE_LOCATION_DEFAULT,
                Importance.HIGH,
                SSL_TRUSTSTORE_LOCATION_DOC
            ).define(
                SSL_TRUSTSTORE_PASSWORD_CONFIG,
                Type.PASSWORD,
                SSL_TRUSTSTORE_PASSWORD_DEFAULT,
                Importance.HIGH,
                SSL_TRUSTSTORE_PASSWORD_DOC)
            .define(
                SSL_TRUSTSTORE_TYPE_CONFIG,
                Type.STRING,
                SSL_TRUSTSTORE_TYPE_DEFAULT,
                Importance.MEDIUM,
                SSL_TRUSTSTORE_TYPE_DOC)
            .define(
                SSL_TRUSTMANAGER_ALGORITHM_CONFIG,
                Type.STRING,
                SSL_TRUSTMANAGER_ALGORITHM_DEFAULT,
                Importance.LOW,
                SSL_TRUSTMANAGER_ALGORITHM_DOC
            ).define(
                SSL_PROTOCOL_CONFIG,
                Type.STRING,
                SSL_PROTOCOL_DEFAULT,
                Importance.MEDIUM,
                SSL_PROTOCOL_DOC)
            .define(
                SSL_PROVIDER_CONFIG,
                Type.STRING,
                SSL_PROVIDER_DEFAULT,
                Importance.MEDIUM,
                SSL_PROVIDER_DOC
            ).define(
                SSL_CLIENT_AUTHENTICATION_CONFIG,
                Type.STRING,
                SSL_CLIENT_AUTHENTICATION_NONE,
                SSL_CLIENT_AUTHENTICATION_VALIDATOR,
                Importance.MEDIUM,
                SSL_CLIENT_AUTHENTICATION_DOC
            ).define(
                SSL_ENABLED_PROTOCOLS_CONFIG,
                Type.LIST,
                SSL_ENABLED_PROTOCOLS_DEFAULT,
                Importance.MEDIUM,
                SSL_ENABLED_PROTOCOLS_DOC
            ).define(
                SSL_CIPHER_SUITES_CONFIG,
                Type.LIST,
                SSL_CIPHER_SUITES_DEFAULT,
                Importance.LOW,
                SSL_CIPHER_SUITES_DOC
            ).define(
                SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_CONFIG,
                Type.STRING,
                SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_DEFAULT,
                Importance.LOW,
                SSL_ENDPOINT_IDENTIFICATION_ALGORITHM_DOC
            ).define(
                AUTHENTICATION_METHOD_CONFIG,
                Type.STRING,
                AUTHENTICATION_METHOD_NONE,
                AUTHENTICATION_METHOD_VALIDATOR,
                Importance.LOW,
                AUTHENTICATION_METHOD_DOC
            ).define(
                AUTHENTICATION_REALM_CONFIG,
                Type.STRING,
                "",
                Importance.LOW,
                AUTHENTICATION_REALM_DOC
            ).define(
                AUTHENTICATION_ROLES_CONFIG,
                Type.LIST,
                AUTHENTICATION_ROLES_DEFAULT,
                Importance.LOW,
                AUTHENTICATION_ROLES_DOC
            );
    }

    public KarelDbConfig(String propsFile) {
        super(config, getPropsFromFile(propsFile));
    }

    public KarelDbConfig(Map<?, ?> props) {
        super(config, props);
    }
}
