
/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.datastore;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import java.io.Serializable;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.JsonNode;

import io.gs2.core.model.AsyncAction;
import io.gs2.core.model.AsyncResult;
import io.gs2.core.exception.*;
import io.gs2.core.net.*;
import io.gs2.core.util.EncodingUtil;

import io.gs2.core.AbstractGs2Client;
import io.gs2.datastore.request.*;
import io.gs2.datastore.result.*;
import io.gs2.datastore.model.*;public class Gs2DatastoreRestClient extends AbstractGs2Client<Gs2DatastoreRestClient> {

	public Gs2DatastoreRestClient(Gs2RestSession gs2RestSession) {
		super(gs2RestSession);
	}

    class DescribeNamespacesTask extends Gs2RestSessionTask<DescribeNamespacesResult> {
        private DescribeNamespacesRequest request;

        public DescribeNamespacesTask(
            DescribeNamespacesRequest request,
            AsyncAction<AsyncResult<DescribeNamespacesResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DescribeNamespacesResult parse(JsonNode data) {
            return DescribeNamespacesResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/";

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            if (this.request.getPageToken() != null) {
                queryStrings.add("pageToken=" + EncodingUtil.urlEncode((String.valueOf(this.request.getPageToken()))));
            }
            if (this.request.getLimit() != null) {
                queryStrings.add("limit=" + String.valueOf(this.request.getLimit()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void describeNamespacesAsync(
            DescribeNamespacesRequest request,
            AsyncAction<AsyncResult<DescribeNamespacesResult>> callback
    ) {
        DescribeNamespacesTask task = new DescribeNamespacesTask(request, callback);
        session.execute(task);
    }

    public DescribeNamespacesResult describeNamespaces(
            DescribeNamespacesRequest request
    ) {
        final AsyncResult<DescribeNamespacesResult>[] resultAsyncResult = new AsyncResult[]{null};
        describeNamespacesAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class CreateNamespaceTask extends Gs2RestSessionTask<CreateNamespaceResult> {
        private CreateNamespaceRequest request;

        public CreateNamespaceTask(
            CreateNamespaceRequest request,
            AsyncAction<AsyncResult<CreateNamespaceResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public CreateNamespaceResult parse(JsonNode data) {
            return CreateNamespaceResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/";

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("name", request.getName());
                    put("description", request.getDescription());
                    put("logSetting", request.getLogSetting() != null ? request.getLogSetting().toJson() : null);
                    put("doneUploadScript", request.getDoneUploadScript() != null ? request.getDoneUploadScript().toJson() : null);
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void createNamespaceAsync(
            CreateNamespaceRequest request,
            AsyncAction<AsyncResult<CreateNamespaceResult>> callback
    ) {
        CreateNamespaceTask task = new CreateNamespaceTask(request, callback);
        session.execute(task);
    }

    public CreateNamespaceResult createNamespace(
            CreateNamespaceRequest request
    ) {
        final AsyncResult<CreateNamespaceResult>[] resultAsyncResult = new AsyncResult[]{null};
        createNamespaceAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class GetNamespaceStatusTask extends Gs2RestSessionTask<GetNamespaceStatusResult> {
        private GetNamespaceStatusRequest request;

        public GetNamespaceStatusTask(
            GetNamespaceStatusRequest request,
            AsyncAction<AsyncResult<GetNamespaceStatusResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public GetNamespaceStatusResult parse(JsonNode data) {
            return GetNamespaceStatusResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/status";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void getNamespaceStatusAsync(
            GetNamespaceStatusRequest request,
            AsyncAction<AsyncResult<GetNamespaceStatusResult>> callback
    ) {
        GetNamespaceStatusTask task = new GetNamespaceStatusTask(request, callback);
        session.execute(task);
    }

    public GetNamespaceStatusResult getNamespaceStatus(
            GetNamespaceStatusRequest request
    ) {
        final AsyncResult<GetNamespaceStatusResult>[] resultAsyncResult = new AsyncResult[]{null};
        getNamespaceStatusAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class GetNamespaceTask extends Gs2RestSessionTask<GetNamespaceResult> {
        private GetNamespaceRequest request;

        public GetNamespaceTask(
            GetNamespaceRequest request,
            AsyncAction<AsyncResult<GetNamespaceResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public GetNamespaceResult parse(JsonNode data) {
            return GetNamespaceResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void getNamespaceAsync(
            GetNamespaceRequest request,
            AsyncAction<AsyncResult<GetNamespaceResult>> callback
    ) {
        GetNamespaceTask task = new GetNamespaceTask(request, callback);
        session.execute(task);
    }

    public GetNamespaceResult getNamespace(
            GetNamespaceRequest request
    ) {
        final AsyncResult<GetNamespaceResult>[] resultAsyncResult = new AsyncResult[]{null};
        getNamespaceAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class UpdateNamespaceTask extends Gs2RestSessionTask<UpdateNamespaceResult> {
        private UpdateNamespaceRequest request;

        public UpdateNamespaceTask(
            UpdateNamespaceRequest request,
            AsyncAction<AsyncResult<UpdateNamespaceResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public UpdateNamespaceResult parse(JsonNode data) {
            return UpdateNamespaceResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("description", request.getDescription());
                    put("logSetting", request.getLogSetting() != null ? request.getLogSetting().toJson() : null);
                    put("doneUploadScript", request.getDoneUploadScript() != null ? request.getDoneUploadScript().toJson() : null);
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.PUT)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void updateNamespaceAsync(
            UpdateNamespaceRequest request,
            AsyncAction<AsyncResult<UpdateNamespaceResult>> callback
    ) {
        UpdateNamespaceTask task = new UpdateNamespaceTask(request, callback);
        session.execute(task);
    }

    public UpdateNamespaceResult updateNamespace(
            UpdateNamespaceRequest request
    ) {
        final AsyncResult<UpdateNamespaceResult>[] resultAsyncResult = new AsyncResult[]{null};
        updateNamespaceAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DeleteNamespaceTask extends Gs2RestSessionTask<DeleteNamespaceResult> {
        private DeleteNamespaceRequest request;

        public DeleteNamespaceTask(
            DeleteNamespaceRequest request,
            AsyncAction<AsyncResult<DeleteNamespaceResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DeleteNamespaceResult parse(JsonNode data) {
            return DeleteNamespaceResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.DELETE)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void deleteNamespaceAsync(
            DeleteNamespaceRequest request,
            AsyncAction<AsyncResult<DeleteNamespaceResult>> callback
    ) {
        DeleteNamespaceTask task = new DeleteNamespaceTask(request, callback);
        session.execute(task);
    }

    public DeleteNamespaceResult deleteNamespace(
            DeleteNamespaceRequest request
    ) {
        final AsyncResult<DeleteNamespaceResult>[] resultAsyncResult = new AsyncResult[]{null};
        deleteNamespaceAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DescribeDataObjectsTask extends Gs2RestSessionTask<DescribeDataObjectsResult> {
        private DescribeDataObjectsRequest request;

        public DescribeDataObjectsTask(
            DescribeDataObjectsRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectsResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DescribeDataObjectsResult parse(JsonNode data) {
            return DescribeDataObjectsResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            if (this.request.getStatus() != null) {
                queryStrings.add("status=" + EncodingUtil.urlEncode((String.valueOf(this.request.getStatus()))));
            }
            if (this.request.getPageToken() != null) {
                queryStrings.add("pageToken=" + EncodingUtil.urlEncode((String.valueOf(this.request.getPageToken()))));
            }
            if (this.request.getLimit() != null) {
                queryStrings.add("limit=" + String.valueOf(this.request.getLimit()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void describeDataObjectsAsync(
            DescribeDataObjectsRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectsResult>> callback
    ) {
        DescribeDataObjectsTask task = new DescribeDataObjectsTask(request, callback);
        session.execute(task);
    }

    public DescribeDataObjectsResult describeDataObjects(
            DescribeDataObjectsRequest request
    ) {
        final AsyncResult<DescribeDataObjectsResult>[] resultAsyncResult = new AsyncResult[]{null};
        describeDataObjectsAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DescribeDataObjectsByUserIdTask extends Gs2RestSessionTask<DescribeDataObjectsByUserIdResult> {
        private DescribeDataObjectsByUserIdRequest request;

        public DescribeDataObjectsByUserIdTask(
            DescribeDataObjectsByUserIdRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectsByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DescribeDataObjectsByUserIdResult parse(JsonNode data) {
            return DescribeDataObjectsByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            if (this.request.getStatus() != null) {
                queryStrings.add("status=" + EncodingUtil.urlEncode((String.valueOf(this.request.getStatus()))));
            }
            if (this.request.getPageToken() != null) {
                queryStrings.add("pageToken=" + EncodingUtil.urlEncode((String.valueOf(this.request.getPageToken()))));
            }
            if (this.request.getLimit() != null) {
                queryStrings.add("limit=" + String.valueOf(this.request.getLimit()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void describeDataObjectsByUserIdAsync(
            DescribeDataObjectsByUserIdRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectsByUserIdResult>> callback
    ) {
        DescribeDataObjectsByUserIdTask task = new DescribeDataObjectsByUserIdTask(request, callback);
        session.execute(task);
    }

    public DescribeDataObjectsByUserIdResult describeDataObjectsByUserId(
            DescribeDataObjectsByUserIdRequest request
    ) {
        final AsyncResult<DescribeDataObjectsByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        describeDataObjectsByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareUploadTask extends Gs2RestSessionTask<PrepareUploadResult> {
        private PrepareUploadRequest request;

        public PrepareUploadTask(
            PrepareUploadRequest request,
            AsyncAction<AsyncResult<PrepareUploadResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareUploadResult parse(JsonNode data) {
            return PrepareUploadResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("name", request.getName());
                    put("contentType", request.getContentType());
                    put("scope", request.getScope());
                    put("allowUserIds", request.getAllowUserIds() == null ? new ArrayList<String>() :
                        request.getAllowUserIds().stream().map(item -> {
                            return item;
                        }
                    ).collect(Collectors.toList()));
                    put("updateIfExists", request.getUpdateIfExists());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareUploadAsync(
            PrepareUploadRequest request,
            AsyncAction<AsyncResult<PrepareUploadResult>> callback
    ) {
        PrepareUploadTask task = new PrepareUploadTask(request, callback);
        session.execute(task);
    }

    public PrepareUploadResult prepareUpload(
            PrepareUploadRequest request
    ) {
        final AsyncResult<PrepareUploadResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareUploadAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareUploadByUserIdTask extends Gs2RestSessionTask<PrepareUploadByUserIdResult> {
        private PrepareUploadByUserIdRequest request;

        public PrepareUploadByUserIdTask(
            PrepareUploadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareUploadByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareUploadByUserIdResult parse(JsonNode data) {
            return PrepareUploadByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("name", request.getName());
                    put("contentType", request.getContentType());
                    put("scope", request.getScope());
                    put("allowUserIds", request.getAllowUserIds() == null ? new ArrayList<String>() :
                        request.getAllowUserIds().stream().map(item -> {
                            return item;
                        }
                    ).collect(Collectors.toList()));
                    put("updateIfExists", request.getUpdateIfExists());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareUploadByUserIdAsync(
            PrepareUploadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareUploadByUserIdResult>> callback
    ) {
        PrepareUploadByUserIdTask task = new PrepareUploadByUserIdTask(request, callback);
        session.execute(task);
    }

    public PrepareUploadByUserIdResult prepareUploadByUserId(
            PrepareUploadByUserIdRequest request
    ) {
        final AsyncResult<PrepareUploadByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareUploadByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class UpdateDataObjectTask extends Gs2RestSessionTask<UpdateDataObjectResult> {
        private UpdateDataObjectRequest request;

        public UpdateDataObjectTask(
            UpdateDataObjectRequest request,
            AsyncAction<AsyncResult<UpdateDataObjectResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public UpdateDataObjectResult parse(JsonNode data) {
            return UpdateDataObjectResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("scope", request.getScope());
                    put("allowUserIds", request.getAllowUserIds() == null ? new ArrayList<String>() :
                        request.getAllowUserIds().stream().map(item -> {
                            return item;
                        }
                    ).collect(Collectors.toList()));
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void updateDataObjectAsync(
            UpdateDataObjectRequest request,
            AsyncAction<AsyncResult<UpdateDataObjectResult>> callback
    ) {
        UpdateDataObjectTask task = new UpdateDataObjectTask(request, callback);
        session.execute(task);
    }

    public UpdateDataObjectResult updateDataObject(
            UpdateDataObjectRequest request
    ) {
        final AsyncResult<UpdateDataObjectResult>[] resultAsyncResult = new AsyncResult[]{null};
        updateDataObjectAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class UpdateDataObjectByUserIdTask extends Gs2RestSessionTask<UpdateDataObjectByUserIdResult> {
        private UpdateDataObjectByUserIdRequest request;

        public UpdateDataObjectByUserIdTask(
            UpdateDataObjectByUserIdRequest request,
            AsyncAction<AsyncResult<UpdateDataObjectByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public UpdateDataObjectByUserIdResult parse(JsonNode data) {
            return UpdateDataObjectByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("scope", request.getScope());
                    put("allowUserIds", request.getAllowUserIds() == null ? new ArrayList<String>() :
                        request.getAllowUserIds().stream().map(item -> {
                            return item;
                        }
                    ).collect(Collectors.toList()));
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void updateDataObjectByUserIdAsync(
            UpdateDataObjectByUserIdRequest request,
            AsyncAction<AsyncResult<UpdateDataObjectByUserIdResult>> callback
    ) {
        UpdateDataObjectByUserIdTask task = new UpdateDataObjectByUserIdTask(request, callback);
        session.execute(task);
    }

    public UpdateDataObjectByUserIdResult updateDataObjectByUserId(
            UpdateDataObjectByUserIdRequest request
    ) {
        final AsyncResult<UpdateDataObjectByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        updateDataObjectByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareReUploadTask extends Gs2RestSessionTask<PrepareReUploadResult> {
        private PrepareReUploadRequest request;

        public PrepareReUploadTask(
            PrepareReUploadRequest request,
            AsyncAction<AsyncResult<PrepareReUploadResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareReUploadResult parse(JsonNode data) {
            return PrepareReUploadResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/file/reUpload";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contentType", request.getContentType());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareReUploadAsync(
            PrepareReUploadRequest request,
            AsyncAction<AsyncResult<PrepareReUploadResult>> callback
    ) {
        PrepareReUploadTask task = new PrepareReUploadTask(request, callback);
        session.execute(task);
    }

    public PrepareReUploadResult prepareReUpload(
            PrepareReUploadRequest request
    ) {
        final AsyncResult<PrepareReUploadResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareReUploadAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareReUploadByUserIdTask extends Gs2RestSessionTask<PrepareReUploadByUserIdResult> {
        private PrepareReUploadByUserIdRequest request;

        public PrepareReUploadByUserIdTask(
            PrepareReUploadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareReUploadByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareReUploadByUserIdResult parse(JsonNode data) {
            return PrepareReUploadByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/file/reUpload";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contentType", request.getContentType());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareReUploadByUserIdAsync(
            PrepareReUploadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareReUploadByUserIdResult>> callback
    ) {
        PrepareReUploadByUserIdTask task = new PrepareReUploadByUserIdTask(request, callback);
        session.execute(task);
    }

    public PrepareReUploadByUserIdResult prepareReUploadByUserId(
            PrepareReUploadByUserIdRequest request
    ) {
        final AsyncResult<PrepareReUploadByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareReUploadByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DoneUploadTask extends Gs2RestSessionTask<DoneUploadResult> {
        private DoneUploadRequest request;

        public DoneUploadTask(
            DoneUploadRequest request,
            AsyncAction<AsyncResult<DoneUploadResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DoneUploadResult parse(JsonNode data) {
            return DoneUploadResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/done";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void doneUploadAsync(
            DoneUploadRequest request,
            AsyncAction<AsyncResult<DoneUploadResult>> callback
    ) {
        DoneUploadTask task = new DoneUploadTask(request, callback);
        session.execute(task);
    }

    public DoneUploadResult doneUpload(
            DoneUploadRequest request
    ) {
        final AsyncResult<DoneUploadResult>[] resultAsyncResult = new AsyncResult[]{null};
        doneUploadAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DoneUploadByUserIdTask extends Gs2RestSessionTask<DoneUploadByUserIdResult> {
        private DoneUploadByUserIdRequest request;

        public DoneUploadByUserIdTask(
            DoneUploadByUserIdRequest request,
            AsyncAction<AsyncResult<DoneUploadByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DoneUploadByUserIdResult parse(JsonNode data) {
            return DoneUploadByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/done";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void doneUploadByUserIdAsync(
            DoneUploadByUserIdRequest request,
            AsyncAction<AsyncResult<DoneUploadByUserIdResult>> callback
    ) {
        DoneUploadByUserIdTask task = new DoneUploadByUserIdTask(request, callback);
        session.execute(task);
    }

    public DoneUploadByUserIdResult doneUploadByUserId(
            DoneUploadByUserIdRequest request
    ) {
        final AsyncResult<DoneUploadByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        doneUploadByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DeleteDataObjectTask extends Gs2RestSessionTask<DeleteDataObjectResult> {
        private DeleteDataObjectRequest request;

        public DeleteDataObjectTask(
            DeleteDataObjectRequest request,
            AsyncAction<AsyncResult<DeleteDataObjectResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DeleteDataObjectResult parse(JsonNode data) {
            return DeleteDataObjectResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.DELETE)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void deleteDataObjectAsync(
            DeleteDataObjectRequest request,
            AsyncAction<AsyncResult<DeleteDataObjectResult>> callback
    ) {
        DeleteDataObjectTask task = new DeleteDataObjectTask(request, callback);
        session.execute(task);
    }

    public DeleteDataObjectResult deleteDataObject(
            DeleteDataObjectRequest request
    ) {
        final AsyncResult<DeleteDataObjectResult>[] resultAsyncResult = new AsyncResult[]{null};
        deleteDataObjectAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DeleteDataObjectByUserIdTask extends Gs2RestSessionTask<DeleteDataObjectByUserIdResult> {
        private DeleteDataObjectByUserIdRequest request;

        public DeleteDataObjectByUserIdTask(
            DeleteDataObjectByUserIdRequest request,
            AsyncAction<AsyncResult<DeleteDataObjectByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DeleteDataObjectByUserIdResult parse(JsonNode data) {
            return DeleteDataObjectByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.DELETE)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void deleteDataObjectByUserIdAsync(
            DeleteDataObjectByUserIdRequest request,
            AsyncAction<AsyncResult<DeleteDataObjectByUserIdResult>> callback
    ) {
        DeleteDataObjectByUserIdTask task = new DeleteDataObjectByUserIdTask(request, callback);
        session.execute(task);
    }

    public DeleteDataObjectByUserIdResult deleteDataObjectByUserId(
            DeleteDataObjectByUserIdRequest request
    ) {
        final AsyncResult<DeleteDataObjectByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        deleteDataObjectByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadTask extends Gs2RestSessionTask<PrepareDownloadResult> {
        private PrepareDownloadRequest request;

        public PrepareDownloadTask(
            PrepareDownloadRequest request,
            AsyncAction<AsyncResult<PrepareDownloadResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadResult parse(JsonNode data) {
            return PrepareDownloadResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("dataObjectId", request.getDataObjectId());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadAsync(
            PrepareDownloadRequest request,
            AsyncAction<AsyncResult<PrepareDownloadResult>> callback
    ) {
        PrepareDownloadTask task = new PrepareDownloadTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadResult prepareDownload(
            PrepareDownloadRequest request
    ) {
        final AsyncResult<PrepareDownloadResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadByUserIdTask extends Gs2RestSessionTask<PrepareDownloadByUserIdResult> {
        private PrepareDownloadByUserIdRequest request;

        public PrepareDownloadByUserIdTask(
            PrepareDownloadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadByUserIdResult parse(JsonNode data) {
            return PrepareDownloadByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("dataObjectId", request.getDataObjectId());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadByUserIdAsync(
            PrepareDownloadByUserIdRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdResult>> callback
    ) {
        PrepareDownloadByUserIdTask task = new PrepareDownloadByUserIdTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadByUserIdResult prepareDownloadByUserId(
            PrepareDownloadByUserIdRequest request
    ) {
        final AsyncResult<PrepareDownloadByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadByGenerationTask extends Gs2RestSessionTask<PrepareDownloadByGenerationResult> {
        private PrepareDownloadByGenerationRequest request;

        public PrepareDownloadByGenerationTask(
            PrepareDownloadByGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByGenerationResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadByGenerationResult parse(JsonNode data) {
            return PrepareDownloadByGenerationResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/file/generation/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("dataObjectId", request.getDataObjectId());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadByGenerationAsync(
            PrepareDownloadByGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByGenerationResult>> callback
    ) {
        PrepareDownloadByGenerationTask task = new PrepareDownloadByGenerationTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadByGenerationResult prepareDownloadByGeneration(
            PrepareDownloadByGenerationRequest request
    ) {
        final AsyncResult<PrepareDownloadByGenerationResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadByGenerationAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadByGenerationAndUserIdTask extends Gs2RestSessionTask<PrepareDownloadByGenerationAndUserIdResult> {
        private PrepareDownloadByGenerationAndUserIdRequest request;

        public PrepareDownloadByGenerationAndUserIdTask(
            PrepareDownloadByGenerationAndUserIdRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByGenerationAndUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadByGenerationAndUserIdResult parse(JsonNode data) {
            return PrepareDownloadByGenerationAndUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/file/generation/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("dataObjectId", request.getDataObjectId());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadByGenerationAndUserIdAsync(
            PrepareDownloadByGenerationAndUserIdRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByGenerationAndUserIdResult>> callback
    ) {
        PrepareDownloadByGenerationAndUserIdTask task = new PrepareDownloadByGenerationAndUserIdTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadByGenerationAndUserIdResult prepareDownloadByGenerationAndUserId(
            PrepareDownloadByGenerationAndUserIdRequest request
    ) {
        final AsyncResult<PrepareDownloadByGenerationAndUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadByGenerationAndUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadOwnDataTask extends Gs2RestSessionTask<PrepareDownloadOwnDataResult> {
        private PrepareDownloadOwnDataRequest request;

        public PrepareDownloadOwnDataTask(
            PrepareDownloadOwnDataRequest request,
            AsyncAction<AsyncResult<PrepareDownloadOwnDataResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadOwnDataResult parse(JsonNode data) {
            return PrepareDownloadOwnDataResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadOwnDataAsync(
            PrepareDownloadOwnDataRequest request,
            AsyncAction<AsyncResult<PrepareDownloadOwnDataResult>> callback
    ) {
        PrepareDownloadOwnDataTask task = new PrepareDownloadOwnDataTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadOwnDataResult prepareDownloadOwnData(
            PrepareDownloadOwnDataRequest request
    ) {
        final AsyncResult<PrepareDownloadOwnDataResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadOwnDataAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadByUserIdAndDataObjectNameTask extends Gs2RestSessionTask<PrepareDownloadByUserIdAndDataObjectNameResult> {
        private PrepareDownloadByUserIdAndDataObjectNameRequest request;

        public PrepareDownloadByUserIdAndDataObjectNameTask(
            PrepareDownloadByUserIdAndDataObjectNameRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdAndDataObjectNameResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadByUserIdAndDataObjectNameResult parse(JsonNode data) {
            return PrepareDownloadByUserIdAndDataObjectNameResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/file";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadByUserIdAndDataObjectNameAsync(
            PrepareDownloadByUserIdAndDataObjectNameRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdAndDataObjectNameResult>> callback
    ) {
        PrepareDownloadByUserIdAndDataObjectNameTask task = new PrepareDownloadByUserIdAndDataObjectNameTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadByUserIdAndDataObjectNameResult prepareDownloadByUserIdAndDataObjectName(
            PrepareDownloadByUserIdAndDataObjectNameRequest request
    ) {
        final AsyncResult<PrepareDownloadByUserIdAndDataObjectNameResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadByUserIdAndDataObjectNameAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadOwnDataByGenerationTask extends Gs2RestSessionTask<PrepareDownloadOwnDataByGenerationResult> {
        private PrepareDownloadOwnDataByGenerationRequest request;

        public PrepareDownloadOwnDataByGenerationTask(
            PrepareDownloadOwnDataByGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadOwnDataByGenerationResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadOwnDataByGenerationResult parse(JsonNode data) {
            return PrepareDownloadOwnDataByGenerationResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/generation/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadOwnDataByGenerationAsync(
            PrepareDownloadOwnDataByGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadOwnDataByGenerationResult>> callback
    ) {
        PrepareDownloadOwnDataByGenerationTask task = new PrepareDownloadOwnDataByGenerationTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadOwnDataByGenerationResult prepareDownloadOwnDataByGeneration(
            PrepareDownloadOwnDataByGenerationRequest request
    ) {
        final AsyncResult<PrepareDownloadOwnDataByGenerationResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadOwnDataByGenerationAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class PrepareDownloadByUserIdAndDataObjectNameAndGenerationTask extends Gs2RestSessionTask<PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult> {
        private PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest request;

        public PrepareDownloadByUserIdAndDataObjectNameAndGenerationTask(
            PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult parse(JsonNode data) {
            return PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/generation/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void prepareDownloadByUserIdAndDataObjectNameAndGenerationAsync(
            PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest request,
            AsyncAction<AsyncResult<PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult>> callback
    ) {
        PrepareDownloadByUserIdAndDataObjectNameAndGenerationTask task = new PrepareDownloadByUserIdAndDataObjectNameAndGenerationTask(request, callback);
        session.execute(task);
    }

    public PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult prepareDownloadByUserIdAndDataObjectNameAndGeneration(
            PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest request
    ) {
        final AsyncResult<PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult>[] resultAsyncResult = new AsyncResult[]{null};
        prepareDownloadByUserIdAndDataObjectNameAndGenerationAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class RestoreDataObjectTask extends Gs2RestSessionTask<RestoreDataObjectResult> {
        private RestoreDataObjectRequest request;

        public RestoreDataObjectTask(
            RestoreDataObjectRequest request,
            AsyncAction<AsyncResult<RestoreDataObjectResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public RestoreDataObjectResult parse(JsonNode data) {
            return RestoreDataObjectResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/file/restore";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("dataObjectId", request.getDataObjectId());
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void restoreDataObjectAsync(
            RestoreDataObjectRequest request,
            AsyncAction<AsyncResult<RestoreDataObjectResult>> callback
    ) {
        RestoreDataObjectTask task = new RestoreDataObjectTask(request, callback);
        session.execute(task);
    }

    public RestoreDataObjectResult restoreDataObject(
            RestoreDataObjectRequest request
    ) {
        final AsyncResult<RestoreDataObjectResult>[] resultAsyncResult = new AsyncResult[]{null};
        restoreDataObjectAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DescribeDataObjectHistoriesTask extends Gs2RestSessionTask<DescribeDataObjectHistoriesResult> {
        private DescribeDataObjectHistoriesRequest request;

        public DescribeDataObjectHistoriesTask(
            DescribeDataObjectHistoriesRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectHistoriesResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DescribeDataObjectHistoriesResult parse(JsonNode data) {
            return DescribeDataObjectHistoriesResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/history";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            if (this.request.getPageToken() != null) {
                queryStrings.add("pageToken=" + EncodingUtil.urlEncode((String.valueOf(this.request.getPageToken()))));
            }
            if (this.request.getLimit() != null) {
                queryStrings.add("limit=" + String.valueOf(this.request.getLimit()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void describeDataObjectHistoriesAsync(
            DescribeDataObjectHistoriesRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectHistoriesResult>> callback
    ) {
        DescribeDataObjectHistoriesTask task = new DescribeDataObjectHistoriesTask(request, callback);
        session.execute(task);
    }

    public DescribeDataObjectHistoriesResult describeDataObjectHistories(
            DescribeDataObjectHistoriesRequest request
    ) {
        final AsyncResult<DescribeDataObjectHistoriesResult>[] resultAsyncResult = new AsyncResult[]{null};
        describeDataObjectHistoriesAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class DescribeDataObjectHistoriesByUserIdTask extends Gs2RestSessionTask<DescribeDataObjectHistoriesByUserIdResult> {
        private DescribeDataObjectHistoriesByUserIdRequest request;

        public DescribeDataObjectHistoriesByUserIdTask(
            DescribeDataObjectHistoriesByUserIdRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectHistoriesByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public DescribeDataObjectHistoriesByUserIdResult parse(JsonNode data) {
            return DescribeDataObjectHistoriesByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/history";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));

            List<String> queryStrings = new ArrayList<> ();
            if (this.request.getContextStack() != null) {
                queryStrings.add("contextStack=" + EncodingUtil.urlEncode(this.request.getContextStack()));
            }
            if (this.request.getPageToken() != null) {
                queryStrings.add("pageToken=" + EncodingUtil.urlEncode((String.valueOf(this.request.getPageToken()))));
            }
            if (this.request.getLimit() != null) {
                queryStrings.add("limit=" + String.valueOf(this.request.getLimit()));
            }
            url += "?" + String.join("&", queryStrings);

            builder
                .setMethod(HttpTask.Method.GET)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void describeDataObjectHistoriesByUserIdAsync(
            DescribeDataObjectHistoriesByUserIdRequest request,
            AsyncAction<AsyncResult<DescribeDataObjectHistoriesByUserIdResult>> callback
    ) {
        DescribeDataObjectHistoriesByUserIdTask task = new DescribeDataObjectHistoriesByUserIdTask(request, callback);
        session.execute(task);
    }

    public DescribeDataObjectHistoriesByUserIdResult describeDataObjectHistoriesByUserId(
            DescribeDataObjectHistoriesByUserIdRequest request
    ) {
        final AsyncResult<DescribeDataObjectHistoriesByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        describeDataObjectHistoriesByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class GetDataObjectHistoryTask extends Gs2RestSessionTask<GetDataObjectHistoryResult> {
        private GetDataObjectHistoryRequest request;

        public GetDataObjectHistoryTask(
            GetDataObjectHistoryRequest request,
            AsyncAction<AsyncResult<GetDataObjectHistoryResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public GetDataObjectHistoryResult parse(JsonNode data) {
            return GetDataObjectHistoryResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/me/data/{dataObjectName}/history/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }
            if (this.request.getAccessToken() != null) {
                builder.setHeader("X-GS2-ACCESS-TOKEN", this.request.getAccessToken());
            }

            builder
                .build()
                .send();
        }
    }

    public void getDataObjectHistoryAsync(
            GetDataObjectHistoryRequest request,
            AsyncAction<AsyncResult<GetDataObjectHistoryResult>> callback
    ) {
        GetDataObjectHistoryTask task = new GetDataObjectHistoryTask(request, callback);
        session.execute(task);
    }

    public GetDataObjectHistoryResult getDataObjectHistory(
            GetDataObjectHistoryRequest request
    ) {
        final AsyncResult<GetDataObjectHistoryResult>[] resultAsyncResult = new AsyncResult[]{null};
        getDataObjectHistoryAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }

    class GetDataObjectHistoryByUserIdTask extends Gs2RestSessionTask<GetDataObjectHistoryByUserIdResult> {
        private GetDataObjectHistoryByUserIdRequest request;

        public GetDataObjectHistoryByUserIdTask(
            GetDataObjectHistoryByUserIdRequest request,
            AsyncAction<AsyncResult<GetDataObjectHistoryByUserIdResult>> userCallback
        ) {
            super(
                    (Gs2RestSession) session,
                    userCallback
            );
            this.request = request;
        }

        @Override
        public GetDataObjectHistoryByUserIdResult parse(JsonNode data) {
            return GetDataObjectHistoryByUserIdResult.fromJson(data);
        }

        @Override
        protected void executeImpl() {

            String url = Gs2RestSession.EndpointHost
                .replace("{service}", "datastore")
                .replace("{region}", session.getRegion().getName())
                + "/{namespaceName}/user/{userId}/data/{dataObjectName}/history/{generation}";

            url = url.replace("{namespaceName}", this.request.getNamespaceName() == null || this.request.getNamespaceName().length() == 0 ? "null" : String.valueOf(this.request.getNamespaceName()));
            url = url.replace("{userId}", this.request.getUserId() == null || this.request.getUserId().length() == 0 ? "null" : String.valueOf(this.request.getUserId()));
            url = url.replace("{dataObjectName}", this.request.getDataObjectName() == null || this.request.getDataObjectName().length() == 0 ? "null" : String.valueOf(this.request.getDataObjectName()));
            url = url.replace("{generation}", this.request.getGeneration() == null || this.request.getGeneration().length() == 0 ? "null" : String.valueOf(this.request.getGeneration()));

            builder.setBody(new ObjectMapper().valueToTree(
                new HashMap<String, Object>() {{
                    put("contextStack", request.getContextStack());
                }}
            ).toString().getBytes());

            builder
                .setMethod(HttpTask.Method.POST)
                .setUrl(url)
                .setHeader("Content-Type", "application/json")
                .setHttpResponseHandler(this);

            if (this.request.getRequestId() != null) {
                builder.setHeader("X-GS2-REQUEST-ID", this.request.getRequestId());
            }

            builder
                .build()
                .send();
        }
    }

    public void getDataObjectHistoryByUserIdAsync(
            GetDataObjectHistoryByUserIdRequest request,
            AsyncAction<AsyncResult<GetDataObjectHistoryByUserIdResult>> callback
    ) {
        GetDataObjectHistoryByUserIdTask task = new GetDataObjectHistoryByUserIdTask(request, callback);
        session.execute(task);
    }

    public GetDataObjectHistoryByUserIdResult getDataObjectHistoryByUserId(
            GetDataObjectHistoryByUserIdRequest request
    ) {
        final AsyncResult<GetDataObjectHistoryByUserIdResult>[] resultAsyncResult = new AsyncResult[]{null};
        getDataObjectHistoryByUserIdAsync(
                request,
                result -> resultAsyncResult[0] = result
        );
        while (resultAsyncResult[0] == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {}
        }

        if(resultAsyncResult[0].getError() != null) {
            throw resultAsyncResult[0].getError();
        }

        return resultAsyncResult[0].getResult();
    }
}