/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package io.gs2.matchmaking.domain.model;

import io.gs2.core.domain.CacheDatabase;
import io.gs2.core.domain.JobQueueDomain;
import io.gs2.core.domain.StampSheetConfiguration;
import io.gs2.core.net.Gs2RestSession;
import io.gs2.auth.model.AccessToken;
import io.gs2.matchmaking.Gs2MatchmakingRestClient;
import io.gs2.matchmaking.domain.iterator.*;
import io.gs2.matchmaking.model.*;
import io.gs2.matchmaking.request.*;
import io.gs2.matchmaking.result.*;

import java.util.List;


public class GatheringDomain {
    private final CacheDatabase cache;
    private final JobQueueDomain jobQueueDomain;
    private final StampSheetConfiguration stampSheetConfiguration;
    private final Gs2RestSession session;
    private final Gs2MatchmakingRestClient client;
    private final String namespaceName;
    private final String userId;
    private final String gatheringName;

    private final String parentKey;
    public String getNamespaceName() {
        return namespaceName;
    }
    public String getUserId() {
        return userId;
    }
    public String getGatheringName() {
        return gatheringName;
    }

    public GatheringDomain(
        CacheDatabase cache,
        JobQueueDomain jobQueueDomain,
        StampSheetConfiguration stampSheetConfiguration,
        Gs2RestSession session,
        String namespaceName,
        String userId,
        String gatheringName
    ) {
        this.cache = cache;
        this.jobQueueDomain = jobQueueDomain;
        this.stampSheetConfiguration = stampSheetConfiguration;
        this.session = session;
        this.client = new Gs2MatchmakingRestClient(
            session
        );
        this.namespaceName = namespaceName;
        this.userId = userId;
        this.gatheringName = gatheringName;
        this.parentKey = io.gs2.matchmaking.domain.model.UserDomain.createCacheParentKey(
            this.namespaceName != null ? this.namespaceName.toString() : null,
            "Singleton",
            "Gathering"
        );
    }

    public GatheringDomain update(
        UpdateGatheringByUserIdRequest request
    ) {
        request
            .withNamespaceName(this.namespaceName)
            .withUserId(this.userId)
            .withGatheringName(this.gatheringName);
        UpdateGatheringByUserIdResult result = this.client.updateGatheringByUserId(
            request
        );
                
        if (result.getItem() != null) {
            cache.put(
                parentKey,
                io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                    request.getGatheringName() != null ? request.getGatheringName().toString() : null
                ),
                result.getItem(),
                result.getItem().getExpiresAt() == null ? System.currentTimeMillis() + 1000 * 60 * io.gs2.core.domain.Gs2.defaultCacheMinutes : result.getItem().getExpiresAt()
            );
        }
        GatheringDomain domain = this;

        return domain;
    }

    private Gathering get(
        GetGatheringRequest request
    ) {
        request
            .withNamespaceName(this.namespaceName)
            .withGatheringName(this.gatheringName);
        GetGatheringResult result = this.client.getGathering(
            request
        );
                
        if (result.getItem() != null) {
            cache.put(
                parentKey,
                io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                    request.getGatheringName() != null ? request.getGatheringName().toString() : null
                ),
                result.getItem(),
                result.getItem().getExpiresAt() == null ? System.currentTimeMillis() + 1000 * 60 * io.gs2.core.domain.Gs2.defaultCacheMinutes : result.getItem().getExpiresAt()
            );
        }
        return result.getItem();
    }

    public GatheringDomain cancelMatchmaking(
        CancelMatchmakingByUserIdRequest request
    ) {
        request
            .withNamespaceName(this.namespaceName)
            .withUserId(this.userId)
            .withGatheringName(this.gatheringName);
        CancelMatchmakingByUserIdResult result = null;
        try {
            result = this.client.cancelMatchmakingByUserId(
                request
            );
        } catch(io.gs2.core.exception.NotFoundException e) {}
        cache.delete(
            parentKey,
            io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                request.getGatheringName() != null ? request.getGatheringName().toString() : null
            ),
            Gathering.class
        );
        GatheringDomain domain = this;

        return domain;
    }

    public static String createCacheParentKey(
        String namespaceName,
        String userId,
        String gatheringName,
        String childType
    )
    {
        return String.join(
            ":",
            "matchmaking",
            namespaceName,
            userId,
            gatheringName,
            childType
        );
    }

    public static String createCacheKey(
        String gatheringName
    )
    {
        return String.join(
            ":",
            gatheringName
        );
    }

    public Gathering model() {
        Gathering value = cache.get(
            parentKey,
            io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                this.getGatheringName() != null ? this.getGatheringName().toString() : null
            ),
            Gathering.class
        );
        if (value == null) {
            try {
                this.get(
                    new GetGatheringRequest()
                );
            } catch(io.gs2.core.exception.NotFoundException e) {
                cache.delete(
                    parentKey,
                    io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                        this.getGatheringName() != null ? this.getGatheringName().toString() : null
                    ),
                    Gathering.class
                );
            }
            value = cache.get(
            parentKey,
            io.gs2.matchmaking.domain.model.GatheringDomain.createCacheKey(
                this.getGatheringName() != null ? this.getGatheringName().toString() : null
            ),
            Gathering.class
        );
        }
        return value;
    }

}
