/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.mission;

import java.util.ArrayList;
import java.util.List;

import io.gs2.model.Region;
import io.gs2.util.EncodingUtil;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONObject;

import io.gs2.AbstractGs2Client;
import io.gs2.Gs2Constant;
import io.gs2.model.IGs2Credential;
import io.gs2.mission.request.*;
import io.gs2.mission.result.*;
import io.gs2.mission.model.*;

/**
 * GS2 Mission API クライアント
 *
 * @author Game Server Services, Inc.
 *
 */
public class Gs2MissionRestClient extends AbstractGs2Client<Gs2MissionRestClient> {

	public static String ENDPOINT = "mission";

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 */
	public Gs2MissionRestClient(IGs2Credential credential) {
		super(credential);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2MissionRestClient(IGs2Credential credential, Region region) {
		super(credential, region);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2MissionRestClient(IGs2Credential credential, String region) {
		super(credential, region);

	}

	/**
	 * ミッショングループの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeMissionGroupModelsResult describeMissionGroupModels(DescribeMissionGroupModelsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelFunctionHandler.describeMissionGroupModels";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeMissionGroupModelsResult.class);
    }

	/**
	 * ミッショングループを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetMissionGroupModelResult getMissionGroupModel(GetMissionGroupModelRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelFunctionHandler.getMissionGroupModel";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetMissionGroupModelResult.class);
    }

	/**
	 * ミッショングループマスターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeMissionGroupModelMastersResult describeMissionGroupModelMasters(DescribeMissionGroupModelMastersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelMasterFunctionHandler.describeMissionGroupModelMasters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeMissionGroupModelMastersResult.class);
    }

	/**
	 * ミッショングループマスターを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateMissionGroupModelMasterResult createMissionGroupModelMaster(CreateMissionGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelMasterFunctionHandler.createMissionGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String completeNotificationNamespaceId = null;
        if(request.getCompleteNotificationNamespaceId() != null) {
            completeNotificationNamespaceId = request.getCompleteNotificationNamespaceId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(completeNotificationNamespaceId != null) {
            _body.put("completeNotificationNamespaceId", completeNotificationNamespaceId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateMissionGroupModelMasterResult.class);
    }

	/**
	 * ミッショングループマスターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetMissionGroupModelMasterResult getMissionGroupModelMaster(GetMissionGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelMasterFunctionHandler.getMissionGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetMissionGroupModelMasterResult.class);
    }

	/**
	 * ミッショングループマスターを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateMissionGroupModelMasterResult updateMissionGroupModelMaster(UpdateMissionGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelMasterFunctionHandler.updateMissionGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String completeNotificationNamespaceId = null;
        if(request.getCompleteNotificationNamespaceId() != null) {
            completeNotificationNamespaceId = request.getCompleteNotificationNamespaceId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(completeNotificationNamespaceId != null) {
            _body.put("completeNotificationNamespaceId", completeNotificationNamespaceId);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateMissionGroupModelMasterResult.class);
    }

	/**
	 * ミッショングループマスターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteMissionGroupModelMasterResult deleteMissionGroupModelMaster(DeleteMissionGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionGroupModelMasterFunctionHandler.deleteMissionGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteMissionGroupModelMasterResult.class);
    }

	/**
	 * カウンターの種類の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCounterModelsResult describeCounterModels(DescribeCounterModelsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelFunctionHandler.describeCounterModels";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeCounterModelsResult.class);
    }

	/**
	 * カウンターの種類を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCounterModelResult getCounterModel(GetCounterModelRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelFunctionHandler.getCounterModel";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetCounterModelResult.class);
    }

	/**
	 * ミッションタスクの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeMissionTaskModelsResult describeMissionTaskModels(DescribeMissionTaskModelsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelFunctionHandler.describeMissionTaskModels";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeMissionTaskModelsResult.class);
    }

	/**
	 * ミッションタスクを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetMissionTaskModelResult getMissionTaskModel(GetMissionTaskModelRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelFunctionHandler.getMissionTaskModel";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetMissionTaskModelResult.class);
    }

	/**
	 * カウンターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCountersResult describeCounters(DescribeCountersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.describeCounters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCountersResult.class);
    }

	/**
	 * ユーザIDを指定してカウンターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCountersByUserIdResult describeCountersByUserId(DescribeCountersByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.describeCountersByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCountersByUserIdResult.class);
    }

	/**
	 * カウンターに加算<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public IncreaseCounterByUserIdResult increaseCounterByUserId(IncreaseCounterByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.increaseCounterByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        Long value = null;
        if(request.getValue() != null) {
            value = request.getValue();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(value != null) {
            _body.put("value", value);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, IncreaseCounterByUserIdResult.class);
    }

	/**
	 * カウンターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCounterResult getCounter(GetCounterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.getCounter";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCounterResult.class);
    }

	/**
	 * ユーザIDを指定してカウンターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCounterByUserIdResult getCounterByUserId(GetCounterByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.getCounterByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCounterByUserIdResult.class);
    }

	/**
	 * カウンターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteCounterByUserIdResult deleteCounterByUserId(DeleteCounterByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.deleteCounterByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteCounterByUserIdResult.class);
    }

	/**
	 * カウンター加算<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public IncreaseByStampSheetResult increaseByStampSheet(IncreaseByStampSheetRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterFunctionHandler.increaseByStampSheet";
        String stampSheet = null;
        if(request.getStampSheet() != null) {
            stampSheet = request.getStampSheet();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(stampSheet != null) {
            _body.put("stampSheet", stampSheet);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, IncreaseByStampSheetResult.class);
    }

	/**
	 * 現在有効なミッションのマスターデータをエクスポートします<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public ExportMasterResult exportMaster(ExportMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCurrentMissionMasterFunctionHandler.exportMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, ExportMasterResult.class);
    }

	/**
	 * 現在有効な現在有効なミッションを取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCurrentMissionMasterResult getCurrentMissionMaster(GetCurrentMissionMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCurrentMissionMasterFunctionHandler.getCurrentMissionMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetCurrentMissionMasterResult.class);
    }

	/**
	 * 現在有効な現在有効なミッションを更新します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateCurrentMissionMasterResult updateCurrentMissionMaster(UpdateCurrentMissionMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCurrentMissionMasterFunctionHandler.updateCurrentMissionMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String settings = null;
        if(request.getSettings() != null) {
            settings = request.getSettings();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(settings != null) {
            _body.put("settings", settings);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateCurrentMissionMasterResult.class);
    }

	/**
	 * ミッションタスクマスターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeMissionTaskModelMastersResult describeMissionTaskModelMasters(DescribeMissionTaskModelMastersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelMasterFunctionHandler.describeMissionTaskModelMasters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeMissionTaskModelMastersResult.class);
    }

	/**
	 * ミッションタスクマスターを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateMissionTaskModelMasterResult createMissionTaskModelMaster(CreateMissionTaskModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelMasterFunctionHandler.createMissionTaskModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        String resetType = null;
        if(request.getResetType() != null) {
            resetType = request.getResetType();
        }
        Long targetValue = null;
        if(request.getTargetValue() != null) {
            targetValue = request.getTargetValue();
        }
        ArrayNode completeAcquireActions = null;
        if(request.getCompleteAcquireActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(AcquireAction item : request.getCompleteAcquireActions()) {
                node.add(item.toJson());
            }
            completeAcquireActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        String premiseMissionTaskName = null;
        if(request.getPremiseMissionTaskName() != null) {
            premiseMissionTaskName = request.getPremiseMissionTaskName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(counterName != null) {
            _body.put("counterName", counterName);
        }
        if(resetType != null) {
            _body.put("resetType", resetType);
        }
        if(targetValue != null) {
            _body.put("targetValue", targetValue);
        }
        if(completeAcquireActions != null) {
            _body.put("completeAcquireActions", completeAcquireActions);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
        if(premiseMissionTaskName != null) {
            _body.put("premiseMissionTaskName", premiseMissionTaskName);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateMissionTaskModelMasterResult.class);
    }

	/**
	 * ミッションタスクマスターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetMissionTaskModelMasterResult getMissionTaskModelMaster(GetMissionTaskModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelMasterFunctionHandler.getMissionTaskModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetMissionTaskModelMasterResult.class);
    }

	/**
	 * ミッションタスクマスターを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateMissionTaskModelMasterResult updateMissionTaskModelMaster(UpdateMissionTaskModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelMasterFunctionHandler.updateMissionTaskModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        String resetType = null;
        if(request.getResetType() != null) {
            resetType = request.getResetType();
        }
        Long targetValue = null;
        if(request.getTargetValue() != null) {
            targetValue = request.getTargetValue();
        }
        ArrayNode completeAcquireActions = null;
        if(request.getCompleteAcquireActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(AcquireAction item : request.getCompleteAcquireActions()) {
                node.add(item.toJson());
            }
            completeAcquireActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        String premiseMissionTaskName = null;
        if(request.getPremiseMissionTaskName() != null) {
            premiseMissionTaskName = request.getPremiseMissionTaskName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(counterName != null) {
            _body.put("counterName", counterName);
        }
        if(resetType != null) {
            _body.put("resetType", resetType);
        }
        if(targetValue != null) {
            _body.put("targetValue", targetValue);
        }
        if(completeAcquireActions != null) {
            _body.put("completeAcquireActions", completeAcquireActions);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
        if(premiseMissionTaskName != null) {
            _body.put("premiseMissionTaskName", premiseMissionTaskName);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateMissionTaskModelMasterResult.class);
    }

	/**
	 * ミッションタスクマスターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteMissionTaskModelMasterResult deleteMissionTaskModelMaster(DeleteMissionTaskModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FMissionTaskModelMasterFunctionHandler.deleteMissionTaskModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteMissionTaskModelMasterResult.class);
    }

	/**
	 * ネームスペースの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeNamespacesResult describeNamespaces(DescribeNamespacesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.describeNamespaces";
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeNamespacesResult.class);
    }

	/**
	 * ネームスペースを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateNamespaceResult createNamespace(CreateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.createNamespace";
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String missionCompleteTriggerScriptId = null;
        if(request.getMissionCompleteTriggerScriptId() != null) {
            missionCompleteTriggerScriptId = request.getMissionCompleteTriggerScriptId();
        }
        String missionCompleteDoneTriggerScriptId = null;
        if(request.getMissionCompleteDoneTriggerScriptId() != null) {
            missionCompleteDoneTriggerScriptId = request.getMissionCompleteDoneTriggerScriptId();
        }
        String missionCompleteDoneTriggerQueueNamespaceId = null;
        if(request.getMissionCompleteDoneTriggerQueueNamespaceId() != null) {
            missionCompleteDoneTriggerQueueNamespaceId = request.getMissionCompleteDoneTriggerQueueNamespaceId();
        }
        String counterIncrementTriggerScriptId = null;
        if(request.getCounterIncrementTriggerScriptId() != null) {
            counterIncrementTriggerScriptId = request.getCounterIncrementTriggerScriptId();
        }
        String counterIncrementDoneTriggerScriptId = null;
        if(request.getCounterIncrementDoneTriggerScriptId() != null) {
            counterIncrementDoneTriggerScriptId = request.getCounterIncrementDoneTriggerScriptId();
        }
        String counterIncrementDoneTriggerQueueNamespaceId = null;
        if(request.getCounterIncrementDoneTriggerQueueNamespaceId() != null) {
            counterIncrementDoneTriggerQueueNamespaceId = request.getCounterIncrementDoneTriggerQueueNamespaceId();
        }
        String receiveRewardsTriggerScriptId = null;
        if(request.getReceiveRewardsTriggerScriptId() != null) {
            receiveRewardsTriggerScriptId = request.getReceiveRewardsTriggerScriptId();
        }
        String receiveRewardsDoneTriggerScriptId = null;
        if(request.getReceiveRewardsDoneTriggerScriptId() != null) {
            receiveRewardsDoneTriggerScriptId = request.getReceiveRewardsDoneTriggerScriptId();
        }
        String receiveRewardsDoneTriggerQueueNamespaceId = null;
        if(request.getReceiveRewardsDoneTriggerQueueNamespaceId() != null) {
            receiveRewardsDoneTriggerQueueNamespaceId = request.getReceiveRewardsDoneTriggerQueueNamespaceId();
        }
        String queueNamespaceId = null;
        if(request.getQueueNamespaceId() != null) {
            queueNamespaceId = request.getQueueNamespaceId();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }
        NotificationSetting completeNotification = null;
        if(request.getCompleteNotification() != null) {
            completeNotification = request.getCompleteNotification();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(missionCompleteTriggerScriptId != null) {
            _body.put("missionCompleteTriggerScriptId", missionCompleteTriggerScriptId);
        }
        if(missionCompleteDoneTriggerScriptId != null) {
            _body.put("missionCompleteDoneTriggerScriptId", missionCompleteDoneTriggerScriptId);
        }
        if(missionCompleteDoneTriggerQueueNamespaceId != null) {
            _body.put("missionCompleteDoneTriggerQueueNamespaceId", missionCompleteDoneTriggerQueueNamespaceId);
        }
        if(counterIncrementTriggerScriptId != null) {
            _body.put("counterIncrementTriggerScriptId", counterIncrementTriggerScriptId);
        }
        if(counterIncrementDoneTriggerScriptId != null) {
            _body.put("counterIncrementDoneTriggerScriptId", counterIncrementDoneTriggerScriptId);
        }
        if(counterIncrementDoneTriggerQueueNamespaceId != null) {
            _body.put("counterIncrementDoneTriggerQueueNamespaceId", counterIncrementDoneTriggerQueueNamespaceId);
        }
        if(receiveRewardsTriggerScriptId != null) {
            _body.put("receiveRewardsTriggerScriptId", receiveRewardsTriggerScriptId);
        }
        if(receiveRewardsDoneTriggerScriptId != null) {
            _body.put("receiveRewardsDoneTriggerScriptId", receiveRewardsDoneTriggerScriptId);
        }
        if(receiveRewardsDoneTriggerQueueNamespaceId != null) {
            _body.put("receiveRewardsDoneTriggerQueueNamespaceId", receiveRewardsDoneTriggerQueueNamespaceId);
        }
        if(queueNamespaceId != null) {
            _body.put("queueNamespaceId", queueNamespaceId);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
        if(completeNotification != null) {
            _body.put("completeNotification", completeNotification.toJson());
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateNamespaceResult.class);
    }

	/**
	 * ネームスペースの状態を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceStatusResult getNamespaceStatus(GetNamespaceStatusRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.getNamespaceStatus";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceStatusResult.class);
    }

	/**
	 * ネームスペースを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceResult getNamespace(GetNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.getNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceResult.class);
    }

	/**
	 * ネームスペースを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateNamespaceResult updateNamespace(UpdateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.updateNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String missionCompleteTriggerScriptId = null;
        if(request.getMissionCompleteTriggerScriptId() != null) {
            missionCompleteTriggerScriptId = request.getMissionCompleteTriggerScriptId();
        }
        String missionCompleteDoneTriggerScriptId = null;
        if(request.getMissionCompleteDoneTriggerScriptId() != null) {
            missionCompleteDoneTriggerScriptId = request.getMissionCompleteDoneTriggerScriptId();
        }
        String missionCompleteDoneTriggerQueueNamespaceId = null;
        if(request.getMissionCompleteDoneTriggerQueueNamespaceId() != null) {
            missionCompleteDoneTriggerQueueNamespaceId = request.getMissionCompleteDoneTriggerQueueNamespaceId();
        }
        String counterIncrementTriggerScriptId = null;
        if(request.getCounterIncrementTriggerScriptId() != null) {
            counterIncrementTriggerScriptId = request.getCounterIncrementTriggerScriptId();
        }
        String counterIncrementDoneTriggerScriptId = null;
        if(request.getCounterIncrementDoneTriggerScriptId() != null) {
            counterIncrementDoneTriggerScriptId = request.getCounterIncrementDoneTriggerScriptId();
        }
        String counterIncrementDoneTriggerQueueNamespaceId = null;
        if(request.getCounterIncrementDoneTriggerQueueNamespaceId() != null) {
            counterIncrementDoneTriggerQueueNamespaceId = request.getCounterIncrementDoneTriggerQueueNamespaceId();
        }
        String receiveRewardsTriggerScriptId = null;
        if(request.getReceiveRewardsTriggerScriptId() != null) {
            receiveRewardsTriggerScriptId = request.getReceiveRewardsTriggerScriptId();
        }
        String receiveRewardsDoneTriggerScriptId = null;
        if(request.getReceiveRewardsDoneTriggerScriptId() != null) {
            receiveRewardsDoneTriggerScriptId = request.getReceiveRewardsDoneTriggerScriptId();
        }
        String receiveRewardsDoneTriggerQueueNamespaceId = null;
        if(request.getReceiveRewardsDoneTriggerQueueNamespaceId() != null) {
            receiveRewardsDoneTriggerQueueNamespaceId = request.getReceiveRewardsDoneTriggerQueueNamespaceId();
        }
        String queueNamespaceId = null;
        if(request.getQueueNamespaceId() != null) {
            queueNamespaceId = request.getQueueNamespaceId();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }
        NotificationSetting completeNotification = null;
        if(request.getCompleteNotification() != null) {
            completeNotification = request.getCompleteNotification();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(missionCompleteTriggerScriptId != null) {
            _body.put("missionCompleteTriggerScriptId", missionCompleteTriggerScriptId);
        }
        if(missionCompleteDoneTriggerScriptId != null) {
            _body.put("missionCompleteDoneTriggerScriptId", missionCompleteDoneTriggerScriptId);
        }
        if(missionCompleteDoneTriggerQueueNamespaceId != null) {
            _body.put("missionCompleteDoneTriggerQueueNamespaceId", missionCompleteDoneTriggerQueueNamespaceId);
        }
        if(counterIncrementTriggerScriptId != null) {
            _body.put("counterIncrementTriggerScriptId", counterIncrementTriggerScriptId);
        }
        if(counterIncrementDoneTriggerScriptId != null) {
            _body.put("counterIncrementDoneTriggerScriptId", counterIncrementDoneTriggerScriptId);
        }
        if(counterIncrementDoneTriggerQueueNamespaceId != null) {
            _body.put("counterIncrementDoneTriggerQueueNamespaceId", counterIncrementDoneTriggerQueueNamespaceId);
        }
        if(receiveRewardsTriggerScriptId != null) {
            _body.put("receiveRewardsTriggerScriptId", receiveRewardsTriggerScriptId);
        }
        if(receiveRewardsDoneTriggerScriptId != null) {
            _body.put("receiveRewardsDoneTriggerScriptId", receiveRewardsDoneTriggerScriptId);
        }
        if(receiveRewardsDoneTriggerQueueNamespaceId != null) {
            _body.put("receiveRewardsDoneTriggerQueueNamespaceId", receiveRewardsDoneTriggerQueueNamespaceId);
        }
        if(queueNamespaceId != null) {
            _body.put("queueNamespaceId", queueNamespaceId);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
        if(completeNotification != null) {
            _body.put("completeNotification", completeNotification.toJson());
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateNamespaceResult.class);
    }

	/**
	 * ネームスペースを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteNamespaceResult deleteNamespace(DeleteNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FNamespaceFunctionHandler.deleteNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteNamespaceResult.class);
    }

	/**
	 * カウンターの種類マスターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCounterModelMastersResult describeCounterModelMasters(DescribeCounterModelMastersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelMasterFunctionHandler.describeCounterModelMasters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeCounterModelMastersResult.class);
    }

	/**
	 * カウンターの種類マスターを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateCounterModelMasterResult createCounterModelMaster(CreateCounterModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelMasterFunctionHandler.createCounterModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        ArrayNode scopes = null;
        if(request.getScopes() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(CounterScopeModel item : request.getScopes()) {
                node.add(item.toJson());
            }
            scopes = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(scopes != null) {
            _body.put("scopes", scopes);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateCounterModelMasterResult.class);
    }

	/**
	 * カウンターの種類マスターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCounterModelMasterResult getCounterModelMaster(GetCounterModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelMasterFunctionHandler.getCounterModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetCounterModelMasterResult.class);
    }

	/**
	 * カウンターの種類マスターを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateCounterModelMasterResult updateCounterModelMaster(UpdateCounterModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelMasterFunctionHandler.updateCounterModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        ArrayNode scopes = null;
        if(request.getScopes() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(CounterScopeModel item : request.getScopes()) {
                node.add(item.toJson());
            }
            scopes = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(scopes != null) {
            _body.put("scopes", scopes);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateCounterModelMasterResult.class);
    }

	/**
	 * カウンターの種類マスターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteCounterModelMasterResult deleteCounterModelMaster(DeleteCounterModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCounterModelMasterFunctionHandler.deleteCounterModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String counterName = null;
        if(request.getCounterName() != null) {
            counterName = request.getCounterName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(counterName != null) {
            _queryString.add(new BasicNameValuePair("counterName", String.valueOf(counterName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteCounterModelMasterResult.class);
    }

	/**
	 * 達成状況の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCompletesResult describeCompletes(DescribeCompletesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.describeCompletes";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCompletesResult.class);
    }

	/**
	 * ユーザIDを指定して達成状況の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCompletesByUserIdResult describeCompletesByUserId(DescribeCompletesByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.describeCompletesByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCompletesByUserIdResult.class);
    }

	/**
	 * ミッション達成報酬を受領するためのスタンプシートを発行<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CompleteResult complete(CompleteRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.complete";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, CompleteResult.class);
    }

	/**
	 * 達成状況を新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CompleteByUserIdResult completeByUserId(CompleteByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.completeByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, CompleteByUserIdResult.class);
    }

	/**
	 * ミッション達成報酬を受領する<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public ReceiveByUserIdResult receiveByUserId(ReceiveByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.receiveByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String missionTaskName = null;
        if(request.getMissionTaskName() != null) {
            missionTaskName = request.getMissionTaskName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(missionTaskName != null) {
            _queryString.add(new BasicNameValuePair("missionTaskName", String.valueOf(missionTaskName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, ReceiveByUserIdResult.class);
    }

	/**
	 * 達成状況を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCompleteResult getComplete(GetCompleteRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.getComplete";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCompleteResult.class);
    }

	/**
	 * ユーザIDを指定して達成状況を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCompleteByUserIdResult getCompleteByUserId(GetCompleteByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.getCompleteByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCompleteByUserIdResult.class);
    }

	/**
	 * 達成状況を削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteCompleteByUserIdResult deleteCompleteByUserId(DeleteCompleteByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.deleteCompleteByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String missionGroupName = null;
        if(request.getMissionGroupName() != null) {
            missionGroupName = request.getMissionGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(missionGroupName != null) {
            _queryString.add(new BasicNameValuePair("missionGroupName", String.valueOf(missionGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteCompleteByUserIdResult.class);
    }

	/**
	 * 達成状況を作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public ReceiveByStampTaskResult receiveByStampTask(ReceiveByStampTaskRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/mission-handler?handler=gs2_mission%2Fhandler%2FCompleteFunctionHandler.receiveByStampTask";
        String stampTask = null;
        if(request.getStampTask() != null) {
            stampTask = request.getStampTask();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(stampTask != null) {
            _body.put("stampTask", stampTask);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, ReceiveByStampTaskResult.class);
    }
}