/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jspecify.annotations.NullMarked;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({"source", "type", "cvssData"})
@NullMarked
public class CvssV4 implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 4239377501678853019L;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CvssV3.
     *
     * @param source The source for the CvssV3 (required JSON property)
     * @param type The type for the CvssV3 (required JSON property)
     * @param cvssData The cvssData for the CvssV3 (required JSON property)
     */
    @JsonCreator
    public CvssV4(@JsonProperty("source") String source, @JsonProperty("type") Type type,
            @JsonProperty("cvssData") CvssV4Data cvssData) {
        this.source = source;
        this.type = type;
        this.cvssData = cvssData;
    }

    /**
     * (Required)
     */
    @JsonProperty("source")
    private String source;
    /**
     * (Required)
     */
    @JsonProperty("type")
    private Type type;
    /**
     * JSON Schema for Common Vulnerability Scoring System version 4.0
     * <p>
     * (Required)
     */
    @JsonProperty("cvssData")
    private CvssV4Data cvssData;

    /**
     * (Required)
     *
     * @return source
     */
    @JsonProperty("source")
    public String getSource() {
        return source;
    }

    /**
     * (Required)
     *
     * @return type
     */
    @JsonProperty("type")
    public Type getType() {
        return type;
    }

    /**
     * JSON Schema for Common Vulnerability Scoring System version 4.0
     * <p>
     * (Required)
     *
     * @return cvssData
     */
    @JsonProperty("cvssData")
    public CvssV4Data getCvssData() {
        return cvssData;
    }

    @Override
    public String toString() {
        return cvssData.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CvssV4 cvssV3 = (CvssV4) o;
        return Objects.equals(source, cvssV3.source) && type == cvssV3.type
                && Objects.equals(cvssData, cvssV3.cvssData);
    }

    @Override
    public int hashCode() {
        return Objects.hash(source, type, cvssData);
    }

    public enum Type {

        PRIMARY("Primary"), SECONDARY("Secondary");

        private static final Map<String, Type> CONSTANTS = new HashMap<>();

        static {
            for (Type c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Type(String value) {
            this.value = value;
        }

        @JsonCreator
        public static Type fromValue(String value) {
            Type constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
