/*
 * Decompiled with CFR 0.152.
 */
package io.fusionauth.http.server.internal;

import io.fusionauth.http.log.Logger;
import io.fusionauth.http.security.SecurityTools;
import io.fusionauth.http.server.HTTPListenerConfiguration;
import io.fusionauth.http.server.HTTPServerConfiguration;
import io.fusionauth.http.server.Instrumenter;
import io.fusionauth.http.server.internal.HTTPWorker;
import io.fusionauth.http.server.io.Throughput;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.security.GeneralSecurityException;
import java.util.Deque;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedDeque;
import javax.net.ssl.SSLContext;

public class HTTPServerThread
extends Thread {
    private final HTTPServerCleanerThread cleaner;
    private final Deque<ClientInfo> clients = new ConcurrentLinkedDeque<ClientInfo>();
    private final HTTPServerConfiguration configuration;
    private final Instrumenter instrumenter;
    private final HTTPListenerConfiguration listener;
    private final Logger logger;
    private final long minimumReadThroughput;
    private final long minimumWriteThroughput;
    private final ServerSocket socket;
    private volatile boolean running;

    public HTTPServerThread(HTTPServerConfiguration configuration, HTTPListenerConfiguration listener) throws IOException, GeneralSecurityException {
        super("HTTP server [" + listener.getBindAddress().toString() + ":" + listener.getPort() + "]");
        this.configuration = configuration;
        this.listener = listener;
        this.instrumenter = configuration.getInstrumenter();
        this.logger = configuration.getLoggerFactory().getLogger(HTTPServerThread.class);
        this.minimumReadThroughput = configuration.getMinimumReadThroughput();
        this.minimumWriteThroughput = configuration.getMinimumWriteThroughput();
        this.cleaner = new HTTPServerCleanerThread();
        if (listener.isTLS()) {
            SSLContext context = SecurityTools.serverContext(listener.getCertificateChain(), listener.getPrivateKey());
            this.socket = context.getServerSocketFactory().createServerSocket();
        } else {
            this.socket = new ServerSocket();
        }
        this.socket.setSoTimeout(0);
        this.socket.bind(new InetSocketAddress(listener.getBindAddress(), listener.getPort()));
        if (this.instrumenter != null) {
            this.instrumenter.serverStarted();
        }
    }

    @Override
    public void run() {
        this.running = true;
        this.cleaner.start();
        while (this.running) {
            try {
                Socket clientSocket = this.socket.accept();
                clientSocket.setSoTimeout((int)this.configuration.getInitialReadTimeoutDuration().toMillis());
                this.logger.debug("Accepted inbound connection with [{}] existing connections and initial read timeout of [{}]", this.clients.size(), this.configuration.getInitialReadTimeoutDuration().toMillis());
                if (this.instrumenter != null) {
                    this.instrumenter.acceptedConnection();
                }
                Throughput throughput = new Throughput(this.configuration.getReadThroughputCalculationDelay().toMillis(), this.configuration.getWriteThroughputCalculationDelay().toMillis());
                HTTPWorker runnable = new HTTPWorker(clientSocket, this.configuration, this.instrumenter, this.listener, throughput);
                Thread client = Thread.ofVirtual().name("HTTP client [" + String.valueOf(clientSocket.getRemoteSocketAddress()) + "]").start(runnable);
                this.clients.add(new ClientInfo(client, runnable, throughput));
            }
            catch (SocketTimeoutException ignore) {
                this.logger.debug("Nothing accepted. Cleaning up existing connections.");
            }
            catch (SocketException e) {
                if (this.socket.isClosed()) {
                    this.running = false;
                    this.logger.debug("The server socket was closed. Shutting down the server.", e);
                    continue;
                }
                this.logger.error("An exception was thrown while accepting incoming connections.", e);
            }
            catch (IOException ignore) {
                this.logger.debug("IO exception. Likely a fuzzer or a bad client or a TLS issue, all of which are common and can mostly be ignored.");
            }
            catch (Throwable t) {
                this.logger.error("An exception was thrown during server processing. This is a fatal issue and we need to shutdown the server.", t);
                break;
            }
        }
        for (ClientInfo client : this.clients) {
            client.thread().interrupt();
        }
    }

    public void shutdown() {
        this.running = false;
        try {
            this.cleaner.interrupt();
            this.socket.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private class HTTPServerCleanerThread
    extends Thread {
        public HTTPServerCleanerThread() {
            super("Cleaner for HTTP server [" + HTTPServerThread.this.listener.getBindAddress().toString() + ":" + HTTPServerThread.this.listener.getPort() + "]");
        }

        @Override
        public void run() {
            while (HTTPServerThread.this.running) {
                HTTPServerThread.this.logger.debug("Cleaning things up");
                Iterator<ClientInfo> iterator = HTTPServerThread.this.clients.iterator();
                while (iterator.hasNext()) {
                    ClientInfo client = iterator.next();
                    Thread thread = client.thread();
                    if (!thread.isAlive()) {
                        HTTPServerThread.this.logger.debug("Thread is dead. Removing.");
                        iterator.remove();
                        continue;
                    }
                    long now = System.currentTimeMillis();
                    Throughput throughput = client.throughput();
                    HTTPWorker worker = client.runnable();
                    HTTPWorker.State state = worker.state();
                    long workerLastUsed = throughput.lastUsed();
                    boolean readingSlow = false;
                    boolean writingSlow = false;
                    boolean timedOut = false;
                    boolean badClient = false;
                    if (state == HTTPWorker.State.Read) {
                        readingSlow = badClient = throughput.readThroughput(now) < HTTPServerThread.this.minimumReadThroughput;
                    } else if (state == HTTPWorker.State.Write) {
                        writingSlow = badClient = throughput.writeThroughput(now) < HTTPServerThread.this.minimumWriteThroughput;
                    } else if (state == HTTPWorker.State.Process) {
                        timedOut = badClient = now - workerLastUsed > HTTPServerThread.this.configuration.getProcessingTimeoutDuration().toMillis();
                    }
                    HTTPServerThread.this.logger.debug("Checking client readingSlow=[{}] writingSlow=[{}] timedOut=[{}] writeThroughput=[{}] minWriteThroughput=[{}]", readingSlow, writingSlow, timedOut, throughput.writeThroughput(now), HTTPServerThread.this.minimumWriteThroughput);
                    if (!badClient) continue;
                    iterator.remove();
                    if (HTTPServerThread.this.logger.isDebugEnabled()) {
                        Object message = "";
                        if (readingSlow) {
                            message = (String)message + String.format(" Min read throughput [%s], actual throughput [%s].", HTTPServerThread.this.minimumReadThroughput, throughput.readThroughput(now));
                        }
                        if (writingSlow) {
                            message = (String)message + String.format(" Min write throughput [%s], actual throughput [%s].", HTTPServerThread.this.minimumWriteThroughput, throughput.writeThroughput(now));
                        }
                        if (timedOut) {
                            message = (String)message + String.format(" Connection timed out while processing. Last used [%s]ms ago. Configured client timeout [%s]ms.", now - workerLastUsed, HTTPServerThread.this.configuration.getProcessingTimeoutDuration().toMillis());
                        }
                        HTTPServerThread.this.logger.debug("Closing connection readingSlow=[{}] writingSlow=[{}] timedOut=[{}] {}", readingSlow, writingSlow, timedOut, message);
                        HTTPServerThread.this.logger.debug("Closing client connection [{}] due to inactivity", worker.getSocket().getRemoteSocketAddress());
                        StringBuilder threadDump = new StringBuilder();
                        for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
                            threadDump.append(entry.getKey()).append(" ").append((Object)entry.getKey().getState()).append("\n");
                            for (StackTraceElement ste : entry.getValue()) {
                                threadDump.append("\tat ").append(ste).append("\n");
                            }
                            threadDump.append("\n");
                        }
                        HTTPServerThread.this.logger.debug("Thread dump from server side.\n" + String.valueOf(threadDump));
                    }
                    try {
                        Socket socket = worker.getSocket();
                        socket.close();
                        if (HTTPServerThread.this.instrumenter == null) continue;
                        HTTPServerThread.this.instrumenter.connectionClosed();
                    }
                    catch (IOException e) {
                        HTTPServerThread.this.logger.debug("Unable to close connection to client. [{}]", e);
                    }
                }
                try {
                    HTTPServerCleanerThread.sleep(2000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    record ClientInfo(Thread thread, HTTPWorker runnable, Throughput throughput) {
    }
}

