/*
 * Decompiled with CFR 0.152.
 */
package io.fusionauth.http.server;

import io.fusionauth.http.ClientAbortException;
import io.fusionauth.http.ClientSSLHandshakeException;
import io.fusionauth.http.ParseException;
import io.fusionauth.http.log.Logger;
import io.fusionauth.http.server.HTTP11Processor;
import io.fusionauth.http.server.HTTPListenerConfiguration;
import io.fusionauth.http.server.HTTPProcessor;
import io.fusionauth.http.server.HTTPS11Processor;
import io.fusionauth.http.server.HTTPServerConfiguration;
import io.fusionauth.http.server.Instrumenter;
import io.fusionauth.http.server.Notifier;
import io.fusionauth.http.server.ProcessorState;
import io.fusionauth.http.util.ThreadPool;
import java.io.Closeable;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.security.GeneralSecurityException;
import java.time.Duration;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class HTTPServerThread
extends Thread
implements Closeable,
Notifier {
    private final ServerSocketChannel channel;
    private final Duration clientTimeout;
    private final HTTPServerConfiguration configuration;
    private final Instrumenter instrumenter;
    private final HTTPListenerConfiguration listenerConfiguration;
    private final Logger logger;
    private final long minimumReadThroughput;
    private final long minimumWriteThroughput;
    private final ByteBuffer preambleBuffer;
    private final Selector selector;
    private final ThreadPool threadPool;
    private volatile boolean running = true;

    public HTTPServerThread(HTTPServerConfiguration configuration, HTTPListenerConfiguration listenerConfiguration, ThreadPool threadPool) throws IOException {
        super("HTTP Server Thread");
        this.clientTimeout = configuration.getClientTimeoutDuration();
        this.configuration = configuration;
        this.listenerConfiguration = listenerConfiguration;
        this.instrumenter = configuration.getInstrumenter();
        this.logger = configuration.getLoggerFactory().getLogger(HTTPServerThread.class);
        this.minimumReadThroughput = configuration.getMinimumReadThroughput();
        this.minimumWriteThroughput = configuration.getMinimumWriteThroughput();
        this.preambleBuffer = ByteBuffer.allocate(configuration.getPreambleBufferSize());
        this.selector = Selector.open();
        this.threadPool = threadPool;
        this.channel = ServerSocketChannel.open();
        this.channel.configureBlocking(false);
        this.channel.bind(new InetSocketAddress(listenerConfiguration.getBindAddress(), listenerConfiguration.getPort()));
        this.channel.register(this.selector, 16);
        if (this.instrumenter != null) {
            this.instrumenter.serverStarted();
        }
    }

    @Override
    public void close() {
        try {
            this.running = false;
            this.selector.wakeup();
            this.join(2000L);
        }
        catch (InterruptedException e) {
            this.logger.error("Unable to shutdown the HTTP server thread after waiting for 2 seconds. \ud83e\udd37\ud83c\udffb\u200d\ufe0f");
        }
        Set<SelectionKey> keys = this.selector.keys();
        for (SelectionKey key : keys) {
            this.cancelAndCloseKey(key);
        }
        try {
            this.selector.close();
        }
        catch (Throwable t) {
            this.logger.error("Unable to close the Selector.", t);
        }
        try {
            this.channel.close();
        }
        catch (Throwable t) {
            this.logger.error("Unable to close the Channel.", t);
        }
        this.notifyNow();
    }

    @Override
    public void notifyNow() {
        if (!this.selector.isOpen()) {
            return;
        }
        this.selector.wakeup();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (this.running) {
            SelectionKey key = null;
            try {
                this.selector.select(1000L);
                Set<SelectionKey> keys = this.selector.selectedKeys();
                Iterator<SelectionKey> iterator = keys.iterator();
                while (iterator.hasNext()) {
                    key = iterator.next();
                    if (key.isAcceptable()) {
                        this.accept(key);
                    } else if (key.isReadable()) {
                        this.read(key);
                    } else if (key.isWritable()) {
                        this.write(key);
                    }
                    iterator.remove();
                    key = null;
                    this.preambleBuffer.clear();
                }
                this.cleanup();
            }
            catch (ClosedSelectorException cse) {
                break;
            }
            catch (SocketException se) {
                this.logger.debug("A socket exception was thrown during processing. These are pretty common.", se);
                this.cancelAndCloseKey(key);
            }
            catch (ParseException pe) {
                this.logger.debug("The HTTP request was unparseable. These are pretty common with fuzzers/hackers, so we just debug them here.", pe);
                if (this.instrumenter != null) {
                    this.instrumenter.badRequest();
                }
                this.cancelAndCloseKey(key);
            }
            catch (ClientAbortException | ClientSSLHandshakeException e) {
                this.logger.debug("A client related exception was thrown during processing", e);
                this.cancelAndCloseKey(key);
            }
            catch (Throwable t) {
                this.logger.error("An exception was thrown during processing", t);
                this.cancelAndCloseKey(key);
            }
            finally {
                this.preambleBuffer.clear();
            }
        }
    }

    private void accept(SelectionKey key) throws GeneralSecurityException, IOException {
        SocketChannel client = this.channel.accept();
        HTTP11Processor httpProcessor = new HTTP11Processor(this.configuration, this.listenerConfiguration, this, this.preambleBuffer, this.threadPool, this.ipAddress(client));
        HTTPS11Processor tlsProcessor = new HTTPS11Processor(httpProcessor, this.configuration, this.listenerConfiguration);
        client.configureBlocking(false);
        client.register(key.selector(), tlsProcessor.initialKeyOps(), tlsProcessor);
        if (this.logger.isDebugEnabled()) {
            try {
                this.logger.debug("Accepted connection from client [{}]", client.getRemoteAddress().toString());
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        if (this.instrumenter != null) {
            this.instrumenter.acceptedConnection();
        }
    }

    private void cancelAndCloseKey(SelectionKey key) {
        if (key == null) {
            return;
        }
        try (SelectableChannel client = key.channel();){
            if (this.logger.isDebugEnabled() && client instanceof SocketChannel) {
                SocketChannel socketChannel = (SocketChannel)client;
                this.logger.debug("Closing connection to client [{}]", socketChannel.getRemoteAddress().toString());
            }
            if (key.attachment() != null) {
                ProcessorState state = ((HTTPProcessor)key.attachment()).close(true);
                if (state == ProcessorState.Read) {
                    this.logger.trace("(HTTP-SERVER-CLOSE-READ)");
                    key.interestOps(1);
                    return;
                }
                if (state == ProcessorState.Write) {
                    this.logger.trace("(HTTP-SERVER-CLOSE-WRITE)");
                    key.interestOps(4);
                    return;
                }
            }
            client.close();
            key.cancel();
            if (client.validOps() != 16 && this.instrumenter != null) {
                this.instrumenter.connectionClosed();
            }
        }
        catch (Throwable t) {
            this.logger.error("An exception was thrown while trying to cancel a SelectionKey and close a channel with a client due to an exception being thrown for that specific client. Enable debug logging to see the error", t);
        }
        this.logger.trace("(C)");
    }

    private void cleanup() {
        long now = System.currentTimeMillis();
        for (SelectionKey key : this.selector.keys()) {
            if (key.attachment() == null) continue;
            HTTPProcessor processor = (HTTPProcessor)key.attachment();
            ProcessorState state = processor.state();
            boolean readingSlow = state == ProcessorState.Read && processor.readThroughput() < this.minimumReadThroughput;
            boolean writingSlow = state == ProcessorState.Write && processor.writeThroughput() < this.minimumWriteThroughput;
            boolean timedOut = processor.lastUsed() < now - this.clientTimeout.toMillis();
            boolean badChannel = readingSlow || writingSlow || timedOut;
            if (!badChannel) continue;
            if (this.logger.isDebugEnabled()) {
                Object message = "";
                if (readingSlow) {
                    message = (String)message + String.format(" Min read throughput [%s], actual throughput [%s].", this.minimumReadThroughput, processor.readThroughput());
                }
                if (writingSlow) {
                    message = (String)message + String.format(" Min write throughput [%s], actual throughput [%s].", this.minimumWriteThroughput, processor.writeThroughput());
                }
                if (timedOut) {
                    message = (String)message + String.format(" Connection timed out. Configured client timeout [%s] ms.", this.clientTimeout.toMillis());
                }
                this.logger.debug("Closing connection readingSlow=[{}] writingSlow=[{}] timedOut=[{}]{}", readingSlow, writingSlow, timedOut, message);
                SocketChannel client = (SocketChannel)key.channel();
                try {
                    this.logger.debug("Closing client connection [{}] due to inactivity", client.getRemoteAddress().toString());
                    StringBuilder threadDump = new StringBuilder();
                    for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
                        threadDump.append(entry.getKey()).append(" ").append((Object)entry.getKey().getState()).append("\n");
                        for (StackTraceElement ste : entry.getValue()) {
                            threadDump.append("\tat ").append(ste).append("\n");
                        }
                        threadDump.append("\n");
                    }
                    this.logger.debug("Thread dump from server side.\n" + threadDump);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            this.cancelAndCloseKey(key);
        }
    }

    private String ipAddress(SocketChannel client) throws IOException {
        InetSocketAddress ipAddress = (InetSocketAddress)client.getRemoteAddress();
        return ipAddress.getAddress().getHostAddress();
    }

    private void read(SelectionKey key) throws IOException {
        ByteBuffer buffer;
        HTTPProcessor processor = (HTTPProcessor)key.attachment();
        ProcessorState state = processor.state();
        this.logger.trace("(R) {}", new Object[]{state});
        SocketChannel client = (SocketChannel)key.channel();
        if (state == ProcessorState.Read && (buffer = processor.readBuffer()) != null) {
            int num;
            try {
                num = client.read(buffer);
            }
            catch (IOException e) {
                throw new ClientAbortException(e);
            }
            if (num < 0) {
                this.logger.debug("Client terminated the connection. Num bytes is [{}]. Closing connection", num);
                state = processor.close(true);
            } else {
                this.logger.debug("Read [{}] bytes from client", num);
                buffer.flip();
                state = processor.read(buffer);
                if (this.instrumenter != null) {
                    this.instrumenter.readFromClient(num);
                }
            }
        }
        if (state == ProcessorState.Close) {
            this.cancelAndCloseKey(key);
        } else if (state == ProcessorState.Write) {
            key.interestOps(4);
        }
    }

    private void write(SelectionKey key) throws IOException {
        HTTPS11Processor processor = (HTTPS11Processor)key.attachment();
        ProcessorState state = processor.state();
        SocketChannel client = (SocketChannel)key.channel();
        ByteBuffer[] buffers = processor.writeBuffers();
        if (state == ProcessorState.Write) {
            long num = 0L;
            if (buffers != null) {
                try {
                    num = client.write(buffers);
                }
                catch (IOException e) {
                    throw new ClientAbortException(e);
                }
            }
            if (num < 0L) {
                this.logger.debug("Client refused bytes or terminated the connection. Num bytes is [{}]. Closing connection", num);
                processor.close(true);
            } else {
                if (num > 0L) {
                    this.logger.debug("Wrote [{}] bytes to the client", num);
                    if (this.instrumenter != null) {
                        this.instrumenter.wroteToClient(num);
                    }
                }
                state = processor.wrote(num);
            }
        }
        if (state == ProcessorState.Close) {
            this.cancelAndCloseKey(key);
        } else if (state == ProcessorState.Read) {
            key.interestOps(1);
        } else if (state == ProcessorState.Reset) {
            HTTP11Processor httpProcessor = new HTTP11Processor(this.configuration, this.listenerConfiguration, this, this.preambleBuffer, this.threadPool, this.ipAddress(client));
            processor.updateDelegate(httpProcessor);
            key.interestOps(1);
        }
    }
}

