/*
 * Licensed to Metamarkets Group Inc. (Metamarkets) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. Metamarkets licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package io.druid.segment;

import org.joda.time.Interval;

import java.io.Closeable;

/**
 */
public interface Segment extends Closeable
{
  public String getIdentifier();
  public Interval getDataInterval();
  public QueryableIndex asQueryableIndex();
  public StorageAdapter asStorageAdapter();
  
  /**
   * Request an implementation of a particular interface.
   *
   * If the passed-in interface is {@link QueryableIndex} or {@link StorageAdapter}, then this method behaves
   * identically to {@link #asQueryableIndex()} or {@link #asStorageAdapter()}. Other interfaces are only
   * expected to be requested by callers that have specific knowledge of extra features provided by specific
   * segment types. For example, an extension might provide a custom Segment type that can offer both
   * StorageAdapter and some new interface. That extension can also offer a Query that uses that new interface.
   * 
   * Implementations which accept classes other than {@link QueryableIndex} or {@link StorageAdapter} are limited 
   * to using those classes within the extension. This means that one extension cannot rely on the `Segment.as` 
   * behavior of another extension.
   *
   * @param clazz desired interface
   * @param <T> desired interface
   * @return instance of clazz, or null if the interface is not supported by this segment
   */
  public <T> T as(Class<T> clazz);
}
