/*
 * Decompiled with CFR 0.152.
 */
package io.druid.segment.data;

import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.metamx.common.logger.Logger;
import com.ning.compress.BufferRecycler;
import com.ning.compress.lzf.LZFDecoder;
import com.ning.compress.lzf.LZFEncoder;
import io.druid.collections.ResourceHolder;
import io.druid.segment.CompressedPools;
import io.druid.segment.data.ObjectStrategy;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Map;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4FastDecompressor;
import net.jpountz.lz4.LZ4SafeDecompressor;

public class CompressedObjectStrategy<T extends Buffer>
implements ObjectStrategy<ResourceHolder<T>> {
    private static final Logger log = new Logger(CompressedObjectStrategy.class);
    public static final CompressionStrategy DEFAULT_COMPRESSION_STRATEGY = CompressionStrategy.LZ4;
    protected final ByteOrder order;
    protected final BufferConverter<T> converter;
    protected final Decompressor decompressor;
    private final Compressor compressor;

    protected CompressedObjectStrategy(ByteOrder order, BufferConverter<T> converter, CompressionStrategy compression) {
        this.order = order;
        this.converter = converter;
        this.decompressor = compression.getDecompressor();
        this.compressor = compression.getCompressor();
    }

    @Override
    public Class<? extends ResourceHolder<T>> getClazz() {
        return ResourceHolder.class;
    }

    @Override
    public ResourceHolder<T> fromByteBuffer(ByteBuffer buffer, int numBytes) {
        final ResourceHolder<ByteBuffer> bufHolder = CompressedPools.getByteBuf(this.order);
        final ByteBuffer buf = (ByteBuffer)bufHolder.get();
        buf.position(0);
        buf.limit(buf.capacity());
        this.decompress(buffer, numBytes, buf);
        return new ResourceHolder<T>(){

            public T get() {
                return (Buffer)CompressedObjectStrategy.this.converter.convert(buf);
            }

            public void close() throws IOException {
                bufHolder.close();
            }
        };
    }

    protected void decompress(ByteBuffer buffer, int numBytes, ByteBuffer buf) {
        this.decompressor.decompress(buffer, numBytes, buf);
    }

    @Override
    public byte[] toBytes(ResourceHolder<T> holder) {
        Buffer val = (Buffer)holder.get();
        ByteBuffer buf = this.bufferFor(val);
        this.converter.combine(buf, val);
        return this.compressor.compress(buf.array());
    }

    protected ByteBuffer bufferFor(T val) {
        return ByteBuffer.allocate(this.converter.sizeOf(((Buffer)val).remaining())).order(this.order);
    }

    @Override
    public int compare(ResourceHolder<T> o1, ResourceHolder<T> o2) {
        return this.converter.compare(o1.get(), o2.get());
    }

    public static interface BufferConverter<T> {
        public T convert(ByteBuffer var1);

        public int compare(T var1, T var2);

        public int sizeOf(int var1);

        public T combine(ByteBuffer var1, T var2);
    }

    public static class LZ4Compressor
    implements Compressor {
        private static final LZ4Compressor defaultCompressor = new LZ4Compressor();
        private static final net.jpountz.lz4.LZ4Compressor lz4High = LZ4Factory.fastestInstance().highCompressor();

        @Override
        public byte[] compress(byte[] bytes) {
            return lz4High.compress(bytes);
        }
    }

    public static class LZ4Decompressor
    implements Decompressor {
        private static final LZ4SafeDecompressor lz4Safe = LZ4Factory.fastestInstance().safeDecompressor();
        private static final LZ4FastDecompressor lz4Fast = LZ4Factory.fastestInstance().fastDecompressor();
        private static final LZ4Decompressor defaultDecompressor = new LZ4Decompressor();

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out) {
            int numDecompressedBytes = lz4Safe.decompress(in, in.position(), numBytes, out, out.position(), out.remaining());
            out.limit(out.position() + numDecompressedBytes);
        }

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out, int decompressedSize) {
            lz4Fast.decompress(in, in.position(), out, out.position(), decompressedSize);
            out.limit(out.position() + decompressedSize);
        }
    }

    public static class LZFCompressor
    implements Compressor {
        private static final LZFCompressor defaultCompressor = new LZFCompressor();

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public byte[] compress(byte[] bytes) {
            try (ResourceHolder<BufferRecycler> bufferRecycler = CompressedPools.getBufferRecycler();){
                byte[] byArray = LZFEncoder.encode((byte[])bytes, (int)0, (int)bytes.length, (BufferRecycler)((BufferRecycler)bufferRecycler.get()));
                return byArray;
            }
            catch (IOException e) {
                log.error((Throwable)e, "Error compressing data", new Object[0]);
                throw Throwables.propagate((Throwable)e);
            }
        }
    }

    public static class LZFDecompressor
    implements Decompressor {
        private static final LZFDecompressor defaultDecompressor = new LZFDecompressor();

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out) {
            byte[] bytes = new byte[numBytes];
            in.get(bytes);
            try (ResourceHolder<byte[]> outputBytesHolder = CompressedPools.getOutputBytes();){
                byte[] outputBytes = (byte[])outputBytesHolder.get();
                int numDecompressedBytes = LZFDecoder.decode((byte[])bytes, (byte[])outputBytes);
                out.put(outputBytes, 0, numDecompressedBytes);
                out.flip();
            }
            catch (IOException e) {
                log.error((Throwable)e, "Error decompressing data", new Object[0]);
            }
        }

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out, int decompressedSize) {
            this.decompress(in, numBytes, out);
        }
    }

    public static class UncompressedDecompressor
    implements Decompressor {
        private static final UncompressedDecompressor defaultDecompressor = new UncompressedDecompressor();

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out) {
            ByteBuffer copyBuffer = in.duplicate();
            copyBuffer.limit(copyBuffer.position() + numBytes);
            out.put(copyBuffer).flip();
            in.position(in.position() + numBytes);
        }

        @Override
        public void decompress(ByteBuffer in, int numBytes, ByteBuffer out, int decompressedSize) {
            this.decompress(in, numBytes, out);
        }
    }

    public static class UncompressedCompressor
    implements Compressor {
        private static final UncompressedCompressor defaultCompressor = new UncompressedCompressor();

        @Override
        public byte[] compress(byte[] bytes) {
            return bytes;
        }
    }

    public static interface Compressor {
        public byte[] compress(byte[] var1);
    }

    public static interface Decompressor {
        public void decompress(ByteBuffer var1, int var2, ByteBuffer var3);

        public void decompress(ByteBuffer var1, int var2, ByteBuffer var3, int var4);
    }

    public static enum CompressionStrategy {
        LZF(0){

            @Override
            public Decompressor getDecompressor() {
                return LZFDecompressor.defaultDecompressor;
            }

            @Override
            public Compressor getCompressor() {
                return LZFCompressor.defaultCompressor;
            }
        }
        ,
        LZ4(1){

            @Override
            public Decompressor getDecompressor() {
                return LZ4Decompressor.defaultDecompressor;
            }

            @Override
            public Compressor getCompressor() {
                return LZ4Compressor.defaultCompressor;
            }
        }
        ,
        UNCOMPRESSED(-1){

            @Override
            public Decompressor getDecompressor() {
                return UncompressedDecompressor.defaultDecompressor;
            }

            @Override
            public Compressor getCompressor() {
                return UncompressedCompressor.defaultCompressor;
            }
        };

        final byte id;
        static final Map<Byte, CompressionStrategy> idMap;

        private CompressionStrategy(byte id) {
            this.id = id;
        }

        public byte getId() {
            return this.id;
        }

        public abstract Compressor getCompressor();

        public abstract Decompressor getDecompressor();

        public static CompressionStrategy forId(byte id) {
            return idMap.get(id);
        }

        static {
            idMap = Maps.newHashMap();
            for (CompressionStrategy strategy : CompressionStrategy.values()) {
                idMap.put(strategy.getId(), strategy);
            }
        }
    }
}

