/*
 * Decompiled with CFR 0.152.
 */
package io.druid.query.aggregation.hyperloglog;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import io.druid.query.aggregation.hyperloglog.HyperLogLogCollector;
import io.druid.query.aggregation.hyperloglog.HyperUniquesAggregatorFactory;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Random;
import javax.annotation.Nullable;
import org.apache.commons.codec.binary.Base64;
import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

public class HyperLogLogCollectorTest {
    private final HashFunction fn = Hashing.murmur3_128();

    @Test
    public void testFolding() throws Exception {
        int[] numValsToCheck;
        Random random = new Random(0L);
        for (int numThings : numValsToCheck = new int[]{10, 20, 50, 100, 1000, 2000}) {
            HyperLogLogCollector allCombined = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector oneHalf = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector otherHalf = HyperLogLogCollector.makeLatestCollector();
            for (int i = 0; i < numThings; ++i) {
                byte[] hashedVal = this.fn.hashLong(random.nextLong()).asBytes();
                allCombined.add(hashedVal);
                if (i % 2 == 0) {
                    oneHalf.add(hashedVal);
                    continue;
                }
                otherHalf.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeLatestCollector();
            folded.fold(oneHalf);
            Assert.assertEquals((Object)oneHalf, (Object)folded);
            Assert.assertEquals((double)oneHalf.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
            folded.fold(otherHalf);
            Assert.assertEquals((Object)allCombined, (Object)folded);
            Assert.assertEquals((double)allCombined.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
        }
    }

    @Ignore
    @Test
    public void testHighCardinalityRollingFold() throws Exception {
        int count;
        HyperLogLogCollector rolling = HyperLogLogCollector.makeLatestCollector();
        HyperLogLogCollector simple = HyperLogLogCollector.makeLatestCollector();
        MessageDigest md = MessageDigest.getInstance("SHA-1");
        HyperLogLogCollector tmp = HyperLogLogCollector.makeLatestCollector();
        for (count = 0; count < 100000000; ++count) {
            md.update(Integer.toString(count).getBytes());
            byte[] hashed = this.fn.hashBytes(md.digest()).asBytes();
            tmp.add(hashed);
            simple.add(hashed);
            if (count % 100 != 0) continue;
            rolling.fold(tmp);
            tmp = HyperLogLogCollector.makeLatestCollector();
        }
        int n = count;
        System.out.println("True cardinality " + n);
        System.out.println("Rolling buffer cardinality " + rolling.estimateCardinality());
        System.out.println("Simple  buffer cardinality " + simple.estimateCardinality());
        System.out.println(String.format("Rolling cardinality estimate off by %4.1f%%", 100.0 * (1.0 - rolling.estimateCardinality() / (double)n)));
        Assert.assertEquals((double)n, (double)simple.estimateCardinality(), (double)((double)n * 0.05));
        Assert.assertEquals((double)n, (double)rolling.estimateCardinality(), (double)((double)n * 0.05));
    }

    @Ignore
    @Test
    public void testHighCardinalityRollingFold2() throws Exception {
        int count;
        HyperLogLogCollector rolling = HyperLogLogCollector.makeLatestCollector();
        long start = System.currentTimeMillis();
        for (count = 0; count < 50000000; ++count) {
            HyperLogLogCollector theCollector = HyperLogLogCollector.makeLatestCollector();
            theCollector.add(this.fn.hashLong((long)count).asBytes());
            rolling.fold(theCollector);
        }
        System.out.printf("testHighCardinalityRollingFold2 took %d ms%n", System.currentTimeMillis() - start);
        int n = count;
        System.out.println("True cardinality " + n);
        System.out.println("Rolling buffer cardinality " + rolling.estimateCardinality());
        System.out.println(String.format("Rolling cardinality estimate off by %4.1f%%", 100.0 * (1.0 - rolling.estimateCardinality() / (double)n)));
        Assert.assertEquals((double)n, (double)rolling.estimateCardinality(), (double)((double)n * 0.05));
    }

    @Test
    public void testFoldingByteBuffers() throws Exception {
        int[] numValsToCheck;
        Random random = new Random(0L);
        for (int numThings : numValsToCheck = new int[]{10, 20, 50, 100, 1000, 2000}) {
            HyperLogLogCollector allCombined = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector oneHalf = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector otherHalf = HyperLogLogCollector.makeLatestCollector();
            for (int i = 0; i < numThings; ++i) {
                byte[] hashedVal = this.fn.hashLong(random.nextLong()).asBytes();
                allCombined.add(hashedVal);
                if (i % 2 == 0) {
                    oneHalf.add(hashedVal);
                    continue;
                }
                otherHalf.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeLatestCollector();
            folded.fold(oneHalf.toByteBuffer());
            Assert.assertEquals((Object)oneHalf, (Object)folded);
            Assert.assertEquals((double)oneHalf.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
            folded.fold(otherHalf.toByteBuffer());
            Assert.assertEquals((Object)allCombined, (Object)folded);
            Assert.assertEquals((double)allCombined.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
        }
    }

    @Test
    public void testFoldingReadOnlyByteBuffers() throws Exception {
        int[] numValsToCheck;
        Random random = new Random(0L);
        for (int numThings : numValsToCheck = new int[]{10, 20, 50, 100, 1000, 2000}) {
            HyperLogLogCollector allCombined = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector oneHalf = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector otherHalf = HyperLogLogCollector.makeLatestCollector();
            for (int i = 0; i < numThings; ++i) {
                byte[] hashedVal = this.fn.hashLong(random.nextLong()).asBytes();
                allCombined.add(hashedVal);
                if (i % 2 == 0) {
                    oneHalf.add(hashedVal);
                    continue;
                }
                otherHalf.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeCollector((ByteBuffer)ByteBuffer.wrap(HyperLogLogCollector.makeEmptyVersionedByteArray()).asReadOnlyBuffer());
            folded.fold(oneHalf.toByteBuffer());
            Assert.assertEquals((Object)oneHalf, (Object)folded);
            Assert.assertEquals((double)oneHalf.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
            folded.fold(otherHalf.toByteBuffer());
            Assert.assertEquals((Object)allCombined, (Object)folded);
            Assert.assertEquals((double)allCombined.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
        }
    }

    @Test
    public void testFoldingReadOnlyByteBuffersWithArbitraryPosition() throws Exception {
        int[] numValsToCheck;
        Random random = new Random(0L);
        for (int numThings : numValsToCheck = new int[]{10, 20, 50, 100, 1000, 2000}) {
            HyperLogLogCollector allCombined = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector oneHalf = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector otherHalf = HyperLogLogCollector.makeLatestCollector();
            for (int i = 0; i < numThings; ++i) {
                byte[] hashedVal = this.fn.hashLong(random.nextLong()).asBytes();
                allCombined.add(hashedVal);
                if (i % 2 == 0) {
                    oneHalf.add(hashedVal);
                    continue;
                }
                otherHalf.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeCollector((ByteBuffer)this.shiftedBuffer(ByteBuffer.wrap(HyperLogLogCollector.makeEmptyVersionedByteArray()).asReadOnlyBuffer(), 17));
            folded.fold(oneHalf.toByteBuffer());
            Assert.assertEquals((Object)oneHalf, (Object)folded);
            Assert.assertEquals((double)oneHalf.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
            folded.fold(otherHalf.toByteBuffer());
            Assert.assertEquals((Object)allCombined, (Object)folded);
            Assert.assertEquals((double)allCombined.estimateCardinality(), (double)folded.estimateCardinality(), (double)0.0);
        }
    }

    @Test
    public void testFoldWithDifferentOffsets1() throws Exception {
        ByteBuffer biggerOffset = this.makeCollectorBuffer(1, (byte)0, 17);
        ByteBuffer smallerOffset = this.makeCollectorBuffer(0, (byte)32, 0);
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(biggerOffset);
        collector.fold(smallerOffset);
        ByteBuffer outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)2047L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertEquals((long)outBuffer.get(), (long)16L);
        while (outBuffer.hasRemaining()) {
            Assert.assertEquals((long)outBuffer.get(), (long)17L);
        }
        collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(smallerOffset);
        collector.fold(biggerOffset);
        outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)2047L);
        Assert.assertEquals((long)outBuffer.get(), (long)0L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        Assert.assertEquals((long)outBuffer.get(), (long)16L);
        while (outBuffer.hasRemaining()) {
            Assert.assertEquals((long)outBuffer.get(), (long)17L);
        }
    }

    @Test
    public void testBufferSwap() throws Exception {
        ByteBuffer biggerOffset = this.makeCollectorBuffer(1, (byte)0, 17);
        ByteBuffer smallerOffset = this.makeCollectorBuffer(0, (byte)32, 0);
        ByteBuffer buffer = ByteBuffer.allocate(HyperLogLogCollector.getLatestNumBytesForDenseStorage());
        HyperLogLogCollector collector = HyperLogLogCollector.makeCollector((ByteBuffer)buffer);
        collector.fold(biggerOffset);
        Assert.assertEquals((Object)collector, (Object)HyperLogLogCollector.makeCollector((ByteBuffer)buffer));
        collector.fold(smallerOffset);
        Assert.assertEquals((Object)collector, (Object)HyperLogLogCollector.makeCollector((ByteBuffer)buffer));
    }

    @Test
    public void testFoldWithArbitraryInitialPositions() throws Exception {
        ByteBuffer biggerOffset = this.shiftedBuffer(this.makeCollectorBuffer(1, (byte)0, 17), 10);
        ByteBuffer smallerOffset = this.shiftedBuffer(this.makeCollectorBuffer(0, (byte)32, 0), 15);
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(biggerOffset);
        collector.fold(smallerOffset);
        ByteBuffer outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)2047L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertEquals((long)outBuffer.get(), (long)16L);
        while (outBuffer.hasRemaining()) {
            Assert.assertEquals((long)outBuffer.get(), (long)17L);
        }
        collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(smallerOffset);
        collector.fold(biggerOffset);
        outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)2047L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertEquals((long)outBuffer.get(), (long)16L);
        while (outBuffer.hasRemaining()) {
            Assert.assertEquals((long)outBuffer.get(), (long)17L);
        }
    }

    protected ByteBuffer shiftedBuffer(ByteBuffer buf, int offset) {
        ByteBuffer shifted = ByteBuffer.allocate(buf.remaining() + offset);
        shifted.position(offset);
        shifted.put(buf);
        shifted.position(offset);
        return shifted;
    }

    @Test
    public void testFoldWithDifferentOffsets2() throws Exception {
        ByteBuffer biggerOffset = this.makeCollectorBuffer(1, (byte)1, 17);
        ByteBuffer smallerOffset = this.makeCollectorBuffer(0, (byte)32, 0);
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(biggerOffset);
        collector.fold(smallerOffset);
        ByteBuffer outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)2L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertFalse((boolean)outBuffer.hasRemaining());
        collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(smallerOffset);
        collector.fold(biggerOffset);
        outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)2L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertFalse((boolean)outBuffer.hasRemaining());
    }

    @Test
    public void testFoldWithUpperNibbleTriggersOffsetChange() throws Exception {
        byte[] arr1 = new byte[HyperLogLogCollector.getLatestNumBytesForDenseStorage()];
        Arrays.fill(arr1, (byte)17);
        ByteBuffer buffer1 = ByteBuffer.wrap(arr1);
        buffer1.put(0, (byte)1);
        buffer1.put(1, (byte)0);
        buffer1.putShort(2, (short)2047);
        buffer1.put(7, (byte)1);
        byte[] arr2 = new byte[HyperLogLogCollector.getLatestNumBytesForDenseStorage()];
        Arrays.fill(arr2, (byte)17);
        ByteBuffer buffer2 = ByteBuffer.wrap(arr2);
        buffer2.put(0, (byte)1);
        buffer2.put(1, (byte)0);
        buffer2.putShort(2, (short)2048);
        HyperLogLogCollector collector = HyperLogLogCollector.makeCollector((ByteBuffer)buffer1);
        collector.fold(buffer2);
        ByteBuffer outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.get(), (long)1L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        outBuffer.get();
        outBuffer.getShort();
        Assert.assertFalse((boolean)outBuffer.hasRemaining());
    }

    @Test
    public void testSparseFoldWithDifferentOffsets1() throws Exception {
        ByteBuffer biggerOffset = this.makeCollectorBuffer(1, new byte[]{17, 16}, 17);
        ByteBuffer sparse = HyperLogLogCollector.makeCollector((ByteBuffer)this.makeCollectorBuffer(0, new byte[]{0, 2}, 0)).toByteBuffer();
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(biggerOffset);
        collector.fold(sparse);
        ByteBuffer outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)2L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        Assert.assertEquals((long)outBuffer.get(), (long)0L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        Assert.assertFalse((boolean)outBuffer.hasRemaining());
        collector = HyperLogLogCollector.makeLatestCollector();
        collector.fold(sparse);
        collector.fold(biggerOffset);
        outBuffer = collector.toByteBuffer();
        Assert.assertEquals((long)outBuffer.get(), (long)collector.getVersion());
        Assert.assertEquals((long)outBuffer.get(), (long)2L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        Assert.assertEquals((long)outBuffer.get(), (long)0L);
        Assert.assertEquals((long)outBuffer.getShort(), (long)0L);
        Assert.assertFalse((boolean)outBuffer.hasRemaining());
    }

    private ByteBuffer makeCollectorBuffer(int offset, byte initialBytes, int remainingBytes) {
        return this.makeCollectorBuffer(offset, new byte[]{initialBytes}, remainingBytes);
    }

    private ByteBuffer makeCollectorBuffer(int offset, byte[] initialBytes, int remainingBytes) {
        short numNonZero = 0;
        for (byte initialByte : initialBytes) {
            numNonZero = (short)(numNonZero + this.computeNumNonZero(initialByte));
        }
        short numNonZeroInRemaining = this.computeNumNonZero((byte)remainingBytes);
        numNonZero = (short)(numNonZero + (1024 - initialBytes.length) * numNonZeroInRemaining);
        ByteBuffer biggerOffset = ByteBuffer.allocate(HyperLogLogCollector.getLatestNumBytesForDenseStorage());
        biggerOffset.put((byte)1);
        biggerOffset.put((byte)offset);
        biggerOffset.putShort(numNonZero);
        biggerOffset.put((byte)0);
        biggerOffset.putShort((short)0);
        biggerOffset.put(initialBytes);
        while (biggerOffset.hasRemaining()) {
            biggerOffset.put((byte)remainingBytes);
        }
        biggerOffset.clear();
        return biggerOffset.asReadOnlyBuffer();
    }

    private short computeNumNonZero(byte theByte) {
        short retVal = 0;
        if ((theByte & 0xF) > 0) {
            retVal = (short)(retVal + 1);
        }
        if ((theByte & 0xF0) > 0) {
            retVal = (short)(retVal + 1);
        }
        return retVal;
    }

    @Ignore
    @Test
    public void testFoldingwithDifferentOffsets() throws Exception {
        Random random = new Random(0L);
        for (int j = 0; j < 10; ++j) {
            HyperLogLogCollector smallVals = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector bigVals = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector all = HyperLogLogCollector.makeLatestCollector();
            int numThings = 500000;
            for (int i = 0; i < numThings; ++i) {
                byte[] hashedVal = this.fn.hashLong(random.nextLong()).asBytes();
                if (i < 1000) {
                    smallVals.add(hashedVal);
                } else {
                    bigVals.add(hashedVal);
                }
                all.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeLatestCollector();
            folded.fold(smallVals);
            folded.fold(bigVals);
            double expected = all.estimateCardinality();
            Assert.assertEquals((double)expected, (double)folded.estimateCardinality(), (double)(expected * 0.025));
            Assert.assertEquals((double)numThings, (double)folded.estimateCardinality(), (double)((double)numThings * 0.05));
        }
    }

    @Ignore
    @Test
    public void testFoldingwithDifferentOffsets2() throws Exception {
        Random random = new Random(0L);
        MessageDigest md = MessageDigest.getInstance("SHA-1");
        for (int j = 0; j < 1; ++j) {
            HyperLogLogCollector evenVals = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector oddVals = HyperLogLogCollector.makeLatestCollector();
            HyperLogLogCollector all = HyperLogLogCollector.makeLatestCollector();
            int numThings = 500000;
            for (int i = 0; i < numThings; ++i) {
                md.update(Integer.toString(random.nextInt()).getBytes());
                byte[] hashedVal = this.fn.hashBytes(md.digest()).asBytes();
                if (i % 2 == 0) {
                    evenVals.add(hashedVal);
                } else {
                    oddVals.add(hashedVal);
                }
                all.add(hashedVal);
            }
            HyperLogLogCollector folded = HyperLogLogCollector.makeLatestCollector();
            folded.fold(evenVals);
            folded.fold(oddVals);
            double expected = all.estimateCardinality();
            Assert.assertEquals((double)expected, (double)folded.estimateCardinality(), (double)(expected * 0.025));
            Assert.assertEquals((double)numThings, (double)folded.estimateCardinality(), (double)((double)numThings * 0.05));
        }
    }

    @Test
    public void testEstimation() throws Exception {
        Random random = new Random(0L);
        int[] valsToCheck = new int[]{10, 20, 50, 100, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 1000000, 2000000};
        double[] expectedVals = new double[]{11.029647221949576, 21.108407720752034, 51.64575281885815, 100.42231726408892, 981.8579991802412, 1943.1337257462792, 4946.192042635218, 9935.088157579434, 20366.1486889433, 49433.56029693898, 100615.26273314281, 980831.624899156, 1982408.2608981386};
        int valsToCheckIndex = 0;
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        for (int i = 0; i < valsToCheck[valsToCheck.length - 1]; ++i) {
            collector.add(this.fn.hashLong(random.nextLong()).asBytes());
            if (i != valsToCheck[valsToCheckIndex]) continue;
            Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
            ++valsToCheckIndex;
        }
        Assert.assertEquals((long)expectedVals.length, (long)(valsToCheckIndex + 1));
        Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
    }

    @Test
    public void testEstimationReadOnlyByteBuffers() throws Exception {
        Random random = new Random(0L);
        int[] valsToCheck = new int[]{10, 20, 50, 100, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 1000000, 2000000};
        double[] expectedVals = new double[]{11.029647221949576, 21.108407720752034, 51.64575281885815, 100.42231726408892, 981.8579991802412, 1943.1337257462792, 4946.192042635218, 9935.088157579434, 20366.1486889433, 49433.56029693898, 100615.26273314281, 980831.624899156, 1982408.2608981386};
        int valsToCheckIndex = 0;
        HyperLogLogCollector collector = HyperLogLogCollector.makeCollector((ByteBuffer)ByteBuffer.allocateDirect(HyperLogLogCollector.getLatestNumBytesForDenseStorage()));
        for (int i = 0; i < valsToCheck[valsToCheck.length - 1]; ++i) {
            collector.add(this.fn.hashLong(random.nextLong()).asBytes());
            if (i != valsToCheck[valsToCheckIndex]) continue;
            Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
            ++valsToCheckIndex;
        }
        Assert.assertEquals((long)expectedVals.length, (long)(valsToCheckIndex + 1));
        Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
    }

    @Test
    public void testEstimationLimitDifferentFromCapacity() throws Exception {
        Random random = new Random(0L);
        int[] valsToCheck = new int[]{10, 20, 50, 100, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 1000000, 2000000};
        double[] expectedVals = new double[]{11.029647221949576, 21.108407720752034, 51.64575281885815, 100.42231726408892, 981.8579991802412, 1943.1337257462792, 4946.192042635218, 9935.088157579434, 20366.1486889433, 49433.56029693898, 100615.26273314281, 980831.624899156, 1982408.2608981386};
        int valsToCheckIndex = 0;
        HyperLogLogCollector collector = HyperLogLogCollector.makeCollector((ByteBuffer)((ByteBuffer)ByteBuffer.allocate(10000).position(0).limit(HyperLogLogCollector.getLatestNumBytesForDenseStorage())));
        for (int i = 0; i < valsToCheck[valsToCheck.length - 1]; ++i) {
            collector.add(this.fn.hashLong(random.nextLong()).asBytes());
            if (i != valsToCheck[valsToCheckIndex]) continue;
            Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
            ++valsToCheckIndex;
        }
        Assert.assertEquals((long)expectedVals.length, (long)(valsToCheckIndex + 1));
        Assert.assertEquals((double)expectedVals[valsToCheckIndex], (double)collector.estimateCardinality(), (double)0.0);
    }

    @Test
    public void testSparseEstimation() throws Exception {
        Random random = new Random(0L);
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        for (int i = 0; i < 100; ++i) {
            collector.add(this.fn.hashLong(random.nextLong()).asBytes());
        }
        Assert.assertEquals((double)collector.estimateCardinality(), (double)collector.estimateByteBuffer(collector.toByteBuffer()), (double)0.0);
    }

    @Test
    public void testHighBits() throws Exception {
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        HyperLogLogCollectorTest.fillBuckets(collector, (byte)0, (byte)49);
        collector.add(new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0});
        Assert.assertEquals((double)8.508968579344168E17, (double)collector.estimateCardinality(), (double)1000.0);
        HyperLogLogCollectorTest.fillBuckets(collector, (byte)0, (byte)63);
        collector.add(new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0});
        Assert.assertEquals((double)Double.MAX_VALUE, (double)collector.estimateCardinality(), (double)1000.0);
    }

    @Test
    public void testCompare1() throws Exception {
        HyperLogLogCollector collector1 = HyperLogLogCollector.makeLatestCollector();
        HyperLogLogCollector collector2 = HyperLogLogCollector.makeLatestCollector();
        collector1.add(this.fn.hashLong(0L).asBytes());
        HyperUniquesAggregatorFactory factory = new HyperUniquesAggregatorFactory("foo", "bar");
        Comparator comparator = factory.getComparator();
        for (int i = 1; i < 100; i += 2) {
            collector1.add(this.fn.hashLong((long)i).asBytes());
            collector2.add(this.fn.hashLong((long)(i + 1)).asBytes());
            Assert.assertEquals((long)1L, (long)comparator.compare(collector1, collector2));
            Assert.assertEquals((long)1L, (long)Double.compare(collector1.estimateCardinality(), collector2.estimateCardinality()));
        }
    }

    @Test
    public void testCompare2() throws Exception {
        int l;
        int k;
        HyperLogLogCollector collector2;
        int j;
        HyperLogLogCollector collector1;
        int i;
        Random rand = new Random(0L);
        HyperUniquesAggregatorFactory factory = new HyperUniquesAggregatorFactory("foo", "bar");
        Comparator comparator = factory.getComparator();
        for (i = 1; i < 1000; ++i) {
            collector1 = HyperLogLogCollector.makeLatestCollector();
            j = rand.nextInt(50);
            for (int l2 = 0; l2 < j; ++l2) {
                collector1.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            collector2 = HyperLogLogCollector.makeLatestCollector();
            k = j + 1 + rand.nextInt(5);
            for (l = 0; l < k; ++l) {
                collector2.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            Assert.assertEquals((long)Double.compare(collector1.estimateCardinality(), collector2.estimateCardinality()), (long)comparator.compare(collector1, collector2));
        }
        for (i = 1; i < 100; ++i) {
            collector1 = HyperLogLogCollector.makeLatestCollector();
            j = rand.nextInt(500);
            for (int l3 = 0; l3 < j; ++l3) {
                collector1.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            collector2 = HyperLogLogCollector.makeLatestCollector();
            k = j + 2 + rand.nextInt(5);
            for (l = 0; l < k; ++l) {
                collector2.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            Assert.assertEquals((long)Double.compare(collector1.estimateCardinality(), collector2.estimateCardinality()), (long)comparator.compare(collector1, collector2));
        }
        for (i = 1; i < 10; ++i) {
            collector1 = HyperLogLogCollector.makeLatestCollector();
            j = rand.nextInt(100000);
            for (int l4 = 0; l4 < j; ++l4) {
                collector1.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            collector2 = HyperLogLogCollector.makeLatestCollector();
            k = j + 20000 + rand.nextInt(100000);
            for (l = 0; l < k; ++l) {
                collector2.add(this.fn.hashLong(rand.nextLong()).asBytes());
            }
            Assert.assertEquals((long)Double.compare(collector1.estimateCardinality(), collector2.estimateCardinality()), (long)comparator.compare(collector1, collector2));
        }
    }

    @Test
    public void testMaxOverflow() {
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.add((short)23, (byte)16);
        Assert.assertEquals((long)23L, (long)collector.getMaxOverflowRegister());
        Assert.assertEquals((long)16L, (long)collector.getMaxOverflowValue());
        Assert.assertEquals((long)0L, (long)collector.getRegisterOffset());
        Assert.assertEquals((long)0L, (long)collector.getNumNonZeroRegisters());
        collector.add((short)56, (byte)17);
        Assert.assertEquals((long)56L, (long)collector.getMaxOverflowRegister());
        Assert.assertEquals((long)17L, (long)collector.getMaxOverflowValue());
        collector.add((short)43, (byte)16);
        Assert.assertEquals((long)56L, (long)collector.getMaxOverflowRegister());
        Assert.assertEquals((long)17L, (long)collector.getMaxOverflowValue());
        Assert.assertEquals((long)0L, (long)collector.getRegisterOffset());
        Assert.assertEquals((long)0L, (long)collector.getNumNonZeroRegisters());
    }

    @Test
    public void testMergeMaxOverflow() {
        HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
        collector.add((short)23, (byte)16);
        HyperLogLogCollector other = HyperLogLogCollector.makeLatestCollector();
        collector.add((short)56, (byte)17);
        collector.fold(other);
        Assert.assertEquals((long)56L, (long)collector.getMaxOverflowRegister());
        Assert.assertEquals((long)17L, (long)collector.getMaxOverflowValue());
        collector = HyperLogLogCollector.makeLatestCollector();
        HyperLogLogCollectorTest.fillBuckets(collector, (byte)0, (byte)49);
        collector.add((short)23, (byte)65);
        other = HyperLogLogCollector.makeLatestCollector();
        HyperLogLogCollectorTest.fillBuckets(other, (byte)0, (byte)43);
        other.add((short)47, (byte)67);
        collector.fold(other);
        Assert.assertEquals((long)47L, (long)collector.getMaxOverflowRegister());
        Assert.assertEquals((long)67L, (long)collector.getMaxOverflowValue());
    }

    private static void fillBuckets(HyperLogLogCollector collector, byte startOffset, byte endOffset) {
        for (byte offset = startOffset; offset <= endOffset; offset = (byte)(offset + 1)) {
            for (short bucket = 0; bucket < 2048; bucket = (short)(bucket + 1)) {
                collector.add(bucket, offset);
            }
        }
    }

    @Test
    public void testFoldOrder() throws Exception {
        ArrayList objects = Lists.newArrayList((Object[])new String[]{"AQcH/xYEMXOjRTVSQ1NXVENEM1RTUlVTRDI1aEVnhkOjNUaCI2MkU2VVhVNkNyVTa4NEYkS0kjZYU1RDdEYzUjglNTUzVFM0NkU3ZFUjOVJCdlU0N2QjRDRUV1MyZjNmVDOUM2RVVFRzhnUzVXY1R1RHUnNziURUdmREM0VjVEQmU0aEInZYNzNZNVRFgzVFNolSJHNIQ3QklEZlNSNoNTJXpDk1dFWjJGNYNiQzQkZFNEYzc1NVhSczM2NmJDZlc3JJRCpVNiRlNEI3dmU1ZGI0Q1RCMhNFZEJDZDYyNFOCM3U0VmRlVlNIRVQ4VVw1djNDVURHVSaFU0VEY0U1JFNIVCYlVEJWM2NWU0eURDOjQ6YyNTYkZjNUVjR1ZDdnVkMzVHZFpjMzlmNEFHM0dHJlRYTHSEQjVZVVZkVVIzIjg2SUU0NSM0VFNDNCdGVlQkhBNENCVTZGZEVlxFQyQ0NYWkUmVUJUYzRlNqg4NVVTNThEJkRGNDNUNFSEYmgkR0dDR1JldCNhVEZGRENGc1NDRUNER3WJRTRHQ4JlOYZoJDVVVVMzZSREZ1Q1UjSHNkdUMlU0ODIzZThSNmNDNjQ1o2I0YiRGYyZkNUJYVEMyN2QpQyMkc2VTE4U2VCNHZFRDNTh0IzI2VFNTMlUkNGMlKTRCIyR3QiQzFUNkRTdDM6RDRFI3VyVlcyWCUlQ0YjNjU2Q2dEVFNTRyRlI7VElHVTVVNGk0JHJTQzQkQyVlV0NCVlRkhWYkQ0RVaDNYdFZHWEWFJEYpM0QjNjNVUzNCVzVkgzZGFzQkRZUzN2U1dUFGVWZTUzVUREZDciZEVVYVNjeCU0ZDdEhzIpU2RTOFRUQkWlk1OFRUVTN1MkZSM3ZFc1VDNnUmc2NKNUaUIzd3M0RWxEZTsiNENLVHU0NFUmQ2RWRFdCNUVENFkxZCEnRLQkNEU0RVNmVDQjl9ZmNkM1QVM0MzQkUjJlVHRkNEVWlENDVUIlUvRkM0RVY1UzY6OGVHVCRDIzRUUlUjM2RDWSVkVIU1U1ZiVFNlNDhTN1VWNTVEZ2RzNzVDQlY0ZUNENUM5NUdkRDJGYzRCUzIjRGR4UmJFI4GDRTUiQ0ZUhVY1ZEYoZSRoVDYnREYkQ1SUU0RWUycjp2RZIySVZkUmZDREZVJGQyVEc1JElBZENEU2VEQlVUUnNDQziLRTNidmNjVCtjRFU2Q0SGYzVHVpGTNoVDxFVSMlWTJFQyRJdV1EI3RDloYyNFQ0c1NVY0ZHVEY0dkM2QkQyVDVUVTNFUyamMUdSNrNz0mlFlERzZTSGhFRjVGM3NWU2NINDI2U1RERUhjY4FHNWNTVTV1U0U2I0VXNEZERWNDNUSjI1WmMmQ4U=", "AQgH+BUFUEUrZVRjM2IjMzJRESMlUnlTJjEjRhRlNBEyMSUpaGJTMjRCIzMTNCRENRdxNiNEZCQzNERYMiAyIiQmUTI+MhEzV1RWJoMjQjIySDN0QiYDUjUzNjRUVEYyQleDEiUmg0ERRjIjIzJUQjMxNlJGUTNDJFNTRzJiE1M0RjQzUzIiFDUmMjIzJWVCNENTIRJVODUzEkIVMhFEIjM0MkMyIRRCNFNxQyNCQ2UzOFQiJSM0EzU1V1M2EjhUVENDclZzImEiMTJBQlQiJCgyIyKkJSUlNBNDE2M3QSIyMicjMlJEUhJDJFQjJ0VSQ0QyYSFhZSNlQ4REUzVFIlOFRHIkYUJEM8RVMkMiMEczQwMlE1EkAlNiQlhCNkISRVI0ITUjRDU1JVNlK1QyGGRHQVM0NUVHQ1MkMyQoIzMzFCFUI0IhU1OIhCIlZUQVIUMyYzMlMUZ0RCKEIigUIlQ0QkQTM0MkM0QyJkUSM2I2tHJDUTQ0RBQ0YyNlUxUzIiIiMUiSMzUlJDNDQjM0ITQyNIM1MyNWM0MDOTZYVDRWIiZhMzc0NCJ0Q0NDZEMUElMyRyMmUhNiMkIZNjMkEyRTIzYkMzNUODUTNDJVM0ZTQjFCJCNWSTUlEiNCM1U2FCZUJzMVMyLjNkMhITVDEjIYMzNiVmIlO1VTMjMiVDQ2NTJFYyE0Q2IjRDN2IjRTRUVTFUVEYVKBVSMVJSFE0zOXNSJIqVElMVM4MiZEFSMhRlJEJUZnMycmQmQyJDl1JzVjMXQ0MzMjE1VUI1JDJUQyYRQ2JVZzQUJDM2IyInEkY1QiZTJEMRMiMxRVNEUjJUNkJHNSQiNCVCIyIjJUQlEhNUdFUhQzgkcSZaJUVUM0YiJEM2SjczUUIUIlQiM0RiQkIzZhRBJSRzQ0ZUI00UUSRSQlQmMkNINzODQhJFRTZ0FRQ3QTRhIzFTJFRBMmMzQzQhZENUMiIlV2VEMiNFRWQ1F1IyFXRSUyRTMqZ3I0YyhUNEJRMjISZRc2NDOEIjIxVGVWIXYyMiNCJBFDQSMhIzMjVFIDElgyJCUyVFgkRSQzIjJFQlNWRTQWMmQzFFOiMzVTZGMxNFZUNmIjRjETNUNURERTQjYVIkEzNEEyNDNTVUJSVzVkMjEyUlMjQ0RGgyFFNUQhRGMmRUQ2ZSOFETUYNlZCUhRiU2QhVUUiIlJDRjMhRVJDZxNSRTNBRCEoI0FGNUVRE0VFOGdCRDM2QkJCFSQhMxITRoE0VFIzVWUiUTNkRhNDMiMmIzRDQSNTFDoldaJDcnNjkSMJg3IkIiRENSQmciUhY2NFQ4RSNoJENkWDMmVCJGMxQjJGJScyNTJDVDNEEiZSMzQyIyVGRTNEIUw=", "AQgH+hQAFyMzlFVXNCNlRxRUYlRUUUZCMnRFJiR0WTgyZiRJZzRFQkVTVVVWc2ZFMlY1QkIxYUQTI0JDY1YkNEVENGUuQTRiNkQ0VUEzNkKUKLSIVkUhNiZURnRFMzcjVEBTdjVVVCIzJDM0hjc0RDVlVjRqMjJVZTNSM0QmQyMTRlNzVCNERFQyMxNBZHMiUSdYIUUjNlVjNzRyYWFHRHI3hKMnYnhFNCZOdlNUZBM0Q0clNTVBiEQRMUQzNSNVQ0IkEmZYNzIyNkRSUik2VBOVRCRDg0IilEMlcjRJMkJDSjRCJURTVDJBMmRTVBM1YyRRMSQoRDV2YzRDVCUkQWFFNDYnQ0IkUzRjRkQ1dGI0VUYzRERCQ1I2dFNhREOUUjJDc0NTN0JFNUZJRGFpU1Q0QyJlNiMzNCZSKFQzYnNUWTMiRGMiRWdSQzMiQnQ0QSgjVUMiE0hRM1NVUiZVIlRkRVMzI2VkRjQWQ1YyRiZWNHQXQ0UllUMSVTJDQzMkWCQiRFglMzIzKEYzJSJFMyREVIQlVFFlYzMDQyVWUZNCQlM0NUJFIkWiNnREdEJDImNWJDOIcmKyQzc5VDVRQ3PVNjQzIkJTQ3FzMjMyRFVFVTUlNUZEMzEjI0Q0M0Y2U1JTREQjIhZScUJjQkYhFRJFQyI0pTVmFTVlMkJXNDI1U3dFZkR2U0NCVRQyRih0UkIhckRUY0ZHSG00EiJUdVIxVjVGNnUVZCxEQkNTQjQ0IkZDciIkODYxM1MzRZRHQxVEZHZWJFIzRRZjVDNBMzI1Q1FEhUMiI0NkJWJWJDJzYlQiRSQjRoRiRhJTIjNSRVJEM1MiYmUiNBkjFkczRWU1SURIJUVDRFQ0QyZCUlRENEImE2FDQxRjlEdTI3RSNEU3RGJyWDNVMTVJNDM1QkJFQmNWRXUlcxNEQzNTGCtDUlNDMzMzY2VlcUQlaUIyZVMzA3NFM1NDc0JjZDUkQiFDY3QUczQzUkVDQjMiUWQ0NEQyNRVTMRJFM2RUMZNSQkQ0MkIiUgGCUkRig1UiElQkdDJFJDciVGIxMjQzI1UlNlRTM1JkRDc+RSM0VFUzMjWCU0RDMxJyJVJGI1VTEUQyM1R0I1c0NFNTM3MhIlUkNFIlZGNURkVURyNIVCMyYzQmQjITRkVHQ2NINGQ0Y0UW0icyUzMydEVBJVJIJENkUjRVIjQSNVYnEzVYMzUmYzGVNFRiQk0iVTVCM0RjJSMyRWRSQkURJBR0M0NzhnRlM3IzQxMTRDJjM1UVUkJCNUQTVGQlEzN0VDMyM0MmO2QoQzNSVURhFEAkU2IldINHRUU00zNFJVQxUkZEcVMyJSJkQjKFNCNUOzIYJEHEUyKCQjJESSY=", "AQcH/x0BbjQ2JTpUlkdFRERHVDRkWDU0RGR1ejRURHZ6IzUqdJN1M1VFQiNHI1NTI0J1VHOGZYVFVTRIRJVkVmUolWVShERjSDRVMlRlJDU2VFh3UmR1Mjg3K0M2SUY0Q0ZUspNiJEdZMmc3YkxGSERGOGdjgzNRVGM1Q1UnN0RHU1Y0WWUzRWVEJSRSeGQ0RlNFJVVJU3YoQkdEQ2M2MiVFUyJWRUVWNmRkM0NkVER2WXNkR0QlNGNEVlYzZSS4RDMyVEQ1ckRTM0ZoMlQ2tURGQ0OFQ0ZiY1ZFNEajdXEjVSI6ZWSjNHVRRTRVMldzUjm0NGU0dlhESFRDM0IzVCYkdjdlJJRFVDaHEzUkRmNWOEVXZTM0U0VkREdSUjRHVVViVCVFVUN0RDNDkl01VHMoNVQzYlZFZmNVVUNDQ1VjUiQ2NTV0UzZVModSNEY4Zpc2JjhjiFJVUGM0SHI0UzRTU1R2R0d3VENUZSQzRUZlY4d0aGNkhTQzWVZFZTZkJ2NEZaVDU1alJWpFJpRGRnIlZUU1ZUR2M1NzOkVEMzVjZERiVlRYSEkmU4RLM0RTQ2Q2RjM3RTNhdVVEQzRXJUZTRmM1OEZTYyJkRGRjZDVTlDhSMzdXQiU1RFUiIoRpVGlXIjY1UVVjc0RDJDNSM0NVJTNkRUU1U0lDdEVXY2NGVVNJVmJJRTREVVNiMyVIQ3U6O0U0M0MzZFVVIzJmNERWJaJjikIlRXk1hFQ2NEU0RUN4UzdENEsVgzZFVidXUnU2VRZFRUQmZmRERCQ0ZER2Q3YnZFNlVpJUkzZVREKFWEUzVVMzYzQzQhfTYzQ0IlI5UoV0RGJCVXSDkyZCRSU3ITUkNoYzJUMkYzhlVVRTNyaDNmQzRDVVRjNkVUhEJyRBR2JlOEREVUU0RjY4Nkc3ZERGUyVDNFZGNFOTY3U1OKNlkjQy1TVlRTQ0M1REU2QhgzUzUzOWlWQ1Z3RTQzIzc7RXVkI0M4NCNYRVRGNEZbhFEyVJI0R1OUZEQ3VUVEQlU1NkNYJEYzdSQ0ZSNGeEWIVVU3KEVFY1RZQ0JSNEJFNFMyM0UzN0hHNTQjMlRGNkiEMyVjRFNVRXNkZGM2M4hENCMnU1VWQjNFRkO2VmO1RndEVzWTQiiHQ0NzM2clM4NjQxpjQjZEVTNEpEdlREJzc3OjZnRlNFNWJVNFeDokNCRmQ5NURJVUZSJyRDRXikVURVITZDNGW0ITNEOUQ0RUklZDQjYjVENURDRCRmRDU1hCY2VTR0RGIzJSZzlSczdTFJJkRlZyU1M1JTdVhDYhVFczQ0hTRIc0RCNDdUJEQxNlZEQ2ZEUiJJRFU3YzVGRER0R2ZlNFOTU1MyRGI0RzMkQ2Q="});
        List collectors = Lists.transform((List)objects, (Function)new Function<String, HyperLogLogCollector>(){

            @Nullable
            public HyperLogLogCollector apply(@Nullable String s) {
                return HyperLogLogCollector.makeCollector((ByteBuffer)ByteBuffer.wrap(Base64.decodeBase64((String)s)));
            }
        });
        Collection permutations = Collections2.permutations((Collection)collectors);
        for (List permutation : permutations) {
            HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
            for (HyperLogLogCollector foldee : permutation) {
                collector.fold(foldee);
            }
            Assert.assertEquals((long)29L, (long)collector.getMaxOverflowValue());
            Assert.assertEquals((long)366L, (long)collector.getMaxOverflowRegister());
            Assert.assertEquals((double)1.0429189446653817E7, (double)collector.estimateCardinality(), (double)1.0);
        }
    }

    @Ignore
    @Test
    public void showErrorRate() throws Exception {
        int[] valsToCheck;
        HashFunction fn = Hashing.murmur3_128();
        Random random = new Random();
        double error = 0.0;
        int count = 0;
        for (int numThings : valsToCheck = new int[]{10, 20, 50, 100, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 1000000, 2000000, 10000000, Integer.MAX_VALUE}) {
            long startTime = System.currentTimeMillis();
            HyperLogLogCollector collector = HyperLogLogCollector.makeLatestCollector();
            for (int i = 0; i < numThings; ++i) {
                if (i != 0 && i % 100000000 == 0) {
                    error = this.computeError(error, ++count, i, startTime, collector);
                }
                collector.add(fn.hashLong(random.nextLong()).asBytes());
            }
            error = this.computeError(error, ++count, numThings, startTime, collector);
        }
    }

    private double computeError(double error, int count, int numThings, long startTime, HyperLogLogCollector collector) {
        double estimatedValue = collector.estimateCardinality();
        double errorThisTime = Math.abs((double)numThings - estimatedValue) / (double)numThings;
        System.out.printf("%,d ==? %,f in %,d millis. actual error[%,f%%], avg. error [%,f%%]%n", numThings, estimatedValue, System.currentTimeMillis() - startTime, 100.0 * errorThisTime, (error += errorThisTime) / (double)count * 100.0);
        return error;
    }
}

